"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error(`Exactly one of 'customMetric' or 'predefinedMetric' must be specified.`);
        }
        if (props.scaleInCooldownSec !== undefined && props.scaleInCooldownSec < 0) {
            throw new RangeError(`scaleInCooldown cannot be negative, got: ${props.scaleInCooldownSec}`);
        }
        if (props.scaleOutCooldownSec !== undefined && props.scaleOutCooldownSec < 0) {
            throw new RangeError(`scaleOutCooldown cannot be negative, got: ${props.scaleOutCooldownSec}`);
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldownSec,
                scaleOutCooldown: props.scaleOutCooldownSec,
                targetValue: props.targetValue
            }
        });
        this.scalingPolicyArn = resource.scalingPolicyArn;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    return {
        dimensions: metric.dimensionsAsList(),
        metricName: metric.metricName,
        namespace: metric.namespace,
        statistic: metric.statistic,
        unit: metric.unit
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    PredefinedMetric["DynamoDBReadCapacityUtilization"] = "DynamoDBReadCapacityUtilization";
    PredefinedMetric["DynamoDBWriteCapacityUtilization"] = "DynamoDBWriteCapacityUtilization";
    PredefinedMetric["ALBRequestCountPerTarget"] = "ALBRequestCountPerTarget";
    PredefinedMetric["RDSReaderAverageCPUUtilization"] = "RDSReaderAverageCPUUtilization";
    PredefinedMetric["RDSReaderAverageDatabaseConnections"] = "RDSReaderAverageDatabaseConnections";
    PredefinedMetric["EC2SpotFleetRequestAverageCPUUtilization"] = "EC2SpotFleetRequestAverageCPUUtilization";
    PredefinedMetric["EC2SpotFleetRequestAverageNetworkIn"] = "EC2SpotFleetRequestAverageNetworkIn";
    PredefinedMetric["EC2SpotFleetRequestAverageNetworkOut"] = "EC2SpotFleetRequestAverageNetworkOut";
    PredefinedMetric["SageMakerVariantInvocationsPerInstance"] = "SageMakerVariantInvocationsPerInstance";
    PredefinedMetric["ECSServiceAverageCPUUtilization"] = "ECSServiceAverageCPUUtilization";
    PredefinedMetric["ECSServiceAverageMemoryUtilization"] = "ECSServiceAverageMemoryUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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