"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldownSec,
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: new cdk.Token(() => this.adjustments),
            }
        });
        this.scalingPolicyArn = resource.scalingPolicyArn;
    }
    /**
     * Add an adjusment interval to the ScalingAction
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted
 */
var AdjustmentType;
(function (AdjustmentType) {
    /**
     * Add the adjustment number to the current capacity.
     *
     * A positive number increases capacity, a negative number decreases capacity.
     */
    AdjustmentType["ChangeInCapacity"] = "ChangeInCapacity";
    /**
     * Add this percentage of the current capacity to itself.
     *
     * The number must be between -100 and 100; a positive number increases
     * capacity and a negative number decreases it.
     */
    AdjustmentType["PercentChangeInCapacity"] = "PercentChangeInCapacity";
    /**
     * Make the capacity equal to the exact number given.
     */
    AdjustmentType["ExactCapacity"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    /**
     * Average
     */
    MetricAggregationType["Average"] = "Average";
    /**
     * Minimum
     */
    MetricAggregationType["Minimum"] = "Minimum";
    /**
     * Maximum
     */
    MetricAggregationType["Maximum"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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