"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends cdk_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        if (props.maxCapacity < 0) {
            throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
        }
        if (props.minCapacity < 0) {
            throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
        }
        if (props.maxCapacity < props.minCapacity) {
            throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
        }
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: this.actions,
            serviceNamespace: props.serviceNamespace
        });
        this.scalableTargetId = resource.scalableTargetId;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends cdk_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["Ecs"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ElasticMapReduce"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["Ec2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["AppStream"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DynamoDb"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["Rds"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SageMaker"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CustomResource"] = "custom-resource";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NhbGFibGUtdGFyZ2V0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2NhbGFibGUtdGFyZ2V0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXlDO0FBQ3pDLHNDQUE4RDtBQUM5RCx5RkFBdUU7QUFDdkUsK0RBQXVGO0FBQ3ZGLHFGQUFzSDtBQStEdEg7O0dBRUc7QUFDSCxNQUFhLGNBQWUsU0FBUSxjQUFRO0lBd0IxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCO1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFIRixZQUFPLEdBQUcsSUFBSSxLQUFLLEVBQTZDLENBQUM7UUFLaEYsSUFBSSxLQUFLLENBQUMsV0FBVyxHQUFHLENBQUMsRUFBRTtZQUN6QixNQUFNLElBQUksVUFBVSxDQUFDLHdDQUF3QyxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUNuRjtRQUNELElBQUksS0FBSyxDQUFDLFdBQVcsR0FBRyxDQUFDLEVBQUU7WUFDekIsTUFBTSxJQUFJLFVBQVUsQ0FBQyx3Q0FBd0MsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7U0FDbkY7UUFDRCxJQUFJLEtBQUssQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUN6QyxNQUFNLElBQUksVUFBVSxDQUFDLGdCQUFnQixLQUFLLENBQUMsV0FBVyx1Q0FBdUMsS0FBSyxDQUFDLFdBQVcsR0FBRyxDQUFDLENBQUM7U0FDcEg7UUFFRCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDbkQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHVDQUF1QyxDQUFDO1NBQzdFLENBQUMsQ0FBQztRQUVILE1BQU0sUUFBUSxHQUFHLElBQUksb0RBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN2RCxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixPQUFPLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQzFCLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7WUFDMUMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDOUIsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtTQUN6QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLGdCQUFnQixDQUFDO0lBQ3BELENBQUM7SUFsRE0sTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGdCQUF3QjtRQUN2RixNQUFNLE1BQU8sU0FBUSxjQUFRO1lBQTdCOztnQkFDa0IscUJBQWdCLEdBQUcsZ0JBQWdCLENBQUM7WUFDdEQsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQStDRDs7T0FFRztJQUNJLGVBQWUsQ0FBQyxTQUE4QjtRQUNuRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxlQUFlLENBQUMsRUFBVSxFQUFFLE1BQXVCO1FBQ3hELElBQUksTUFBTSxDQUFDLFdBQVcsS0FBSyxTQUFTLElBQUksTUFBTSxDQUFDLFdBQVcsS0FBSyxTQUFTLEVBQUU7WUFDeEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRUFBbUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDOUc7UUFDRCxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNoQixtQkFBbUIsRUFBRSxFQUFFO1lBQ3ZCLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUTtZQUN6QixTQUFTLEVBQUUsTUFBTSxDQUFDLFNBQVM7WUFDM0IsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPO1lBQ3ZCLG9CQUFvQixFQUFFO2dCQUNwQixXQUFXLEVBQUUsTUFBTSxDQUFDLFdBQVc7Z0JBQy9CLFdBQVcsRUFBRSxNQUFNLENBQUMsV0FBVzthQUNoQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLGFBQWEsQ0FBQyxFQUFVLEVBQUUsS0FBa0M7UUFDakUsT0FBTyxJQUFJLHVDQUFpQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsRUFBRSxHQUFHLEtBQUssRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7O09BRUc7SUFDSSxrQkFBa0IsQ0FBQyxFQUFVLEVBQUUsS0FBNEM7UUFDaEYsT0FBTyxJQUFJLDREQUEyQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsRUFBRSxHQUFHLEtBQUssRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUN0RixDQUFDO0NBQ0Y7QUE3RkQsd0NBNkZDO0FBaUVEOztHQUVHO0FBQ0gsSUFBWSxnQkF3Q1g7QUF4Q0QsV0FBWSxnQkFBZ0I7SUFDMUI7O09BRUc7SUFDSCwrQkFBVyxDQUFBO0lBRVg7O09BRUc7SUFDSCx5REFBcUMsQ0FBQTtJQUVyQzs7T0FFRztJQUNILCtCQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILDJDQUF1QixDQUFBO0lBRXZCOztPQUVHO0lBQ0gseUNBQXFCLENBQUE7SUFFckI7O09BRUc7SUFDSCwrQkFBVyxDQUFBO0lBRVg7O09BRUc7SUFDSCwyQ0FBdUIsQ0FBQTtJQUV2Qjs7T0FFRztJQUNILHNEQUFrQyxDQUFBO0FBQ3BDLENBQUMsRUF4Q1csZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUF3QzNCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGlhbSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1pYW0nKTtcbmltcG9ydCB7IENvbnN0cnVjdCwgSVJlc291cmNlLCBSZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2Nkayc7XG5pbXBvcnQgeyBDZm5TY2FsYWJsZVRhcmdldCB9IGZyb20gJy4vYXBwbGljYXRpb25hdXRvc2NhbGluZy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQmFzaWNTdGVwU2NhbGluZ1BvbGljeVByb3BzLCBTdGVwU2NhbGluZ1BvbGljeSB9IGZyb20gJy4vc3RlcC1zY2FsaW5nLXBvbGljeSc7XG5pbXBvcnQgeyBCYXNpY1RhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeVByb3BzLCBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3kgfSBmcm9tICcuL3RhcmdldC10cmFja2luZy1zY2FsaW5nLXBvbGljeSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSVNjYWxhYmxlVGFyZ2V0IGV4dGVuZHMgSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHNjYWxhYmxlVGFyZ2V0SWQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIHNjYWxhYmxlIHRhcmdldFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNjYWxhYmxlVGFyZ2V0UHJvcHMge1xuICAvKipcbiAgICogVGhlIG1pbmltdW0gdmFsdWUgdGhhdCBBcHBsaWNhdGlvbiBBdXRvIFNjYWxpbmcgY2FuIHVzZSB0byBzY2FsZSBhIHRhcmdldCBkdXJpbmcgYSBzY2FsaW5nIGFjdGl2aXR5LlxuICAgKi9cbiAgcmVhZG9ubHkgbWluQ2FwYWNpdHk6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gdmFsdWUgdGhhdCBBcHBsaWNhdGlvbiBBdXRvIFNjYWxpbmcgY2FuIHVzZSB0byBzY2FsZSBhIHRhcmdldCBkdXJpbmcgYSBzY2FsaW5nIGFjdGl2aXR5LlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4Q2FwYWNpdHk6IG51bWJlcjtcblxuICAvKipcbiAgICogUm9sZSB0aGF0IGFsbG93cyBBcHBsaWNhdGlvbiBBdXRvIFNjYWxpbmcgdG8gbW9kaWZ5IHlvdXIgc2NhbGFibGUgdGFyZ2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCBBIHJvbGUgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkXG4gICAqL1xuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVzb3VyY2UgaWRlbnRpZmllciB0byBhc3NvY2lhdGUgd2l0aCB0aGlzIHNjYWxhYmxlIHRhcmdldC5cbiAgICpcbiAgICogVGhpcyBzdHJpbmcgY29uc2lzdHMgb2YgdGhlIHJlc291cmNlIHR5cGUgYW5kIHVuaXF1ZSBpZGVudGlmaWVyLlxuICAgKlxuICAgKiBAZXhhbXBsZSBzZXJ2aWNlL2Vjc1N0YWNrLU15RUNTQ2x1c3Rlci1BQjEyQ0RFM0Y0R0gvZWNzU3RhY2stTXlFQ1NTZXJ2aWNlLUFCMTJDREUzRjRHSFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9hcHBsaWNhdGlvbi9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVyU2NhbGFibGVUYXJnZXQuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2VJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgc2NhbGFibGUgZGltZW5zaW9uIHRoYXQncyBhc3NvY2lhdGVkIHdpdGggdGhlIHNjYWxhYmxlIHRhcmdldC5cbiAgICpcbiAgICogU3BlY2lmeSB0aGUgc2VydmljZSBuYW1lc3BhY2UsIHJlc291cmNlIHR5cGUsIGFuZCBzY2FsaW5nIHByb3BlcnR5LlxuICAgKlxuICAgKiBAZXhhbXBsZSBlY3M6c2VydmljZTpEZXNpcmVkQ291bnRcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXV0b3NjYWxpbmcvYXBwbGljYXRpb24vQVBJUmVmZXJlbmNlL0FQSV9TY2FsaW5nUG9saWN5Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHNjYWxhYmxlRGltZW5zaW9uOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lc3BhY2Ugb2YgdGhlIEFXUyBzZXJ2aWNlIHRoYXQgcHJvdmlkZXMgdGhlIHJlc291cmNlIG9yXG4gICAqIGN1c3RvbS1yZXNvdXJjZSBmb3IgYSByZXNvdXJjZSBwcm92aWRlZCBieSB5b3VyIG93biBhcHBsaWNhdGlvbiBvclxuICAgKiBzZXJ2aWNlLlxuICAgKlxuICAgKiBGb3IgdmFsaWQgQVdTIHNlcnZpY2UgbmFtZXNwYWNlIHZhbHVlcywgc2VlIHRoZSBSZWdpc3RlclNjYWxhYmxlVGFyZ2V0XG4gICAqIGFjdGlvbiBpbiB0aGUgQXBwbGljYXRpb24gQXV0byBTY2FsaW5nIEFQSSBSZWZlcmVuY2UuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL0FQSVJlZmVyZW5jZS9BUElfUmVnaXN0ZXJTY2FsYWJsZVRhcmdldC5odG1sXG4gICAqL1xuICByZWFkb25seSBzZXJ2aWNlTmFtZXNwYWNlOiBTZXJ2aWNlTmFtZXNwYWNlO1xufVxuXG4vKipcbiAqIERlZmluZSBhIHNjYWxhYmxlIHRhcmdldFxuICovXG5leHBvcnQgY2xhc3MgU2NhbGFibGVUYXJnZXQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElTY2FsYWJsZVRhcmdldCB7XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tU2NhbGFibGVUYXJnZXRJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBzY2FsYWJsZVRhcmdldElkOiBzdHJpbmcpOiBJU2NhbGFibGVUYXJnZXQge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVNjYWxhYmxlVGFyZ2V0IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBzY2FsYWJsZVRhcmdldElkID0gc2NhbGFibGVUYXJnZXRJZDtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJRCBvZiB0aGUgU2NhbGFibGUgVGFyZ2V0XG4gICAqXG4gICAqIEBleGFtcGxlIHNlcnZpY2UvZWNzU3RhY2stTXlFQ1NDbHVzdGVyLUFCMTJDREUzRjRHSC9lY3NTdGFjay1NeUVDU1NlcnZpY2UtQUIxMkNERTNGNEdIfGVjczpzZXJ2aWNlOkRlc2lyZWRDb3VudHxlY3NcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNjYWxhYmxlVGFyZ2V0SWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJvbGUgdXNlZCB0byBnaXZlIEF1dG9TY2FsaW5nIHBlcm1pc3Npb25zIHRvIHlvdXIgcmVzb3VyY2VcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByb2xlOiBpYW0uSVJvbGU7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBhY3Rpb25zID0gbmV3IEFycmF5PENmblNjYWxhYmxlVGFyZ2V0LlNjaGVkdWxlZEFjdGlvblByb3BlcnR5PigpO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTY2FsYWJsZVRhcmdldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChwcm9wcy5tYXhDYXBhY2l0eSA8IDApIHtcbiAgICAgIHRocm93IG5ldyBSYW5nZUVycm9yKGBtYXhDYXBhY2l0eSBjYW5ub3QgYmUgbmVnYXRpdmUsIGdvdDogJHtwcm9wcy5tYXhDYXBhY2l0eX1gKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLm1pbkNhcGFjaXR5IDwgMCkge1xuICAgICAgdGhyb3cgbmV3IFJhbmdlRXJyb3IoYG1pbkNhcGFjaXR5IGNhbm5vdCBiZSBuZWdhdGl2ZSwgZ290OiAke3Byb3BzLm1pbkNhcGFjaXR5fWApO1xuICAgIH1cbiAgICBpZiAocHJvcHMubWF4Q2FwYWNpdHkgPCBwcm9wcy5taW5DYXBhY2l0eSkge1xuICAgICAgdGhyb3cgbmV3IFJhbmdlRXJyb3IoYG1pbkNhcGFjaXR5ICgke3Byb3BzLm1pbkNhcGFjaXR5fSkgc2hvdWxkIGJlIGxvd2VyIHRoYW4gbWF4Q2FwYWNpdHkgKCR7cHJvcHMubWF4Q2FwYWNpdHl9KWApO1xuICAgIH1cblxuICAgIHRoaXMucm9sZSA9IHByb3BzLnJvbGUgfHwgbmV3IGlhbS5Sb2xlKHRoaXMsICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2FwcGxpY2F0aW9uLWF1dG9zY2FsaW5nLmFtYXpvbmF3cy5jb20nKVxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuU2NhbGFibGVUYXJnZXQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgbWF4Q2FwYWNpdHk6IHByb3BzLm1heENhcGFjaXR5LFxuICAgICAgbWluQ2FwYWNpdHk6IHByb3BzLm1pbkNhcGFjaXR5LFxuICAgICAgcmVzb3VyY2VJZDogcHJvcHMucmVzb3VyY2VJZCxcbiAgICAgIHJvbGVBcm46IHRoaXMucm9sZS5yb2xlQXJuLFxuICAgICAgc2NhbGFibGVEaW1lbnNpb246IHByb3BzLnNjYWxhYmxlRGltZW5zaW9uLFxuICAgICAgc2NoZWR1bGVkQWN0aW9uczogdGhpcy5hY3Rpb25zLFxuICAgICAgc2VydmljZU5hbWVzcGFjZTogcHJvcHMuc2VydmljZU5hbWVzcGFjZVxuICAgIH0pO1xuXG4gICAgdGhpcy5zY2FsYWJsZVRhcmdldElkID0gcmVzb3VyY2Uuc2NhbGFibGVUYXJnZXRJZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBwb2xpY3kgc3RhdGVtZW50IHRvIHRoZSByb2xlJ3MgcG9saWN5XG4gICAqL1xuICBwdWJsaWMgYWRkVG9Sb2xlUG9saWN5KHN0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCkge1xuICAgIHRoaXMucm9sZS5hZGRUb1BvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFNjYWxlIG91dCBvciBpbiBiYXNlZCBvbiB0aW1lXG4gICAqL1xuICBwdWJsaWMgc2NhbGVPblNjaGVkdWxlKGlkOiBzdHJpbmcsIGFjdGlvbjogU2NhbGluZ1NjaGVkdWxlKSB7XG4gICAgaWYgKGFjdGlvbi5taW5DYXBhY2l0eSA9PT0gdW5kZWZpbmVkICYmIGFjdGlvbi5tYXhDYXBhY2l0eSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFlvdSBtdXN0IHN1cHBseSBhdCBsZWFzdCBvbmUgb2YgbWluQ2FwYWNpdHkgb3IgbWF4Q2FwYWNpdHksIGdvdCAke0pTT04uc3RyaW5naWZ5KGFjdGlvbil9YCk7XG4gICAgfVxuICAgIHRoaXMuYWN0aW9ucy5wdXNoKHtcbiAgICAgIHNjaGVkdWxlZEFjdGlvbk5hbWU6IGlkLFxuICAgICAgc2NoZWR1bGU6IGFjdGlvbi5zY2hlZHVsZSxcbiAgICAgIHN0YXJ0VGltZTogYWN0aW9uLnN0YXJ0VGltZSxcbiAgICAgIGVuZFRpbWU6IGFjdGlvbi5lbmRUaW1lLFxuICAgICAgc2NhbGFibGVUYXJnZXRBY3Rpb246IHtcbiAgICAgICAgbWF4Q2FwYWNpdHk6IGFjdGlvbi5tYXhDYXBhY2l0eSxcbiAgICAgICAgbWluQ2FwYWNpdHk6IGFjdGlvbi5taW5DYXBhY2l0eVxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTY2FsZSBvdXQgb3IgaW4sIGluIHJlc3BvbnNlIHRvIGEgbWV0cmljXG4gICAqL1xuICBwdWJsaWMgc2NhbGVPbk1ldHJpYyhpZDogc3RyaW5nLCBwcm9wczogQmFzaWNTdGVwU2NhbGluZ1BvbGljeVByb3BzKSB7XG4gICAgcmV0dXJuIG5ldyBTdGVwU2NhbGluZ1BvbGljeSh0aGlzLCBpZCwgeyAuLi5wcm9wcywgc2NhbGluZ1RhcmdldDogdGhpcyB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTY2FsZSBvdXQgb3IgaW4gaW4gb3JkZXIgdG8ga2VlcCBhIG1ldHJpYyBhcm91bmQgYSB0YXJnZXQgdmFsdWVcbiAgICovXG4gIHB1YmxpYyBzY2FsZVRvVHJhY2tNZXRyaWMoaWQ6IHN0cmluZywgcHJvcHM6IEJhc2ljVGFyZ2V0VHJhY2tpbmdTY2FsaW5nUG9saWN5UHJvcHMpIHtcbiAgICByZXR1cm4gbmV3IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeSh0aGlzLCBpZCwgeyAuLi5wcm9wcywgc2NhbGluZ1RhcmdldDogdGhpcyB9KTtcbiAgfVxufVxuXG4vKipcbiAqIEEgc2NoZWR1bGVkIHNjYWxpbmcgYWN0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2NhbGluZ1NjaGVkdWxlIHtcbiAgLyoqXG4gICAqIFdoZW4gdG8gcGVyZm9ybSB0aGlzIGFjdGlvbi5cbiAgICpcbiAgICogU3VwcG9ydCBmb3JtYXRzOlxuICAgKiAtIGF0KHl5eXktbW0tZGRUaGg6bW06c3MpXG4gICAqIC0gcmF0ZSh2YWx1ZSB1bml0KVxuICAgKiAtIGNyb24oZmllbGRzKVxuICAgKlxuICAgKiBcIkF0XCIgZXhwcmVzc2lvbnMgYXJlIHVzZWZ1bCBmb3Igb25lLXRpbWUgc2NoZWR1bGVzLiBTcGVjaWZ5IHRoZSB0aW1lIGluXG4gICAqIFVUQy5cbiAgICpcbiAgICogRm9yIFwicmF0ZVwiIGV4cHJlc3Npb25zLCB2YWx1ZSBpcyBhIHBvc2l0aXZlIGludGVnZXIsIGFuZCB1bml0IGlzIG1pbnV0ZSxcbiAgICogbWludXRlcywgaG91ciwgaG91cnMsIGRheSwgb3IgZGF5cy5cbiAgICpcbiAgICogRm9yIG1vcmUgaW5mb3JtYXRpb24gYWJvdXQgY3JvbiBleHByZXNzaW9ucywgc2VlIGh0dHBzOi8vZW4ud2lraXBlZGlhLm9yZy93aWtpL0Nyb24uXG4gICAqXG4gICAqIEBleGFtcGxlIHJhdGUoMTIgaG91cnMpXG4gICAqL1xuICByZWFkb25seSBzY2hlZHVsZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBXaGVuIHRoaXMgc2NoZWR1bGVkIGFjdGlvbiBiZWNvbWVzIGFjdGl2ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIHJ1bGUgaXMgYWN0aXZhdGUgaW1tZWRpYXRlbHlcbiAgICovXG4gIHJlYWRvbmx5IHN0YXJ0VGltZT86IERhdGVcblxuICAvKipcbiAgICogV2hlbiB0aGlzIHNjaGVkdWxlZCBhY3Rpb24gZXhwaXJlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIHJ1bGUgbmV2ZXIgZXhwaXJlcy5cbiAgICovXG4gIHJlYWRvbmx5IGVuZFRpbWU/OiBEYXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgbmV3IG1pbmltdW0gY2FwYWNpdHkuXG4gICAqXG4gICAqIER1cmluZyB0aGUgc2NoZWR1bGVkIHRpbWUsIGlmIHRoZSBjdXJyZW50IGNhcGFjaXR5IGlzIGJlbG93IHRoZSBtaW5pbXVtXG4gICAqIGNhcGFjaXR5LCBBcHBsaWNhdGlvbiBBdXRvIFNjYWxpbmcgc2NhbGVzIG91dCB0byB0aGUgbWluaW11bSBjYXBhY2l0eS5cbiAgICpcbiAgICogQXQgbGVhc3Qgb25lIG9mIG1heENhcGFjaXR5IGFuZCBtaW5DYXBhY2l0eSBtdXN0IGJlIHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBuZXcgbWluaW11bSBjYXBhY2l0eVxuICAgKi9cbiAgcmVhZG9ubHkgbWluQ2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBuZXcgbWF4aW11bSBjYXBhY2l0eS5cbiAgICpcbiAgICogRHVyaW5nIHRoZSBzY2hlZHVsZWQgdGltZSwgdGhlIGN1cnJlbnQgY2FwYWNpdHkgaXMgYWJvdmUgdGhlIG1heGltdW1cbiAgICogY2FwYWNpdHksIEFwcGxpY2F0aW9uIEF1dG8gU2NhbGluZyBzY2FsZXMgaW4gdG8gdGhlIG1heGltdW0gY2FwYWNpdHkuXG4gICAqXG4gICAqIEF0IGxlYXN0IG9uZSBvZiBtYXhDYXBhY2l0eSBhbmQgbWluQ2FwYWNpdHkgbXVzdCBiZSBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8gbmV3IG1heGltdW0gY2FwYWNpdHlcbiAgICovXG4gIHJlYWRvbmx5IG1heENhcGFjaXR5PzogbnVtYmVyO1xufVxuXG4vKipcbiAqIFRoZSBzZXJ2aWNlIHRoYXQgc3VwcG9ydHMgQXBwbGljYXRpb24gQXV0b1NjYWxpbmdcbiAqL1xuZXhwb3J0IGVudW0gU2VydmljZU5hbWVzcGFjZSB7XG4gIC8qKlxuICAgKiBFbGFzdGljIENvbnRhaW5lciBTZXJ2aWNlXG4gICAqL1xuICBFY3MgPSAnZWNzJyxcblxuICAvKipcbiAgICogRWxhc3RpYyBNYXAgUmVkdWNlXG4gICAqL1xuICBFbGFzdGljTWFwUmVkdWNlID0gJ2VsYXN0aWNtYXByZWR1Y2UnLFxuXG4gIC8qKlxuICAgKiBFbGFzdGljIENvbXB1dGUgQ2xvdWRcbiAgICovXG4gIEVjMiA9ICdlYzInLFxuXG4gIC8qKlxuICAgKiBBcHAgU3RyZWFtXG4gICAqL1xuICBBcHBTdHJlYW0gPSAnYXBwc3RyZWFtJyxcblxuICAvKipcbiAgICogRHluYW1vIERCXG4gICAqL1xuICBEeW5hbW9EYiA9ICdkeW5hbW9kYicsXG5cbiAgLyoqXG4gICAqIFJlbGF0aW9uYWwgRGF0YWJhc2UgU2VydmljZVxuICAgKi9cbiAgUmRzID0gJ3JkcycsXG5cbiAgLyoqXG4gICAqIFNhZ2VNYWtlclxuICAgKi9cbiAgU2FnZU1ha2VyID0gJ3NhZ2VtYWtlcicsXG5cbiAgLyoqXG4gICAqIEN1c3RvbSBSZXNvdXJjZVxuICAgKi9cbiAgQ3VzdG9tUmVzb3VyY2UgPSAnY3VzdG9tLXJlc291cmNlJyxcbn0iXX0=