/*
 * Decompiled with CFR 0.152.
 */
package org.mapsforge.map.writer;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.mapsforge.core.model.LatLong;
import org.mapsforge.core.util.LatLongUtils;
import org.mapsforge.core.util.MercatorProjection;
import org.mapsforge.map.writer.DeltaEncoder;
import org.mapsforge.map.writer.OSMTagMapping;
import org.mapsforge.map.writer.Serializer;
import org.mapsforge.map.writer.model.Encoding;
import org.mapsforge.map.writer.model.MapWriterConfiguration;
import org.mapsforge.map.writer.model.OSMTag;
import org.mapsforge.map.writer.model.TDNode;
import org.mapsforge.map.writer.model.TDWay;
import org.mapsforge.map.writer.model.TileBasedDataProcessor;
import org.mapsforge.map.writer.model.TileCoordinate;
import org.mapsforge.map.writer.model.TileData;
import org.mapsforge.map.writer.model.TileInfo;
import org.mapsforge.map.writer.model.WayDataBlock;
import org.mapsforge.map.writer.model.ZoomIntervalConfiguration;
import org.mapsforge.map.writer.util.GeoUtils;
import org.mapsforge.map.writer.util.JTSUtils;
import org.mapsforge.map.writer.util.OSMUtils;
import org.mapsforge.map.writer.util.PolyLabel;

public final class MapFileWriter {
    static final int HEADER_BUFFER_SIZE = 0x100000;
    static final Logger LOGGER = Logger.getLogger(MapFileWriter.class.getName());
    static final int MIN_TILE_BUFFER_SIZE = 0xF00000;
    static final int POI_DATA_BUFFER_SIZE = 0x100000;
    static final int TILE_BUFFER_SIZE = 0xA00000;
    static final int TILES_BUFFER_SIZE = 0x3200000;
    static final int WAY_BUFFER_SIZE = 0x100000;
    static final int WAY_DATA_BUFFER_SIZE = 0xA00000;
    private static final short BITMAP_COMMENT = 8;
    private static final short BITMAP_CREATED_WITH = 4;
    private static final short BITMAP_DEBUG = 128;
    private static final short BITMAP_MAP_START_POSITION = 64;
    private static final short BITMAP_MAP_START_ZOOM = 32;
    private static final short BITMAP_PREFERRED_LANGUAGES = 16;
    private static final short BITMAP_HOUSENUMBER = 64;
    private static final short BITMAP_NAME = 128;
    private static final short BITMAP_ELEVATION = 32;
    private static final short BITMAP_ENCODING = 4;
    private static final short BITMAP_LABEL = 16;
    private static final short BITMAP_MULTIPLE_WAY_BLOCKS = 8;
    private static final short BITMAP_REF = 32;
    private static final int BITMAP_INDEX_ENTRY_WATER = 128;
    private static final int BYTE_AMOUNT_SUBFILE_INDEX_PER_TILE = 5;
    private static final int BYTES_INT = 4;
    private static final int DEBUG_BLOCK_SIZE = 32;
    private static final int HALF_BYTE_SHIFT = 4;
    private static final String DEBUG_INDEX_START_STRING = "+++IndexStart+++";
    private static final String DEBUG_STRING_POI_HEAD = "***POIStart";
    private static final String DEBUG_STRING_POI_TAIL = "***";
    private static final String DEBUG_STRING_TILE_HEAD = "###TileStart";
    private static final String DEBUG_STRING_TILE_TAIL = "###";
    private static final String DEBUG_STRING_WAY_HEAD = "---WayStart";
    private static final String DEBUG_STRING_WAY_TAIL = "---";
    private static final int DUMMY_INT = -252645136;
    private static final long DUMMY_LONG = -1085102592571150096L;
    private static ExecutorService EXECUTOR_SERVICE;
    private static final int JTS_GEOMETRY_CACHE_SIZE = 50000;
    private static final String MAGIC_BYTE = "mapsforge binary OSM";
    private static final int OFFSET_FILE_SIZE = 28;
    private static final float PROGRESS_PERCENT_STEP = 10.0f;
    private static final String PROJECTION = "Mercator";
    private static final int SIZE_ZOOMINTERVAL_CONFIGURATION = 19;
    private static final TileInfo TILE_INFO;
    private static final int tileSize = 256;
    private static final Charset UTF8_CHARSET;

    public static void writeFile(MapWriterConfiguration configuration, TileBasedDataProcessor dataProcessor) throws IOException {
        EXECUTOR_SERVICE = Executors.newFixedThreadPool(configuration.getThreads());
        RandomAccessFile randomAccessFile = new RandomAccessFile(configuration.getOutputFile(), "rw");
        int amountOfZoomIntervals = dataProcessor.getZoomIntervalConfiguration().getNumberOfZoomIntervals();
        ByteBuffer containerHeaderBuffer = ByteBuffer.allocate(0x100000);
        int totalHeaderSize = MapFileWriter.writeHeaderBuffer(configuration, dataProcessor, containerHeaderBuffer);
        containerHeaderBuffer.reset();
        LoadingCache<TDWay, Geometry> jtsGeometryCache = CacheBuilder.newBuilder().maximumSize(50000L).concurrencyLevel(Runtime.getRuntime().availableProcessors() * 2).build(new JTSGeometryCacheLoader(dataProcessor));
        long currentFileSize = totalHeaderSize;
        for (int i = 0; i < amountOfZoomIntervals; ++i) {
            long subfileSize = MapFileWriter.writeSubfile(currentFileSize, i, dataProcessor, jtsGeometryCache, randomAccessFile, configuration);
            MapFileWriter.writeSubfileMetaDataToContainerHeader(dataProcessor.getZoomIntervalConfiguration(), i, currentFileSize, subfileSize, containerHeaderBuffer);
            currentFileSize += subfileSize;
        }
        randomAccessFile.seek(0L);
        randomAccessFile.write(containerHeaderBuffer.array(), 0, totalHeaderSize);
        long fileSize = randomAccessFile.length();
        randomAccessFile.seek(28L);
        randomAccessFile.writeLong(fileSize);
        randomAccessFile.close();
        CacheStats stats = jtsGeometryCache.stats();
        LOGGER.fine("Tag values stats:\n" + OSMUtils.logValueTypeCount());
        LOGGER.info("JTS Geometry cache hit rate: " + stats.hitRate());
        LOGGER.info("JTS Geometry total load time: " + stats.totalLoadTime() / 1000L);
        LOGGER.info("Finished writing file.");
    }

    public static void release() {
        EXECUTOR_SERVICE.shutdown();
    }

    static byte infoByteOptmizationParams(MapWriterConfiguration configuration) {
        byte infoByte = 0;
        if (configuration.isDebugStrings()) {
            infoByte = (byte)(infoByte | 0x80);
        }
        if (configuration.getMapStartPosition() != null) {
            infoByte = (byte)(infoByte | 0x40);
        }
        if (configuration.hasMapStartZoomLevel()) {
            infoByte = (byte)(infoByte | 0x20);
        }
        if (configuration.getPreferredLanguages() != null && !configuration.getPreferredLanguages().isEmpty()) {
            infoByte = (byte)(infoByte | 0x10);
        }
        if (configuration.getComment() != null) {
            infoByte = (byte)(infoByte | 8);
        }
        infoByte = (byte)(infoByte | 4);
        return infoByte;
    }

    static byte infoBytePOIFeatures(String name, int elevation, String housenumber) {
        byte infoByte = 0;
        if (name != null && !name.isEmpty()) {
            infoByte = (byte)(infoByte | 0x80);
        }
        if (housenumber != null && !housenumber.isEmpty()) {
            infoByte = (byte)(infoByte | 0x40);
        }
        if (elevation != 0) {
            infoByte = (byte)(infoByte | 0x20);
        }
        return infoByte;
    }

    static byte infoBytePoiLayerAndTagAmount(TDNode node) {
        int tagAmount;
        int layer = node.getLayer();
        layer = layer < 0 ? 0 : (layer > 10 ? 10 : layer);
        int n = tagAmount = node.getTags() == null ? 0 : node.getTags().size();
        if (tagAmount > 15) {
            LOGGER.severe("Too many tags: " + node.toStringDetailed());
            throw new RuntimeException("more than 15 tags aren't supported");
        }
        return (byte)(layer << 4 | (short)tagAmount);
    }

    static byte infoByteWayFeatures(TDWay way, WayPreprocessingResult wpr) {
        WayDataBlock wayDataBlock;
        byte infoByte = 0;
        if (way.getName() != null && !way.getName().isEmpty()) {
            infoByte = (byte)(infoByte | 0x80);
        }
        if (way.getHouseNumber() != null && !way.getHouseNumber().isEmpty()) {
            infoByte = (byte)(infoByte | 0x40);
        }
        if (way.getRef() != null && !way.getRef().isEmpty()) {
            infoByte = (byte)(infoByte | 0x20);
        }
        if (wpr.getLabelPosition() != null) {
            infoByte = (byte)(infoByte | 0x10);
        }
        if (wpr.getWayDataBlocks().size() > 1) {
            infoByte = (byte)(infoByte | 8);
        }
        if (!wpr.getWayDataBlocks().isEmpty() && (wayDataBlock = wpr.getWayDataBlocks().get(0)).getEncoding() == Encoding.DOUBLE_DELTA) {
            infoByte = (byte)(infoByte | 4);
        }
        return infoByte;
    }

    static byte infoByteWayLayerAndTagAmount(TDWay way) {
        int tagAmount;
        int layer = way.getLayer();
        layer = layer < 0 ? 0 : (layer > 10 ? 10 : layer);
        int n = tagAmount = way.getTags() == null ? 0 : way.getTags().size();
        if (tagAmount > 15) {
            LOGGER.severe("Too many tags: " + way.toStringDetailed());
            throw new RuntimeException("more than 15 tags aren't supported");
        }
        return (byte)(layer << 4 | (short)tagAmount);
    }

    static void processPOI(TDNode poi, int currentTileLat, int currentTileLon, boolean debugStrings, ByteBuffer poiBuffer) {
        if (debugStrings) {
            StringBuilder sb = new StringBuilder();
            sb.append(DEBUG_STRING_POI_HEAD).append(poi.getId()).append(DEBUG_STRING_POI_TAIL);
            poiBuffer.put(sb.toString().getBytes(UTF8_CHARSET));
            MapFileWriter.appendWhitespace(32 - sb.toString().getBytes(UTF8_CHARSET).length, poiBuffer);
        }
        poiBuffer.put(Serializer.getVariableByteSigned(poi.getLatitude() - currentTileLat));
        poiBuffer.put(Serializer.getVariableByteSigned(poi.getLongitude() - currentTileLon));
        poiBuffer.put(MapFileWriter.infoBytePoiLayerAndTagAmount(poi));
        Map<Short, Object> tags = poi.getTags();
        if (tags != null) {
            ArrayList<Object> values = new ArrayList<Object>();
            for (Map.Entry<Short, Object> entry : tags.entrySet()) {
                poiBuffer.put(Serializer.getVariableByteUnsigned(MapFileWriter.mappedPoiTagID(entry.getKey())));
                if (entry.getValue() == null) continue;
                values.add(entry.getValue());
            }
            for (Map.Entry<Short, Object> entry : values) {
                if (entry instanceof Byte) {
                    poiBuffer.putShort(((Byte)((Object)entry)).byteValue());
                    continue;
                }
                if (entry instanceof Integer) {
                    poiBuffer.putInt((Integer)((Object)entry));
                    continue;
                }
                if (entry instanceof Float) {
                    poiBuffer.putFloat(((Float)((Object)entry)).floatValue());
                    continue;
                }
                if (entry instanceof Short) {
                    poiBuffer.putShort((Short)((Object)entry));
                    continue;
                }
                if (!(entry instanceof String)) continue;
                MapFileWriter.writeUTF8((String)((Object)entry), poiBuffer);
            }
        }
        poiBuffer.put(MapFileWriter.infoBytePOIFeatures(poi.getName(), poi.getElevation(), poi.getHouseNumber()));
        if (poi.getName() != null && !poi.getName().isEmpty()) {
            MapFileWriter.writeUTF8(poi.getName(), poiBuffer);
        }
        if (poi.getHouseNumber() != null && !poi.getHouseNumber().isEmpty()) {
            MapFileWriter.writeUTF8(poi.getHouseNumber(), poiBuffer);
        }
        if (poi.getElevation() != 0) {
            poiBuffer.put(Serializer.getVariableByteSigned(poi.getElevation()));
        }
    }

    static void processWay(WayPreprocessingResult wpr, TDWay way, int currentTileLat, int currentTileLon, ByteBuffer wayBuffer) {
        wayBuffer.putShort(wpr.getSubtileMask());
        wayBuffer.put(MapFileWriter.infoByteWayLayerAndTagAmount(way));
        Map<Short, Object> tags = way.getTags();
        if (tags != null) {
            ArrayList<Object> values = new ArrayList<Object>();
            for (Map.Entry<Short, Object> entry : tags.entrySet()) {
                wayBuffer.put(Serializer.getVariableByteUnsigned(MapFileWriter.mappedWayTagID(entry.getKey())));
                if (entry.getValue() == null) continue;
                values.add(entry.getValue());
            }
            for (Map.Entry<Short, Object> entry : values) {
                if (entry instanceof Byte) {
                    wayBuffer.put((Byte)((Object)entry));
                    continue;
                }
                if (entry instanceof Integer) {
                    wayBuffer.putInt((Integer)((Object)entry));
                    continue;
                }
                if (entry instanceof Float) {
                    wayBuffer.putFloat(((Float)((Object)entry)).floatValue());
                    continue;
                }
                if (entry instanceof Short) {
                    wayBuffer.putShort((Short)((Object)entry));
                    continue;
                }
                if (!(entry instanceof String)) continue;
                MapFileWriter.writeUTF8((String)((Object)entry), wayBuffer);
            }
        }
        wayBuffer.put(MapFileWriter.infoByteWayFeatures(way, wpr));
        if (way.getName() != null && !way.getName().isEmpty()) {
            MapFileWriter.writeUTF8(way.getName(), wayBuffer);
        }
        if (way.getHouseNumber() != null && !way.getHouseNumber().isEmpty()) {
            MapFileWriter.writeUTF8(way.getHouseNumber(), wayBuffer);
        }
        if (way.getRef() != null && !way.getRef().isEmpty()) {
            MapFileWriter.writeUTF8(way.getRef(), wayBuffer);
        }
        if (wpr.getLabelPosition() != null) {
            int firstWayStartLat = wpr.getWayDataBlocks().get(0).getOuterWay().get(0);
            int firstWayStartLon = wpr.getWayDataBlocks().get(0).getOuterWay().get(1);
            wayBuffer.put(Serializer.getVariableByteSigned(LatLongUtils.degreesToMicrodegrees(wpr.getLabelPosition().latitude) - firstWayStartLat));
            wayBuffer.put(Serializer.getVariableByteSigned(LatLongUtils.degreesToMicrodegrees(wpr.getLabelPosition().longitude) - firstWayStartLon));
        }
        if (wpr.getWayDataBlocks().size() > 1) {
            wayBuffer.put(Serializer.getVariableByteUnsigned(wpr.getWayDataBlocks().size()));
        }
        for (WayDataBlock wayDataBlock : wpr.getWayDataBlocks()) {
            if (wayDataBlock.getInnerWays() != null && !wayDataBlock.getInnerWays().isEmpty()) {
                wayBuffer.put(Serializer.getVariableByteUnsigned(1 + wayDataBlock.getInnerWays().size()));
            } else {
                wayBuffer.put(Serializer.getVariableByteUnsigned(1));
            }
            MapFileWriter.writeWay(wayDataBlock.getOuterWay(), currentTileLat, currentTileLon, wayBuffer);
            if (wayDataBlock.getInnerWays() == null || wayDataBlock.getInnerWays().isEmpty()) continue;
            for (List<Integer> innerWayCoordinates : wayDataBlock.getInnerWays()) {
                MapFileWriter.writeWay(innerWayCoordinates, currentTileLat, currentTileLon, wayBuffer);
            }
        }
    }

    static int writeHeaderBuffer(MapWriterConfiguration configuration, TileBasedDataProcessor dataProcessor, ByteBuffer containerHeaderBuffer) {
        OSMTag tag;
        LOGGER.fine("writing header");
        LOGGER.fine("Bounding box for file: " + dataProcessor.getBoundingBox().toString());
        byte[] magicBytes = MAGIC_BYTE.getBytes(UTF8_CHARSET);
        containerHeaderBuffer.put(magicBytes);
        int headerSizePosition = containerHeaderBuffer.position();
        containerHeaderBuffer.putInt(-252645136);
        containerHeaderBuffer.putInt(configuration.getFileSpecificationVersion());
        containerHeaderBuffer.putLong(-1085102592571150096L);
        containerHeaderBuffer.putLong(System.currentTimeMillis());
        containerHeaderBuffer.putInt(LatLongUtils.degreesToMicrodegrees(dataProcessor.getBoundingBox().minLatitude));
        containerHeaderBuffer.putInt(LatLongUtils.degreesToMicrodegrees(dataProcessor.getBoundingBox().minLongitude));
        containerHeaderBuffer.putInt(LatLongUtils.degreesToMicrodegrees(dataProcessor.getBoundingBox().maxLatitude));
        containerHeaderBuffer.putInt(LatLongUtils.degreesToMicrodegrees(dataProcessor.getBoundingBox().maxLongitude));
        containerHeaderBuffer.putShort((short)256);
        MapFileWriter.writeUTF8(PROJECTION, containerHeaderBuffer);
        containerHeaderBuffer.put(MapFileWriter.infoByteOptmizationParams(configuration));
        LatLong mapStartPosition = configuration.getMapStartPosition();
        if (mapStartPosition != null) {
            containerHeaderBuffer.putInt(LatLongUtils.degreesToMicrodegrees(mapStartPosition.latitude));
            containerHeaderBuffer.putInt(LatLongUtils.degreesToMicrodegrees(mapStartPosition.longitude));
        }
        if (configuration.hasMapStartZoomLevel()) {
            containerHeaderBuffer.put((byte)configuration.getMapStartZoomLevel());
        }
        if (configuration.getPreferredLanguages() != null && !configuration.getPreferredLanguages().isEmpty()) {
            String langStr = "";
            for (String preferredLanguage : configuration.getPreferredLanguages()) {
                langStr = langStr + (langStr.length() > 0 ? "," : "") + preferredLanguage;
            }
            MapFileWriter.writeUTF8(langStr, containerHeaderBuffer);
        }
        if (configuration.getComment() != null) {
            MapFileWriter.writeUTF8(configuration.getComment(), containerHeaderBuffer);
        }
        MapFileWriter.writeUTF8(configuration.getWriterVersion(), containerHeaderBuffer);
        OSMTagMapping mapping = configuration.getTagMapping();
        containerHeaderBuffer.putShort((short)mapping.getOptimizedPoiIds().size());
        Iterator<Object> iterator = mapping.getOptimizedPoiIds().keySet().iterator();
        while (iterator.hasNext()) {
            short tagId = (Short)iterator.next();
            tag = mapping.getPoiTag(tagId);
            MapFileWriter.writeUTF8(tag.tagKey(), containerHeaderBuffer);
        }
        containerHeaderBuffer.putShort((short)mapping.getOptimizedWayIds().size());
        iterator = mapping.getOptimizedWayIds().keySet().iterator();
        while (iterator.hasNext()) {
            short tagId = (Short)iterator.next();
            tag = mapping.getWayTag(tagId);
            MapFileWriter.writeUTF8(tag.tagKey(), containerHeaderBuffer);
        }
        int numberOfZoomIntervals = dataProcessor.getZoomIntervalConfiguration().getNumberOfZoomIntervals();
        containerHeaderBuffer.put((byte)numberOfZoomIntervals);
        containerHeaderBuffer.mark();
        containerHeaderBuffer.position(containerHeaderBuffer.position() + 19 * numberOfZoomIntervals);
        int headerSize = containerHeaderBuffer.position() - headerSizePosition - 4;
        containerHeaderBuffer.putInt(headerSizePosition, headerSize);
        return containerHeaderBuffer.position();
    }

    static void writeWayNodes(List<Integer> waynodes, int currentTileLat, int currentTileLon, ByteBuffer buffer) {
        if (!waynodes.isEmpty() && waynodes.size() % 2 == 0) {
            Iterator<Integer> waynodeIterator = waynodes.iterator();
            buffer.put(Serializer.getVariableByteSigned(waynodeIterator.next() - currentTileLat));
            buffer.put(Serializer.getVariableByteSigned(waynodeIterator.next() - currentTileLon));
            while (waynodeIterator.hasNext()) {
                buffer.put(Serializer.getVariableByteSigned(waynodeIterator.next()));
            }
        }
    }

    static void writeZoomLevelTable(int[][] entitiesPerZoomLevel, ByteBuffer tileBuffer) {
        for (int[] entityCount : entitiesPerZoomLevel) {
            tileBuffer.put(Serializer.getVariableByteUnsigned(entityCount[0]));
            tileBuffer.put(Serializer.getVariableByteUnsigned(entityCount[1]));
        }
    }

    private static void appendWhitespace(int amount, ByteBuffer buffer) {
        for (int i = 0; i < amount; ++i) {
            buffer.put((byte)32);
        }
    }

    private static int mappedPoiTagID(short original) {
        return OSMTagMapping.getInstance().getOptimizedPoiIds().get(original).intValue();
    }

    private static int mappedWayTagID(short original) {
        return OSMTagMapping.getInstance().getOptimizedWayIds().get(original).intValue();
    }

    private static void processIndexEntry(TileCoordinate tileCoordinate, ByteBuffer indexBuffer, long currentSubfileOffset) {
        byte[] indexBytes = Serializer.getFiveBytes(currentSubfileOffset);
        if (TILE_INFO.isWaterTile(tileCoordinate)) {
            indexBytes[0] = (byte)(indexBytes[0] | 0x80);
        }
        indexBuffer.put(indexBytes);
    }

    private static void processTile(MapWriterConfiguration configuration, TileCoordinate tileCoordinate, TileBasedDataProcessor dataProcessor, LoadingCache<TDWay, Geometry> jtsGeometryCache, int zoomIntervalIndex, ByteBuffer tileBuffer, ByteBuffer poiDataBuffer, ByteBuffer wayDataBuffer, ByteBuffer wayBuffer) {
        tileBuffer.clear();
        poiDataBuffer.clear();
        wayDataBuffer.clear();
        wayBuffer.clear();
        TileData currentTile = dataProcessor.getTile(zoomIntervalIndex, tileCoordinate.getX(), tileCoordinate.getY());
        int currentTileLat = LatLongUtils.degreesToMicrodegrees(MercatorProjection.tileYToLatitude(tileCoordinate.getY(), tileCoordinate.getZoomlevel()));
        int currentTileLon = LatLongUtils.degreesToMicrodegrees(MercatorProjection.tileXToLongitude(tileCoordinate.getX(), tileCoordinate.getZoomlevel()));
        byte minZoomCurrentInterval = dataProcessor.getZoomIntervalConfiguration().getMinZoom(zoomIntervalIndex);
        byte maxZoomCurrentInterval = dataProcessor.getZoomIntervalConfiguration().getMaxZoom(zoomIntervalIndex);
        Map<Byte, List<TDNode>> poisByZoomlevel = currentTile.poisByZoomlevel(minZoomCurrentInterval, maxZoomCurrentInterval);
        Map<Byte, List<TDWay>> waysByZoomlevel = currentTile.waysByZoomlevel(minZoomCurrentInterval, maxZoomCurrentInterval);
        if (!poisByZoomlevel.isEmpty() || !waysByZoomlevel.isEmpty()) {
            int indexEntitiesPerZoomLevelTable;
            byte zoomlevel;
            if (configuration.isDebugStrings()) {
                MapFileWriter.writeTileSignature(tileCoordinate, tileBuffer);
            }
            int amountZoomLevels = maxZoomCurrentInterval - minZoomCurrentInterval + 1;
            int[][] entitiesPerZoomLevel = new int[amountZoomLevels][2];
            for (zoomlevel = minZoomCurrentInterval; zoomlevel <= maxZoomCurrentInterval; zoomlevel = (byte)(zoomlevel + 1)) {
                indexEntitiesPerZoomLevelTable = zoomlevel - minZoomCurrentInterval;
                List<TDNode> pois = poisByZoomlevel.get(zoomlevel);
                if (pois == null) continue;
                for (TDNode tDNode : pois) {
                    MapFileWriter.processPOI(tDNode, currentTileLat, currentTileLon, configuration.isDebugStrings(), poiDataBuffer);
                }
                int[] nArray = entitiesPerZoomLevel[indexEntitiesPerZoomLevelTable];
                nArray[0] = nArray[0] + pois.size();
            }
            for (zoomlevel = minZoomCurrentInterval; zoomlevel <= maxZoomCurrentInterval; zoomlevel = (byte)(zoomlevel + 1)) {
                indexEntitiesPerZoomLevelTable = zoomlevel - minZoomCurrentInterval;
                List<TDWay> ways = waysByZoomlevel.get(zoomlevel);
                if (ways == null) continue;
                ArrayList<WayPreprocessingCallable> callables = new ArrayList<WayPreprocessingCallable>();
                for (TDWay way : ways) {
                    if (way.isInvalid()) continue;
                    callables.add(new WayPreprocessingCallable(way, tileCoordinate, maxZoomCurrentInterval, jtsGeometryCache, configuration));
                }
                try {
                    List list = EXECUTOR_SERVICE.invokeAll(callables);
                    for (Future wprFuture : list) {
                        WayPreprocessingResult wpr;
                        try {
                            wpr = (WayPreprocessingResult)wprFuture.get();
                        }
                        catch (ExecutionException e) {
                            LOGGER.log(Level.WARNING, "error in parallel preprocessing of ways", e);
                            continue;
                        }
                        if (wpr == null) continue;
                        wayBuffer.clear();
                        int[] nArray = entitiesPerZoomLevel[indexEntitiesPerZoomLevelTable];
                        nArray[1] = nArray[1] + 1;
                        if (configuration.isDebugStrings()) {
                            MapFileWriter.writeWaySignature(wpr.getWay(), wayDataBuffer);
                        }
                        MapFileWriter.processWay(wpr, wpr.getWay(), currentTileLat, currentTileLon, wayBuffer);
                        wayDataBuffer.put(Serializer.getVariableByteUnsigned(wayBuffer.position()));
                        wayDataBuffer.put(wayBuffer.array(), 0, wayBuffer.position());
                    }
                    continue;
                }
                catch (InterruptedException interruptedException) {
                    LOGGER.log(Level.WARNING, "error in parallel preprocessing of ways", interruptedException);
                    throw new RuntimeException(interruptedException);
                }
            }
            MapFileWriter.writeZoomLevelTable(entitiesPerZoomLevel, tileBuffer);
            tileBuffer.put(Serializer.getVariableByteUnsigned(poiDataBuffer.position()));
            tileBuffer.put(poiDataBuffer.array(), 0, poiDataBuffer.position());
            tileBuffer.put(wayDataBuffer.array(), 0, wayDataBuffer.position());
        }
    }

    private static void writeIndex(ByteBuffer indexBuffer, long startPositionSubfile, long subFileSize, RandomAccessFile randomAccessFile) throws IOException {
        randomAccessFile.seek(startPositionSubfile);
        randomAccessFile.write(indexBuffer.array());
        randomAccessFile.seek(subFileSize);
    }

    private static long writeSubfile(long startPositionSubfile, int zoomIntervalIndex, TileBasedDataProcessor dataStore, LoadingCache<TDWay, Geometry> jtsGeometryCache, RandomAccessFile randomAccessFile, MapWriterConfiguration configuration) throws IOException {
        LOGGER.fine("writing data for zoom interval " + zoomIntervalIndex + ", number of tiles: " + dataStore.getTileGridLayout(zoomIntervalIndex).getAmountTilesHorizontal() * dataStore.getTileGridLayout(zoomIntervalIndex).getAmountTilesVertical());
        TileCoordinate upperLeft = dataStore.getTileGridLayout(zoomIntervalIndex).getUpperLeft();
        int lengthX = dataStore.getTileGridLayout(zoomIntervalIndex).getAmountTilesHorizontal();
        int lengthY = dataStore.getTileGridLayout(zoomIntervalIndex).getAmountTilesVertical();
        int amountTiles = lengthX * lengthY;
        double amountOfTilesInPercentStep = amountTiles;
        if ((float)amountTiles > 10.0f) {
            amountOfTilesInPercentStep = Math.ceil((float)amountTiles / 10.0f);
        }
        int processedTiles = 0;
        byte baseZoomCurrentInterval = dataStore.getZoomIntervalConfiguration().getBaseZoom(zoomIntervalIndex);
        int tileAmountInBytes = lengthX * lengthY * 5;
        int indexBufferSize = tileAmountInBytes + (configuration.isDebugStrings() ? DEBUG_INDEX_START_STRING.getBytes(UTF8_CHARSET).length : 0);
        ByteBuffer indexBuffer = ByteBuffer.allocate(indexBufferSize);
        ByteBuffer tileBuffer = ByteBuffer.allocate(0xA00000);
        ByteBuffer wayDataBuffer = ByteBuffer.allocate(0xA00000);
        ByteBuffer wayBuffer = ByteBuffer.allocate(0x100000);
        ByteBuffer poiDataBuffer = ByteBuffer.allocate(0x100000);
        ByteBuffer multipleTilesBuffer = ByteBuffer.allocate(0x3200000);
        if (configuration.isDebugStrings()) {
            indexBuffer.put(DEBUG_INDEX_START_STRING.getBytes(UTF8_CHARSET));
        }
        long currentSubfileOffset = indexBufferSize;
        randomAccessFile.seek(startPositionSubfile + (long)indexBufferSize);
        for (int tileY = upperLeft.getY(); tileY < upperLeft.getY() + lengthY; ++tileY) {
            for (int tileX = upperLeft.getX(); tileX < upperLeft.getX() + lengthX; ++tileX) {
                TileCoordinate tileCoordinate = new TileCoordinate(tileX, tileY, baseZoomCurrentInterval);
                MapFileWriter.processIndexEntry(tileCoordinate, indexBuffer, currentSubfileOffset);
                MapFileWriter.processTile(configuration, tileCoordinate, dataStore, jtsGeometryCache, zoomIntervalIndex, tileBuffer, poiDataBuffer, wayDataBuffer, wayBuffer);
                currentSubfileOffset += (long)tileBuffer.position();
                MapFileWriter.writeTile(multipleTilesBuffer, tileBuffer, randomAccessFile);
                if ((double)(++processedTiles) % amountOfTilesInPercentStep != 0.0) continue;
                if (processedTiles == amountTiles) {
                    LOGGER.info("written 100% of sub file for zoom interval " + (zoomIntervalIndex + 1) + " (" + dataStore.getZoomIntervalConfiguration().toString(zoomIntervalIndex) + ")");
                    continue;
                }
                LOGGER.info("written " + Math.round((double)processedTiles / amountOfTilesInPercentStep / 10.0 * 100.0) + "% of sub file for zoom interval " + (zoomIntervalIndex + 1) + " (" + dataStore.getZoomIntervalConfiguration().toString(zoomIntervalIndex) + ")");
            }
        }
        if (multipleTilesBuffer.position() > 0) {
            randomAccessFile.write(multipleTilesBuffer.array(), 0, multipleTilesBuffer.position());
        }
        MapFileWriter.writeIndex(indexBuffer, startPositionSubfile, currentSubfileOffset, randomAccessFile);
        return currentSubfileOffset;
    }

    private static void writeSubfileMetaDataToContainerHeader(ZoomIntervalConfiguration zoomIntervalConfiguration, int i, long startIndexOfSubfile, long subfileSize, ByteBuffer buffer) {
        byte minZoomCurrentInterval = zoomIntervalConfiguration.getMinZoom(i);
        byte maxZoomCurrentInterval = zoomIntervalConfiguration.getMaxZoom(i);
        byte baseZoomCurrentInterval = zoomIntervalConfiguration.getBaseZoom(i);
        buffer.put(baseZoomCurrentInterval);
        buffer.put(minZoomCurrentInterval);
        buffer.put(maxZoomCurrentInterval);
        buffer.putLong(startIndexOfSubfile);
        buffer.putLong(subfileSize);
    }

    private static void writeTile(ByteBuffer multipleTilesBuffer, ByteBuffer tileBuffer, RandomAccessFile randomAccessFile) throws IOException {
        multipleTilesBuffer.put(tileBuffer.array(), 0, tileBuffer.position());
        if (multipleTilesBuffer.remaining() < 0xF00000) {
            randomAccessFile.write(multipleTilesBuffer.array(), 0, multipleTilesBuffer.position());
            multipleTilesBuffer.clear();
        }
    }

    private static void writeTileSignature(TileCoordinate tileCoordinate, ByteBuffer tileBuffer) {
        StringBuilder sb = new StringBuilder();
        sb.append(DEBUG_STRING_TILE_HEAD).append(tileCoordinate.getX()).append(",").append(tileCoordinate.getY()).append(DEBUG_STRING_TILE_TAIL);
        tileBuffer.put(sb.toString().getBytes(UTF8_CHARSET));
        MapFileWriter.appendWhitespace(32 - sb.toString().getBytes(UTF8_CHARSET).length, tileBuffer);
    }

    private static void writeUTF8(String string, ByteBuffer buffer) {
        buffer.put(Serializer.getVariableByteUnsigned(string.getBytes(UTF8_CHARSET).length));
        buffer.put(string.getBytes(UTF8_CHARSET));
    }

    private static void writeWay(List<Integer> wayNodes, int currentTileLat, int currentTileLon, ByteBuffer buffer) {
        int wayNodeCount = wayNodes.size() / 2;
        if (wayNodeCount < 2) {
            LOGGER.warning("Invalid way node count: " + wayNodeCount);
        }
        buffer.put(Serializer.getVariableByteUnsigned(wayNodeCount));
        MapFileWriter.writeWayNodes(wayNodes, currentTileLat, currentTileLon, buffer);
    }

    private static void writeWaySignature(TDWay way, ByteBuffer tileBuffer) {
        StringBuilder sb = new StringBuilder();
        sb.append(DEBUG_STRING_WAY_HEAD).append(way.getId()).append(DEBUG_STRING_WAY_TAIL);
        tileBuffer.put(sb.toString().getBytes(UTF8_CHARSET));
        MapFileWriter.appendWhitespace(32 - sb.toString().getBytes(UTF8_CHARSET).length, tileBuffer);
    }

    private MapFileWriter() {
    }

    static {
        TILE_INFO = TileInfo.getInstance();
        UTF8_CHARSET = Charset.forName("utf8");
    }

    private static class WayPreprocessingResult {
        final LatLong labelPosition;
        final short subtileMask;
        final TDWay way;
        final List<WayDataBlock> wayDataBlocks;

        WayPreprocessingResult(TDWay way, List<WayDataBlock> wayDataBlocks, LatLong labelPosition, short subtileMask) {
            this.way = way;
            this.wayDataBlocks = wayDataBlocks;
            this.labelPosition = labelPosition;
            this.subtileMask = subtileMask;
        }

        LatLong getLabelPosition() {
            return this.labelPosition;
        }

        short getSubtileMask() {
            return this.subtileMask;
        }

        TDWay getWay() {
            return this.way;
        }

        List<WayDataBlock> getWayDataBlocks() {
            return this.wayDataBlocks;
        }
    }

    private static class WayPreprocessingCallable
    implements Callable<WayPreprocessingResult> {
        private final MapWriterConfiguration configuration;
        private final LoadingCache<TDWay, Geometry> jtsGeometryCache;
        private final byte maxZoomInterval;
        private final TileCoordinate tile;
        private final TDWay way;

        WayPreprocessingCallable(TDWay way, TileCoordinate tile, byte maxZoomInterval, LoadingCache<TDWay, Geometry> jtsGeometryCache, MapWriterConfiguration configuration) {
            this.way = way;
            this.tile = tile;
            this.maxZoomInterval = maxZoomInterval;
            this.jtsGeometryCache = jtsGeometryCache;
            this.configuration = configuration;
        }

        @Override
        public WayPreprocessingResult call() {
            List<WayDataBlock> blocks;
            Geometry originalGeometry;
            try {
                originalGeometry = this.jtsGeometryCache.get(this.way);
            }
            catch (ExecutionException e) {
                this.way.setInvalid(true);
                return null;
            }
            Geometry processedGeometry = originalGeometry;
            if (((originalGeometry instanceof Polygon || originalGeometry instanceof MultiPolygon) && this.configuration.isPolygonClipping() || (originalGeometry instanceof LineString || originalGeometry instanceof MultiLineString) && this.configuration.isWayClipping()) && (processedGeometry = GeoUtils.clipToTile(this.way, originalGeometry, this.tile, this.configuration.getBboxEnlargement())) == null) {
                return null;
            }
            if (this.configuration.getSimplification() > 0.0 && this.tile.getZoomlevel() <= this.configuration.getSimplificationMaxZoom() && (processedGeometry = GeoUtils.simplifyGeometry(this.way, processedGeometry, this.maxZoomInterval, 256, this.configuration.getSimplification())) == null) {
                return null;
            }
            if (processedGeometry.getCoordinates().length > 2000) {
                LOGGER.info("Large geometry " + this.way.getId() + " (" + processedGeometry.getCoordinates().length + " coords, down from " + originalGeometry.getCoordinates().length + " coords)");
            }
            if ((blocks = GeoUtils.toWayDataBlockList(processedGeometry)) == null) {
                return null;
            }
            if (blocks.isEmpty()) {
                LOGGER.finer("empty list of way data blocks after preprocessing way: " + this.way.getId());
                return null;
            }
            short subtileMask = GeoUtils.computeBitmask(processedGeometry, this.tile, this.configuration.getBboxEnlargement());
            LatLong labelCoordinate = null;
            if (this.way.isValidClosedLine()) {
                boolean labelPosition = this.configuration.isLabelPosition();
                if (!labelPosition) {
                    OSMTag tag;
                    List<OSMTag> tags = this.configuration.getTagMapping().getWayTags(this.way.getTags().keySet());
                    Iterator<OSMTag> iterator = tags.iterator();
                    while (iterator.hasNext() && !(labelPosition = (tag = iterator.next()).isLabelPosition())) {
                    }
                }
                if (labelPosition) {
                    if (this.configuration.isPolylabel()) {
                        Point labelPoint = PolyLabel.get(originalGeometry);
                        labelCoordinate = new LatLong(labelPoint.getY(), labelPoint.getX());
                    } else {
                        labelCoordinate = GeoUtils.computeInteriorPoint(originalGeometry);
                    }
                }
            }
            switch (this.configuration.getEncodingChoice()) {
                case SINGLE: {
                    blocks = DeltaEncoder.encode(blocks, Encoding.DELTA);
                    break;
                }
                case DOUBLE: {
                    blocks = DeltaEncoder.encode(blocks, Encoding.DOUBLE_DELTA);
                    break;
                }
                case AUTO: {
                    List<WayDataBlock> blocksDelta = DeltaEncoder.encode(blocks, Encoding.DELTA);
                    List<WayDataBlock> blocksDoubleDelta = DeltaEncoder.encode(blocks, Encoding.DOUBLE_DELTA);
                    int simDelta = DeltaEncoder.simulateSerialization(blocksDelta);
                    int simDoubleDelta = DeltaEncoder.simulateSerialization(blocksDoubleDelta);
                    blocks = simDelta <= simDoubleDelta ? blocksDelta : blocksDoubleDelta;
                }
            }
            return new WayPreprocessingResult(this.way, blocks, labelCoordinate, subtileMask);
        }
    }

    private static class JTSGeometryCacheLoader
    extends CacheLoader<TDWay, Geometry> {
        private final TileBasedDataProcessor datastore;

        JTSGeometryCacheLoader(TileBasedDataProcessor datastore) {
            this.datastore = datastore;
        }

        @Override
        public Geometry load(TDWay way) throws Exception {
            if (way.isInvalid()) {
                throw new Exception("way is known to be invalid: " + way.getId());
            }
            List<TDWay> innerWaysOfMultipolygon = this.datastore.getInnerWaysOfMultipolygon(way.getId());
            Geometry geometry = JTSUtils.toJtsGeometry(way, innerWaysOfMultipolygon);
            if (geometry == null) {
                way.setInvalid(true);
                throw new Exception("cannot create geometry for way with id: " + way.getId());
            }
            return geometry;
        }
    }
}

