"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldUtils = exports.Context = exports.Data = exports.JsonPath = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const json_path_1 = require("./private/json-path");
/**
 * Extract a field from the State Machine data or context that gets passed around between states.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 * @stability stable
 */
class JsonPath {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path.
     *
     * @stability stable
     */
    static stringAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path.
     *
     * @stability stable
     */
    static listAt(path) {
        // does not apply to task context
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path.
     *
     * @stability stable
     */
    static numberAt(path) {
        validateJsonPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Reference a complete (complex) object in a JSON path location.
     *
     * @stability stable
     */
    static objectAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path);
    }
    /**
     * Use the entire data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @stability stable
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path.
     *
     * @param value string to be evaluated.
     * @stability stable
     */
    static isEncodedJsonPath(value) {
        return !!json_path_1.jsonPathString(value);
    }
    /**
     * Return the Task Token field.
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     *
     * @stability stable
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @stability stable
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    /**
     * Make an intrinsic States.Array expression.
     *
     * Combine any number of string literals or JsonPath expressions into an array.
     *
     * Use this function if the value of an array element directly has to come
     * from a JSON Path expression (either the State object or the Context object).
     *
     * If the array contains object literals whose values come from a JSON path
     * expression, you do not need to use this function.
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static array(...values) {
        return new json_path_1.JsonPathToken(`States.Array(${values.map(json_path_1.renderInExpression).join(', ')})`).toString();
    }
    /**
     * Make an intrinsic States.Format expression.
     *
     * This can be used to embed JSON Path variables inside a format string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.format('Hello, my name is {}.', sfn.JsonPath.stringAt('$.name'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static format(formatString, ...values) {
        const allArgs = [formatString, ...values];
        return new json_path_1.JsonPathToken(`States.Format(${allArgs.map(json_path_1.renderInExpression).join(', ')})`).toString();
    }
    /**
     * Make an intrinsic States.StringToJson expression.
     *
     * During the execution of the Step Functions state machine, parse the given
     * argument as JSON into its object form.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.stringToJson(sfn.JsonPath.stringAt('$.someJsonBody'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static stringToJson(jsonString) {
        return new json_path_1.JsonPathToken(`States.StringToJson(${json_path_1.renderInExpression(jsonString)})`);
    }
    /**
     * Make an intrinsic States.JsonToString expression.
     *
     * During the execution of the Step Functions state machine, encode the
     * given object into a JSON string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.jsonToString(sfn.JsonPath.objectAt('$.someObject'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static jsonToString(value) {
        const path = json_path_1.jsonPathFromAny(value);
        if (!path) {
            throw new Error('Argument to JsonPath.jsonToString() must be a JsonPath object');
        }
        return new json_path_1.JsonPathToken(`States.JsonToString(${path})`).toString();
    }
}
exports.JsonPath = JsonPath;
_a = JSII_RTTI_SYMBOL_1;
JsonPath[_a] = { fqn: "@aws-cdk/aws-stepfunctions.JsonPath", version: "1.144.0" };
/**
 * Special string value to discard state input, output or result.
 *
 * @stability stable
 */
JsonPath.DISCARD = 'DISCARD';
/**
 * (deprecated) Extract a field from the State Machine data that gets passed around between states.
 *
 * @deprecated replaced by `JsonPath`
 */
class Data {
    constructor() { }
    /**
     * (deprecated) Instead of using a literal string, get the value from a JSON path.
     *
     * @deprecated
     */
    static stringAt(path) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#stringAt", "replaced by `JsonPath`");
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * (deprecated) Instead of using a literal string list, get the value from a JSON path.
     *
     * @deprecated
     */
    static listAt(path) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#listAt", "replaced by `JsonPath`");
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * (deprecated) Instead of using a literal number, get the value from a JSON path.
     *
     * @deprecated
     */
    static numberAt(path) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#numberAt", "replaced by `JsonPath`");
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * (deprecated) Use the entire data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @deprecated
     */
    static get entirePayload() {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#entirePayload", "replaced by `JsonPath`");
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * (deprecated) Determines if the indicated string is an encoded JSON path.
     *
     * @param value string to be evaluated.
     * @deprecated
     */
    static isJsonPathString(value) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Data#isJsonPathString", "replaced by `JsonPath`");
        return !!json_path_1.jsonPathString(value);
    }
}
exports.Data = Data;
_b = JSII_RTTI_SYMBOL_1;
Data[_b] = { fqn: "@aws-cdk/aws-stepfunctions.Data", version: "1.144.0" };
/**
 * (deprecated) Extract a field from the State Machine Context data.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 * @deprecated replaced by `JsonPath`
 */
class Context {
    constructor() { }
    /**
     * (deprecated) Instead of using a literal string, get the value from a JSON path.
     *
     * @deprecated
     */
    static stringAt(path) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#stringAt", "replaced by `JsonPath`");
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * (deprecated) Instead of using a literal number, get the value from a JSON path.
     *
     * @deprecated
     */
    static numberAt(path) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#numberAt", "replaced by `JsonPath`");
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * (deprecated) Return the Task Token field.
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     *
     * @deprecated
     */
    static get taskToken() {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#taskToken", "replaced by `JsonPath`");
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * (deprecated) Use the entire context data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @deprecated
     */
    static get entireContext() {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-stepfunctions.Context#entireContext", "replaced by `JsonPath`");
        return new json_path_1.JsonPathToken('$$').toString();
    }
}
exports.Context = Context;
_c = JSII_RTTI_SYMBOL_1;
Context[_c] = { fqn: "@aws-cdk/aws-stepfunctions.Context", version: "1.144.0" };
/**
 * Helper functions to work with structures containing fields.
 *
 * @stability stable
 */
class FieldUtils {
    constructor() { }
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure.
     *
     * @stability stable
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure.
     *
     * @stability stable
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere.
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     *
     * @stability stable
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
}
exports.FieldUtils = FieldUtils;
_d = JSII_RTTI_SYMBOL_1;
FieldUtils[_d] = { fqn: "@aws-cdk/aws-stepfunctions.FieldUtils", version: "1.144.0" };
function validateJsonPath(path) {
    if (path !== '$'
        && !path.startsWith('$.')
        && path !== '$$'
        && !path.startsWith('$$.')
        && !path.startsWith('$[')
        && ['Format', 'StringToJson', 'JsonToString', 'Array'].every(fn => !path.startsWith(`States.${fn}`))) {
        throw new Error(`JSON path values must be exactly '$', '$$', start with '$.', start with '$$.', start with '$[', or start with an intrinsic function: States.Format, States.StringToJson, States.JsonToString, or States.Array. Received: ${path}`);
    }
}
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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