import { IResolvable } from '@aws-cdk/core';
/**
 * Extract a field from the State Machine data or context that gets passed around between states.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 * @stability stable
 */
export declare class JsonPath {
    /**
     * Special string value to discard state input, output or result.
     *
     * @stability stable
     */
    static readonly DISCARD = "DISCARD";
    /**
     * Instead of using a literal string, get the value from a JSON path.
     *
     * @stability stable
     */
    static stringAt(path: string): string;
    /**
     * Instead of using a literal string list, get the value from a JSON path.
     *
     * @stability stable
     */
    static listAt(path: string): string[];
    /**
     * Instead of using a literal number, get the value from a JSON path.
     *
     * @stability stable
     */
    static numberAt(path: string): number;
    /**
     * Reference a complete (complex) object in a JSON path location.
     *
     * @stability stable
     */
    static objectAt(path: string): IResolvable;
    /**
     * Use the entire data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @stability stable
     */
    static get entirePayload(): string;
    /**
     * Determines if the indicated string is an encoded JSON path.
     *
     * @param value string to be evaluated.
     * @stability stable
     */
    static isEncodedJsonPath(value: string): boolean;
    /**
     * Return the Task Token field.
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     *
     * @stability stable
     */
    static get taskToken(): string;
    /**
     * Use the entire context data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @stability stable
     */
    static get entireContext(): string;
    /**
     * Make an intrinsic States.Array expression.
     *
     * Combine any number of string literals or JsonPath expressions into an array.
     *
     * Use this function if the value of an array element directly has to come
     * from a JSON Path expression (either the State object or the Context object).
     *
     * If the array contains object literals whose values come from a JSON path
     * expression, you do not need to use this function.
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static array(...values: string[]): string;
    /**
     * Make an intrinsic States.Format expression.
     *
     * This can be used to embed JSON Path variables inside a format string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.format('Hello, my name is {}.', sfn.JsonPath.stringAt('$.name'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static format(formatString: string, ...values: string[]): string;
    /**
     * Make an intrinsic States.StringToJson expression.
     *
     * During the execution of the Step Functions state machine, parse the given
     * argument as JSON into its object form.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.stringToJson(sfn.JsonPath.stringAt('$.someJsonBody'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static stringToJson(jsonString: string): IResolvable;
    /**
     * Make an intrinsic States.JsonToString expression.
     *
     * During the execution of the Step Functions state machine, encode the
     * given object into a JSON string.
     *
     * For example:
     *
     * ```ts
     * sfn.JsonPath.jsonToString(sfn.JsonPath.objectAt('$.someObject'))
     * ```
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html
     * @stability stable
     */
    static jsonToString(value: any): string;
    private constructor();
}
/**
 * (deprecated) Extract a field from the State Machine data that gets passed around between states.
 *
 * @deprecated replaced by `JsonPath`
 */
export declare class Data {
    /**
     * (deprecated) Instead of using a literal string, get the value from a JSON path.
     *
     * @deprecated
     */
    static stringAt(path: string): string;
    /**
     * (deprecated) Instead of using a literal string list, get the value from a JSON path.
     *
     * @deprecated
     */
    static listAt(path: string): string[];
    /**
     * (deprecated) Instead of using a literal number, get the value from a JSON path.
     *
     * @deprecated
     */
    static numberAt(path: string): number;
    /**
     * (deprecated) Use the entire data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @deprecated
     */
    static get entirePayload(): string;
    /**
     * (deprecated) Determines if the indicated string is an encoded JSON path.
     *
     * @param value string to be evaluated.
     * @deprecated
     */
    static isJsonPathString(value: string): boolean;
    private constructor();
}
/**
 * (deprecated) Extract a field from the State Machine Context data.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 * @deprecated replaced by `JsonPath`
 */
export declare class Context {
    /**
     * (deprecated) Instead of using a literal string, get the value from a JSON path.
     *
     * @deprecated
     */
    static stringAt(path: string): string;
    /**
     * (deprecated) Instead of using a literal number, get the value from a JSON path.
     *
     * @deprecated
     */
    static numberAt(path: string): number;
    /**
     * (deprecated) Return the Task Token field.
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     *
     * @deprecated
     */
    static get taskToken(): string;
    /**
     * (deprecated) Use the entire context data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @deprecated
     */
    static get entireContext(): string;
    private constructor();
}
/**
 * Helper functions to work with structures containing fields.
 *
 * @stability stable
 */
export declare class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure.
     *
     * @stability stable
     */
    static renderObject(obj?: {
        [key: string]: any;
    }): {
        [key: string]: any;
    } | undefined;
    /**
     * Return all JSON paths used in the given structure.
     *
     * @stability stable
     */
    static findReferencedPaths(obj?: {
        [key: string]: any;
    }): string[];
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere.
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     *
     * @stability stable
     */
    static containsTaskToken(obj?: {
        [key: string]: any;
    }): boolean;
    private constructor();
}
