"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkloadScheduling = exports.Workload = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const pod = require("./pod");
const utils_1 = require("./utils");
/**
 * A workload is an application running on Kubernetes. Whether your workload is a single
 * component or several that work together, on Kubernetes you run it inside a set of pods.
 * In Kubernetes, a Pod represents a set of running containers on your cluster.
 */
class Workload extends pod.AbstractPod {
    constructor(scope, id, props) {
        super(scope, id, props);
        this._matchLabels = {};
        this._matchExpressions = [];
        this._props = props;
        this.scheduling = new WorkloadScheduling(this);
        this.connections = new pod.PodConnections(this);
        this.spread = props.spread ?? false;
        this._matcher = cdk8s_1.Names.toLabelValue(this);
        if (props.select ?? true) {
            this.select(pod.LabelSelector.of({ labels: { [pod.Pod.ADDRESS_LABEL]: this._matcher } }));
        }
    }
    /**
     * The metadata of pods in this workload.
     */
    get podMetadata() {
        if (!this._podMetadata) {
            this._podMetadata = new cdk8s_1.ApiObjectMetadataDefinition({
                ...this._props.podMetadata,
                apiObject: this.apiObject,
            });
            this._podMetadata.addLabel(pod.Pod.ADDRESS_LABEL, this._matcher);
        }
        return this._podMetadata;
    }
    /**
     * Configure selectors for this workload.
     */
    select(...selectors) {
        for (const selector of selectors) {
            const kube = selector._toKube();
            this._matchExpressions.push(...kube.matchExpressions ?? []);
            for (const [key, value] of Object.entries(kube.matchLabels ?? {})) {
                this._matchLabels[key] = value;
            }
        }
    }
    /**
     * The label matchers this workload will use in order to select pods.
     *
     * Returns a a copy. Use `select()` to add label matchers.
     */
    get matchLabels() {
        return { ...this._matchLabels };
    }
    /**
     * The expression matchers this workload will use in order to select pods.
     *
     * Returns a a copy. Use `select()` to add expression matchers.
     */
    get matchExpressions() {
        return [...this._matchExpressions];
    }
    /**
     * @internal
     */
    _toLabelSelector() {
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this._matchExpressions),
            matchLabels: utils_1.undefinedIfEmpty(this._matchLabels),
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.spread) {
            {
                this.scheduling.spread({
                    topology: pod.Topology.HOSTNAME,
                });
                this.scheduling.spread({
                    topology: pod.Topology.ZONE,
                });
            }
        }
        ;
        const scheduling = this.scheduling._toKube();
        return {
            ...super._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Workload = Workload;
_a = JSII_RTTI_SYMBOL_1;
Workload[_a] = { fqn: "cdk8s-plus-28.Workload", version: "2.5.1" };
/**
 * Controls the pod scheduling strategy of this workload.
 * It offers some additional API's on top of the core pod scheduling.
 */
class WorkloadScheduling extends pod.PodScheduling {
    /**
     * Spread the pods in this workload by the topology key.
     * A spread is a separation of the pod from itself and is used to
     * balance out pod replicas across a given topology.
     */
    spread(options = {}) {
        this.separate(this.instance, { weight: options.weight, topology: options.topology ?? pod.Topology.HOSTNAME });
    }
}
exports.WorkloadScheduling = WorkloadScheduling;
_b = JSII_RTTI_SYMBOL_1;
WorkloadScheduling[_b] = { fqn: "cdk8s-plus-28.WorkloadScheduling", version: "2.5.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid29ya2xvYWQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvd29ya2xvYWQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBOEU7QUFHOUUsNkJBQTZCO0FBQzdCLG1DQUEyQztBQXFEM0M7Ozs7R0FJRztBQUNILE1BQXNCLFFBQVMsU0FBUSxHQUFHLENBQUMsV0FBVztJQWdCcEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQVRULGlCQUFZLEdBQTJCLEVBQUUsQ0FBQztRQUMxQyxzQkFBaUIsR0FBK0IsRUFBRSxDQUFDO1FBVWxFLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDO1FBQ3BCLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDO1FBRXBDLElBQUksQ0FBQyxRQUFRLEdBQUcsYUFBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV6QyxJQUFJLEtBQUssQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFO1lBQ3hCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO1NBQzNGO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ3RCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxtQ0FBMkIsQ0FBQztnQkFDbEQsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVc7Z0JBQzFCLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUzthQUMxQixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLGFBQWEsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDbEU7UUFDRCxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUM7SUFDM0IsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsU0FBOEI7UUFDN0MsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEdBQUcsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQ2hDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLElBQUksRUFBRSxDQUFDLENBQUM7WUFDNUQsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsSUFBSSxFQUFFLENBQUMsRUFBRTtnQkFDakUsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7YUFDaEM7U0FDRjtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLE9BQU8sRUFBRSxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILElBQVcsZ0JBQWdCO1FBQ3pCLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7T0FFRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPO1lBQ0wsZ0JBQWdCLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDO1lBQzFELFdBQVcsRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO1NBQ2pELENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2Y7Z0JBQ0UsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUM7b0JBQ3JCLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVE7aUJBQ2hDLENBQUMsQ0FBQztnQkFDSCxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQztvQkFDckIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSTtpQkFDNUIsQ0FBQyxDQUFDO2FBQ0o7U0FDRjtRQUFBLENBQUM7UUFFRixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRTdDLE9BQU87WUFDTCxHQUFHLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDckIsUUFBUSxFQUFFLFVBQVUsQ0FBQyxRQUFRO1lBQzdCLFFBQVEsRUFBRSxVQUFVLENBQUMsUUFBUTtZQUM3QixXQUFXLEVBQUUsVUFBVSxDQUFDLFdBQVc7U0FDcEMsQ0FBQztJQUNKLENBQUM7O0FBN0dILDRCQThHQzs7O0FBdUJEOzs7R0FHRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsR0FBRyxDQUFDLGFBQWE7SUFFdkQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxVQUEyQyxFQUFFO1FBQ3pELElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUSxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztJQUNoSCxDQUFDOztBQVRILGdEQVdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXBpT2JqZWN0TWV0YWRhdGEsIEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbiwgTmFtZXMgfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCAqIGFzIHBvZCBmcm9tICcuL3BvZCc7XG5pbXBvcnQgeyB1bmRlZmluZWRJZkVtcHR5IH0gZnJvbSAnLi91dGlscyc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYFdvcmtsb2FkYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXb3JrbG9hZFByb3BzIGV4dGVuZHMgcG9kLkFic3RyYWN0UG9kUHJvcHMge1xuXG4gIC8qKlxuICAgKiBUaGUgcG9kIG1ldGFkYXRhIG9mIHRoaXMgd29ya2xvYWQuXG4gICAqL1xuICByZWFkb25seSBwb2RNZXRhZGF0YT86IEFwaU9iamVjdE1ldGFkYXRhO1xuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGFsbG9jYXRlcyBhIHBvZCBsYWJlbCBzZWxlY3RvciBmb3IgdGhpcyB3b3JrbG9hZCBhbmQgYWRkXG4gICAqIGl0IHRvIHRoZSBwb2QgbWV0YWRhdGEuIFRoaXMgZW5zdXJlcyB0aGlzIHdvcmtsb2FkIG1hbmFnZXMgcG9kcyBjcmVhdGVkIGJ5XG4gICAqIGl0cyBwb2QgdGVtcGxhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHNlbGVjdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgc3ByZWFkIHBvZHMgYWNyb3NzIGhvc3RuYW1lIGFuZCB6b25lcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zY2hlZHVsaW5nLWV2aWN0aW9uL3RvcG9sb2d5LXNwcmVhZC1jb25zdHJhaW50cy8jaW50ZXJuYWwtZGVmYXVsdC1jb25zdHJhaW50c1xuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgc3ByZWFkPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIGxhYmVsIHNlbGVjdG9yIHJlcXVpcmVtZW50IGlzIGEgc2VsZWN0b3IgdGhhdCBjb250YWlucyB2YWx1ZXMsIGEga2V5LCBhbmQgYW4gb3BlcmF0b3IgdGhhdFxuICogcmVsYXRlcyB0aGUga2V5IGFuZCB2YWx1ZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFiZWxTZWxlY3RvclJlcXVpcmVtZW50IHtcbiAgLyoqXG4gICAqIFRoZSBsYWJlbCBrZXkgdGhhdCB0aGUgc2VsZWN0b3IgYXBwbGllcyB0by5cbiAgICovXG4gIHJlYWRvbmx5IGtleTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXByZXNlbnRzIGEga2V5J3MgcmVsYXRpb25zaGlwIHRvIGEgc2V0IG9mIHZhbHVlcy5cbiAgICovXG4gIHJlYWRvbmx5IG9wZXJhdG9yOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIGFycmF5IG9mIHN0cmluZyB2YWx1ZXMuIElmIHRoZSBvcGVyYXRvciBpcyBJbiBvciBOb3RJbiwgdGhlIHZhbHVlcyBhcnJheVxuICAgKiBtdXN0IGJlIG5vbi1lbXB0eS4gSWYgdGhlIG9wZXJhdG9yIGlzIEV4aXN0cyBvciBEb2VzTm90RXhpc3QsXG4gICAqIHRoZSB2YWx1ZXMgYXJyYXkgbXVzdCBiZSBlbXB0eS4gVGhpcyBhcnJheSBpcyByZXBsYWNlZCBkdXJpbmcgYSBzdHJhdGVnaWMgbWVyZ2UgcGF0Y2guXG4gICAqL1xuICByZWFkb25seSB2YWx1ZXM/OiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBBIHdvcmtsb2FkIGlzIGFuIGFwcGxpY2F0aW9uIHJ1bm5pbmcgb24gS3ViZXJuZXRlcy4gV2hldGhlciB5b3VyIHdvcmtsb2FkIGlzIGEgc2luZ2xlXG4gKiBjb21wb25lbnQgb3Igc2V2ZXJhbCB0aGF0IHdvcmsgdG9nZXRoZXIsIG9uIEt1YmVybmV0ZXMgeW91IHJ1biBpdCBpbnNpZGUgYSBzZXQgb2YgcG9kcy5cbiAqIEluIEt1YmVybmV0ZXMsIGEgUG9kIHJlcHJlc2VudHMgYSBzZXQgb2YgcnVubmluZyBjb250YWluZXJzIG9uIHlvdXIgY2x1c3Rlci5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFdvcmtsb2FkIGV4dGVuZHMgcG9kLkFic3RyYWN0UG9kIHtcblxuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IHBvZC5Qb2RDb25uZWN0aW9ucztcblxuICBwdWJsaWMgcmVhZG9ubHkgc2NoZWR1bGluZzogV29ya2xvYWRTY2hlZHVsaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgc3ByZWFkOiBib29sZWFuO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX21hdGNoTGFiZWxzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgX21hdGNoRXhwcmVzc2lvbnM6IExhYmVsU2VsZWN0b3JSZXF1aXJlbWVudFtdID0gW107XG5cbiAgcHJpdmF0ZSBfcG9kTWV0YWRhdGE/OiBBcGlPYmplY3RNZXRhZGF0YURlZmluaXRpb247XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfcHJvcHM6IFdvcmtsb2FkUHJvcHM7XG4gIHByaXZhdGUgcmVhZG9ubHkgX21hdGNoZXI6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogV29ya2xvYWRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy5fcHJvcHMgPSBwcm9wcztcbiAgICB0aGlzLnNjaGVkdWxpbmcgPSBuZXcgV29ya2xvYWRTY2hlZHVsaW5nKHRoaXMpO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgcG9kLlBvZENvbm5lY3Rpb25zKHRoaXMpO1xuICAgIHRoaXMuc3ByZWFkID0gcHJvcHMuc3ByZWFkID8/IGZhbHNlO1xuXG4gICAgdGhpcy5fbWF0Y2hlciA9IE5hbWVzLnRvTGFiZWxWYWx1ZSh0aGlzKTtcblxuICAgIGlmIChwcm9wcy5zZWxlY3QgPz8gdHJ1ZSkge1xuICAgICAgdGhpcy5zZWxlY3QocG9kLkxhYmVsU2VsZWN0b3Iub2YoeyBsYWJlbHM6IHsgW3BvZC5Qb2QuQUREUkVTU19MQUJFTF06IHRoaXMuX21hdGNoZXIgfSB9KSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBtZXRhZGF0YSBvZiBwb2RzIGluIHRoaXMgd29ya2xvYWQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHBvZE1ldGFkYXRhKCk6IEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbiB7XG4gICAgaWYgKCF0aGlzLl9wb2RNZXRhZGF0YSkge1xuICAgICAgdGhpcy5fcG9kTWV0YWRhdGEgPSBuZXcgQXBpT2JqZWN0TWV0YWRhdGFEZWZpbml0aW9uKHtcbiAgICAgICAgLi4udGhpcy5fcHJvcHMucG9kTWV0YWRhdGEsXG4gICAgICAgIGFwaU9iamVjdDogdGhpcy5hcGlPYmplY3QsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuX3BvZE1ldGFkYXRhLmFkZExhYmVsKHBvZC5Qb2QuQUREUkVTU19MQUJFTCwgdGhpcy5fbWF0Y2hlcik7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLl9wb2RNZXRhZGF0YTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmUgc2VsZWN0b3JzIGZvciB0aGlzIHdvcmtsb2FkLlxuICAgKi9cbiAgcHVibGljIHNlbGVjdCguLi5zZWxlY3RvcnM6IHBvZC5MYWJlbFNlbGVjdG9yW10pIHtcbiAgICBmb3IgKGNvbnN0IHNlbGVjdG9yIG9mIHNlbGVjdG9ycykge1xuICAgICAgY29uc3Qga3ViZSA9IHNlbGVjdG9yLl90b0t1YmUoKTtcbiAgICAgIHRoaXMuX21hdGNoRXhwcmVzc2lvbnMucHVzaCguLi5rdWJlLm1hdGNoRXhwcmVzc2lvbnMgPz8gW10pO1xuICAgICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoa3ViZS5tYXRjaExhYmVscyA/PyB7fSkpIHtcbiAgICAgICAgdGhpcy5fbWF0Y2hMYWJlbHNba2V5XSA9IHZhbHVlO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbGFiZWwgbWF0Y2hlcnMgdGhpcyB3b3JrbG9hZCB3aWxsIHVzZSBpbiBvcmRlciB0byBzZWxlY3QgcG9kcy5cbiAgICpcbiAgICogUmV0dXJucyBhIGEgY29weS4gVXNlIGBzZWxlY3QoKWAgdG8gYWRkIGxhYmVsIG1hdGNoZXJzLlxuICAgKi9cbiAgcHVibGljIGdldCBtYXRjaExhYmVscygpOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+IHtcbiAgICByZXR1cm4geyAuLi50aGlzLl9tYXRjaExhYmVscyB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBleHByZXNzaW9uIG1hdGNoZXJzIHRoaXMgd29ya2xvYWQgd2lsbCB1c2UgaW4gb3JkZXIgdG8gc2VsZWN0IHBvZHMuXG4gICAqXG4gICAqIFJldHVybnMgYSBhIGNvcHkuIFVzZSBgc2VsZWN0KClgIHRvIGFkZCBleHByZXNzaW9uIG1hdGNoZXJzLlxuICAgKi9cbiAgcHVibGljIGdldCBtYXRjaEV4cHJlc3Npb25zKCk6IExhYmVsU2VsZWN0b3JSZXF1aXJlbWVudFtdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX21hdGNoRXhwcmVzc2lvbnNdO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0xhYmVsU2VsZWN0b3IoKTogazhzLkxhYmVsU2VsZWN0b3Ige1xuICAgIHJldHVybiB7XG4gICAgICBtYXRjaEV4cHJlc3Npb25zOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuX21hdGNoRXhwcmVzc2lvbnMpLFxuICAgICAgbWF0Y2hMYWJlbHM6IHVuZGVmaW5lZElmRW1wdHkodGhpcy5fbWF0Y2hMYWJlbHMpLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvUG9kU3BlYygpOiBrOHMuUG9kU3BlYyB7XG4gICAgaWYgKHRoaXMuc3ByZWFkKSB7XG4gICAgICB7XG4gICAgICAgIHRoaXMuc2NoZWR1bGluZy5zcHJlYWQoe1xuICAgICAgICAgIHRvcG9sb2d5OiBwb2QuVG9wb2xvZ3kuSE9TVE5BTUUsXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLnNjaGVkdWxpbmcuc3ByZWFkKHtcbiAgICAgICAgICB0b3BvbG9neTogcG9kLlRvcG9sb2d5LlpPTkUsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH07XG5cbiAgICBjb25zdCBzY2hlZHVsaW5nID0gdGhpcy5zY2hlZHVsaW5nLl90b0t1YmUoKTtcblxuICAgIHJldHVybiB7XG4gICAgICAuLi5zdXBlci5fdG9Qb2RTcGVjKCksXG4gICAgICBhZmZpbml0eTogc2NoZWR1bGluZy5hZmZpbml0eSxcbiAgICAgIG5vZGVOYW1lOiBzY2hlZHVsaW5nLm5vZGVOYW1lLFxuICAgICAgdG9sZXJhdGlvbnM6IHNjaGVkdWxpbmcudG9sZXJhdGlvbnMsXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBXb3JrbG9hZFNjaGVkdWxpbmcuc3ByZWFkYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXb3JrbG9hZFNjaGVkdWxpbmdTcHJlYWRPcHRpb25zIHtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHRoZSBzcHJlYWQgaXMgb3B0aW9uYWwsIHdpdGggdGhpcyB3ZWlnaHQgc2NvcmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gd2VpZ2h0LiBzcHJlYWQgaXMgYXNzdW1lZCB0byBiZSByZXF1aXJlZC5cbiAgICovXG4gIHJlYWRvbmx5IHdlaWdodD86IG51bWJlcjtcblxuICAvKipcbiAgICogV2hpY2ggdG9wb2xvZ3kgdG8gc3ByZWFkIG9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRvcG9sb2d5LkhPU1ROQU1FXG4gICAqL1xuICByZWFkb25seSB0b3BvbG9neT86IHBvZC5Ub3BvbG9neTtcblxufVxuXG4vKipcbiAqIENvbnRyb2xzIHRoZSBwb2Qgc2NoZWR1bGluZyBzdHJhdGVneSBvZiB0aGlzIHdvcmtsb2FkLlxuICogSXQgb2ZmZXJzIHNvbWUgYWRkaXRpb25hbCBBUEkncyBvbiB0b3Agb2YgdGhlIGNvcmUgcG9kIHNjaGVkdWxpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBXb3JrbG9hZFNjaGVkdWxpbmcgZXh0ZW5kcyBwb2QuUG9kU2NoZWR1bGluZyB7XG5cbiAgLyoqXG4gICAqIFNwcmVhZCB0aGUgcG9kcyBpbiB0aGlzIHdvcmtsb2FkIGJ5IHRoZSB0b3BvbG9neSBrZXkuXG4gICAqIEEgc3ByZWFkIGlzIGEgc2VwYXJhdGlvbiBvZiB0aGUgcG9kIGZyb20gaXRzZWxmIGFuZCBpcyB1c2VkIHRvXG4gICAqIGJhbGFuY2Ugb3V0IHBvZCByZXBsaWNhcyBhY3Jvc3MgYSBnaXZlbiB0b3BvbG9neS5cbiAgICovXG4gIHB1YmxpYyBzcHJlYWQob3B0aW9uczogV29ya2xvYWRTY2hlZHVsaW5nU3ByZWFkT3B0aW9ucyA9IHt9KSB7XG4gICAgdGhpcy5zZXBhcmF0ZSh0aGlzLmluc3RhbmNlLCB7IHdlaWdodDogb3B0aW9ucy53ZWlnaHQsIHRvcG9sb2d5OiBvcHRpb25zLnRvcG9sb2d5ID8/IHBvZC5Ub3BvbG9neS5IT1NUTkFNRSB9KTtcbiAgfVxuXG59Il19