"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterRole = exports.Role = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const rb = require("./role-binding");
const utils_1 = require("./utils");
class ImportedRole extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'roles';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeRole.GVK.apiVersion;
    }
    get apiGroup() {
        return 'rbac.authorization.k8s.io';
    }
    get kind() {
        return k8s.KubeRole.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * Role is a namespaced, logical grouping of PolicyRules that can be referenced
 * as a unit by a RoleBinding.
 */
class Role extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'roles';
        this._rules = [];
        this.apiObject = new k8s.KubeRole(this, 'Resource', {
            metadata: props.metadata,
            rules: cdk8s_1.Lazy.any({ produce: () => this.synthesizeRules() }),
        });
        for (const rule of props.rules ?? []) {
            this.allow(rule.verbs, ...rule.resources);
        }
    }
    /**
     * Imports a role from the cluster as a reference.
     */
    static fromRoleName(scope, id, name) {
        return new ImportedRole(scope, id, name);
    }
    /**
     * Rules associaated with this Role.
     * Returns a copy, use `allow` to add rules.
     */
    get rules() {
        return [...this._rules];
    }
    /**
     * Add permission to perform a list of HTTP verbs on a collection of
     * resources.
     *
     * @param resources The resource(s) to apply to
     * @see https://kubernetes.io/docs/reference/access-authn-authz/authorization/#determine-the-request-verb
     */
    allow(verbs, ...resources) {
        this._rules.push({ verbs, resources });
    }
    /**
     * Add "create" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowCreate(...resources) {
        this.allow(['create'], ...resources);
    }
    /**
     * Add "get" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowGet(...resources) {
        this.allow(['get'], ...resources);
    }
    /**
     * Add "list" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowList(...resources) {
        this.allow(['list'], ...resources);
    }
    /**
     * Add "watch" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowWatch(...resources) {
        this.allow(['watch'], ...resources);
    }
    /**
     * Add "update" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowUpdate(...resources) {
        this.allow(['update'], ...resources);
    }
    /**
     * Add "patch" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowPatch(...resources) {
        this.allow(['patch'], ...resources);
    }
    /**
     * Add "delete" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowDelete(...resources) {
        this.allow(['delete'], ...resources);
    }
    /**
     * Add "deletecollection" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowDeleteCollection(...resources) {
        this.allow(['deletecollection'], ...resources);
    }
    /**
     * Add "get", "list", and "watch" permissions for the resources.
     * @param resources The resource(s) to apply to
     */
    allowRead(...resources) {
        this.allow(['get', 'list', 'watch'], ...resources);
    }
    /**
     * Add "get", "list", "watch", "create", "update", "patch", "delete", and
     * "deletecollection" permissions for the resources.
     *
     * @param resources The resource(s) to apply to
     */
    allowReadWrite(...resources) {
        this.allow(['get', 'list', 'watch', 'create', 'update', 'patch', 'delete', 'deletecollection'], ...resources);
    }
    /**
     * Create a RoleBinding that binds the permissions in this Role
     * to a list of subjects, that will only apply this role's namespace.
     * @param subjects a list of subjects to bind to
     */
    bind(...subjects) {
        const subjectsAddress = utils_1.address(...subjects);
        const binding = new rb.RoleBinding(this, `RoleBinding${subjectsAddress}`, {
            metadata: {
                namespace: this.metadata.namespace,
            },
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    synthesizeRules() {
        const rules = [];
        for (const rule of this._rules) {
            for (const resource of rule.resources) {
                rules.push({
                    verbs: rule.verbs,
                    apiGroups: [resource.apiGroup === 'core' ? '' : resource.apiGroup],
                    resourceNames: resource.resourceName ? [resource.resourceName] : undefined,
                    resources: resource.resourceType ? [resource.resourceType] : undefined,
                });
            }
        }
        return rules;
    }
}
exports.Role = Role;
_a = JSII_RTTI_SYMBOL_1;
Role[_a] = { fqn: "cdk8s-plus-28.Role", version: "2.5.1" };
class ImportedClusterRole extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'clusterroles';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubeClusterRole.GVK.apiVersion;
    }
    get apiGroup() {
        return 'rbac.authorization.k8s.io';
    }
    get kind() {
        return k8s.KubeClusterRole.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * ClusterRole is a cluster level, logical grouping of PolicyRules that can be
 * referenced as a unit by a RoleBinding or ClusterRoleBinding.
 */
class ClusterRole extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'clusterroles';
        this._labelSelector = {};
        this._rules = [];
        this.apiObject = new k8s.KubeClusterRole(this, 'Resource', {
            metadata: props.metadata,
            rules: cdk8s_1.Lazy.any({ produce: () => this.synthesizeRules() }),
            aggregationRule: cdk8s_1.Lazy.any({ produce: () => this.synthesizeAggregationRules() }),
        });
        for (const rule of props.rules ?? []) {
            this.allow(rule.verbs, ...rule.endpoints);
        }
        for (const [key, value] of Object.entries(props.aggregationLabels ?? {})) {
            this.aggregate(key, value);
        }
    }
    /**
     * Imports a role from the cluster as a reference.
     */
    static fromClusterRoleName(scope, id, name) {
        return new ImportedClusterRole(scope, id, name);
    }
    /**
     * Rules associaated with this Role.
     * Returns a copy, use `allow` to add rules.
     */
    get rules() {
        return [...this._rules];
    }
    /**
     * Add permission to perform a list of HTTP verbs on a collection of
     * resources.
     *
     * @param endpoints The endpoints(s) to apply to
     * @see https://kubernetes.io/docs/reference/access-authn-authz/authorization/#determine-the-request-verb
     */
    allow(verbs, ...endpoints) {
        this._rules.push({ verbs, endpoints });
    }
    /**
     * Add "create" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowCreate(...endpoints) {
        this.allow(['create'], ...endpoints);
    }
    /**
     * Add "get" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowGet(...endpoints) {
        this.allow(['get'], ...endpoints);
    }
    /**
     * Add "list" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowList(...endpoints) {
        this.allow(['list'], ...endpoints);
    }
    /**
     * Add "watch" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowWatch(...endpoints) {
        this.allow(['watch'], ...endpoints);
    }
    /**
     * Add "update" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowUpdate(...endpoints) {
        this.allow(['update'], ...endpoints);
    }
    /**
     * Add "patch" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowPatch(...endpoints) {
        this.allow(['patch'], ...endpoints);
    }
    /**
     * Add "delete" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowDelete(...endpoints) {
        this.allow(['delete'], ...endpoints);
    }
    /**
     * Add "deletecollection" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowDeleteCollection(...endpoints) {
        this.allow(['deletecollection'], ...endpoints);
    }
    /**
     * Add "get", "list", and "watch" permissions for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowRead(...endpoints) {
        this.allow(['get', 'list', 'watch'], ...endpoints);
    }
    /**
     * Add "get", "list", "watch", "create", "update", "patch", "delete", and
     * "deletecollection" permissions for the resources.
     *
     * @param endpoints The resource(s) to apply to
     */
    allowReadWrite(...endpoints) {
        this.allow(['get', 'list', 'watch', 'create', 'update', 'patch', 'delete', 'deletecollection'], ...endpoints);
    }
    /**
     * Aggregate rules from roles matching this label selector.
     */
    aggregate(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Combines the rules of the argument ClusterRole into this ClusterRole
     * using aggregation labels.
     * @param rol
     */
    combine(rol) {
        const key = `cdk8s.cluster-role/aggregate-to-${cdk8s_1.Names.toLabelValue(this)}`;
        const value = 'true';
        rol.metadata.addLabel(key, value);
        this.aggregate(key, value);
    }
    /**
     * Create a RoleBinding that binds the permissions in this ClusterRole
     * to a list of subjects, that will only apply to the given namespace.
     * @param namespace the namespace to limit permissions to.
     * @param subjects a list of subjects to bind to
     */
    bindInNamespace(namespace, ...subjects) {
        const binding = new rb.RoleBinding(this, `RoleBinding-${namespace}`, {
            metadata: {
                namespace,
            },
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    /**
     * Create a ClusterRoleBinding that binds the permissions in this
     * ClusterRole to a list of subjects, without namespace restrictions.
     * @param subjects a list of subjects to bind to
     */
    bind(...subjects) {
        const binding = new rb.ClusterRoleBinding(this, 'ClusterRoleBinding', {
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    synthesizeRules() {
        const rules = [];
        for (const rule of this._rules) {
            for (const endpoint of rule.endpoints) {
                const resource = endpoint.asApiResource();
                const nonResource = endpoint.asNonApiResource();
                if (resource && nonResource) {
                    throw new Error('Endpoint must be either resource or non resource. not both.');
                }
                if (!resource && !nonResource) {
                    throw new Error('Endpoint must be either resource or non resource. not neither.');
                }
                if (resource) {
                    rules.push({
                        apiGroups: [resource.apiGroup === 'core' ? '' : resource.apiGroup],
                        resources: [resource.resourceType],
                        resourceNames: resource.resourceName ? [resource.resourceName] : [],
                        verbs: rule.verbs,
                    });
                }
                if (nonResource) {
                    rules.push({ verbs: rule.verbs, nonResourceUrLs: [nonResource] });
                }
            }
        }
        return rules;
    }
    synthesizeAggregationRules() {
        if (Object.keys(this._labelSelector).length === 0) {
            return undefined;
        }
        return { clusterRoleSelectors: [{ matchLabels: this._labelSelector }] };
    }
}
exports.ClusterRole = ClusterRole;
_b = JSII_RTTI_SYMBOL_1;
ClusterRole[_b] = { fqn: "cdk8s-plus-28.ClusterRole", version: "2.5.1" };
//# sourceMappingURL=data:application/json;base64,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