"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PersistentVolumeMode = exports.PersistentVolumeAccessMode = exports.PersistentVolumeClaim = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
class ImportedPersistentVolumeClaim extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'persistentvolumeclaims';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubePersistentVolumeClaim.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubePersistentVolumeClaim.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * A PersistentVolumeClaim (PVC) is a request for storage by a user.
 * It is similar to a Pod. Pods consume node resources and PVCs consume PV resources.
 * Pods can request specific levels of resources (CPU and Memory).
 * Claims can request specific size and access modes
 */
class PersistentVolumeClaim extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'persistentvolumeclaims';
        this.storage = props.storage;
        this.volumeMode = props.volumeMode ?? PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        if (props.volume) {
            this.bind(props.volume);
        }
        this.apiObject = new k8s.KubePersistentVolumeClaim(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pvc from the cluster as a reference.
     */
    static fromClaimName(scope, id, claimName) {
        return new ImportedPersistentVolumeClaim(scope, id, claimName);
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PV this claim is bound to. Undefined means the claim is not bound
     * to any specific volume.
     */
    get volume() {
        return this._volume;
    }
    /**
     * Bind a claim to a specific volume.
     * Note that you must also bind the volume to the claim.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param vol The PV to bind to.
     */
    bind(vol) {
        if (this._volume && this._volume.name !== vol.name) {
            throw new Error(`Cannot bind claim '${this.name}' to volume '${vol.name}' since it is already bound to volume '${this._volume.name}'`);
        }
        this._volume = vol;
    }
    /**
     * @internal
     */
    _toKube() {
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            volumeName: this.volume ? this.volume.name : undefined,
            accessModes: this.accessModes?.map(a => a.toString()),
            resources: storage ? { requests: { storage } } : undefined,
            volumeMode: this.volumeMode,
            storageClassName: this.storageClassName,
        };
    }
}
exports.PersistentVolumeClaim = PersistentVolumeClaim;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolumeClaim[_a] = { fqn: "cdk8s-plus-28.PersistentVolumeClaim", version: "2.5.1" };
/**
 * Access Modes.
 */
var PersistentVolumeAccessMode;
(function (PersistentVolumeAccessMode) {
    /**
     * The volume can be mounted as read-write by a single node.
     * ReadWriteOnce access mode still can allow multiple pods to access
     * the volume when the pods are running on the same node.
     */
    PersistentVolumeAccessMode["READ_WRITE_ONCE"] = "ReadWriteOnce";
    /**
     * The volume can be mounted as read-only by many nodes.
     */
    PersistentVolumeAccessMode["READ_ONLY_MANY"] = "ReadOnlyMany";
    /**
     * The volume can be mounted as read-write by many nodes.
     */
    PersistentVolumeAccessMode["READ_WRITE_MANY"] = "ReadWriteMany";
    /**
     * The volume can be mounted as read-write by a single Pod.
     * Use ReadWriteOncePod access mode if you want to ensure that
     * only one pod across whole cluster can read that PVC or write to it.
     * This is only supported for CSI volumes and Kubernetes version 1.22+.
     */
    PersistentVolumeAccessMode["READ_WRITE_ONCE_POD"] = "ReadWriteOncePod";
})(PersistentVolumeAccessMode = exports.PersistentVolumeAccessMode || (exports.PersistentVolumeAccessMode = {}));
/**
 * Volume Modes.
 */
var PersistentVolumeMode;
(function (PersistentVolumeMode) {
    /**
     * Volume is ounted into Pods into a directory.
     * If the volume is backed by a block device and the device is empty,
     * Kubernetes creates a filesystem on the device before mounting it
     * for the first time.
     */
    PersistentVolumeMode["FILE_SYSTEM"] = "Filesystem";
    /**
     * Use a volume as a raw block device. Such volume is presented into a Pod as a block device,
     * without any filesystem on it. This mode is useful to provide a Pod the fastest possible way
     * to access a volume, without any filesystem layer between the Pod
     * and the volume. On the other hand, the application running in
     * the Pod must know how to handle a raw block device
     */
    PersistentVolumeMode["BLOCK"] = "Block";
})(PersistentVolumeMode = exports.PersistentVolumeMode || (exports.PersistentVolumeMode = {}));
//# sourceMappingURL=data:application/json;base64,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