"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy ?? RestartPolicy.ALWAYS;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? false;
        this.isolate = props.isolate ?? false;
        this.hostNetwork = props.hostNetwork ?? false;
        this.terminationGracePeriod = props.terminationGracePeriod ?? cdk8s_1.Duration.seconds(30);
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this.attachContainer(impl);
        return impl;
    }
    attachContainer(cont) {
        this._containers.push(cont);
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (!this.isSidecarContainer(cont) && cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (!this.isSidecarContainer(cont) && cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (!this.isSidecarContainer(cont) && cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    // Any initContainer that has `restartPolicy=Always` is a sidecar container. Please refer to
    // documentation for more details:
    // https://kubernetes.io/docs/concepts/workloads/pods/sidecar-containers/#differences-from-init-containers
    isSidecarContainer(cont) {
        return cont.restartPolicy === container.ContainerRestartPolicy.ALWAYS;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // check if restartPolicy is defined for containers
            if (cont.restartPolicy) {
                throw new Error(`Invalid container spec: ${cont.name} has non-empty restartPolicy field. The field can only be specified for initContainers`);
            }
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
            hostNetwork: this.hostNetwork,
            terminationGracePeriodSeconds: this.terminationGracePeriod?.toSeconds(),
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-28.AbstractPod", version: "2.5.1" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-28.LabelSelector", version: "2.5.1" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
        if (this.isolate) {
            this.connections.isolate();
        }
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-28.Pod", version: "2.5.1" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-28.PodDns", version: "2.5.1" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-28.PodSecurityContext", version: "2.5.1" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            key: this.key,
            operator: this.operator,
            values: this.values,
        };
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-28.NodeLabelQuery", version: "2.5.1" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-28.LabelExpression", version: "2.5.1" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            effect: this.effect,
            key: this.key,
            operator: this.operator,
            tolerationSeconds: this.evictAfter?.toSeconds(),
            value: this.value,
        };
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-28.NodeTaintQuery", version: "2.5.1" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-28.Pods", version: "2.5.1" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-28.LabeledNode", version: "2.5.1" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-28.TaintedNode", version: "2.5.1" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-28.NamedNode", version: "2.5.1" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-28.Node", version: "2.5.1" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-28.Topology", version: "2.5.1" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push(query._toKube());
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => s._toKube()) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-28.PodScheduling", version: "2.5.1" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n.number));
    }
    /**
     * Sets the default network policy for Pod/Workload to have all egress and ingress connections as disabled
     */
    isolate() {
        new networkpolicy.NetworkPolicy(this.instance, 'DefaultDenyAll', {
            selector: this.instance,
            // the policy must be defined in the namespace of the pod
            // so it can select it.
            metadata: { namespace: this.instance.metadata.namespace },
            egress: {
                default: networkpolicy.NetworkPolicyTrafficDefault.DENY,
            },
            ingress: {
                default: networkpolicy.NetworkPolicyTrafficDefault.DENY,
            },
        });
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-28.PodConnections", version: "2.5.1" };
//# sourceMappingURL=data:application/json;base64,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