"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ContainerRestartPolicy = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = exports.SeccompProfileType = exports.Capability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Capability - complete list of POSIX capabilities
 */
var Capability;
(function (Capability) {
    /**
     * ALL
     */
    Capability["ALL"] = "ALL";
    /**
     * CAP_AUDIT_CONTROL
     */
    Capability["AUDIT_CONTROL"] = "AUDIT_CONTROL";
    /**
     * CAP_AUDIT_READ
     */
    Capability["AUDIT_READ"] = "AUDIT_READ";
    /**
     * CAP_AUDIT_WRITE
     */
    Capability["AUDIT_WRITE"] = "AUDIT_WRITE";
    /**
     * CAP_BLOCK_SUSPEND
     */
    Capability["BLOCK_SUSPEND"] = "BLOCK_SUSPEND";
    /**
     * CAP_BPF
     */
    Capability["BPF"] = "BPF";
    /**
     * CAP_CHECKPOINT_RESTORE
     */
    Capability["CHECKPOINT_RESTORE"] = "CHECKPOINT_RESTORE";
    /**
     * CAP_CHOWN
     */
    Capability["CHOWN"] = "CHOWN";
    /**
     * CAP_DAC_OVERRIDE
     */
    Capability["DAC_OVERRIDE"] = "DAC_OVERRIDE";
    /**
     * CAP_DAC_READ_SEARCH
     */
    Capability["DAC_READ_SEARCH"] = "DAC_READ_SEARCH";
    /**
     * CAP_FOWNER
     */
    Capability["FOWNER"] = "FOWNER";
    /**
     * CAP_FSETID
     */
    Capability["FSETID"] = "FSETID";
    /**
     * CAP_IPC_LOCK
     */
    Capability["IPC_LOCK"] = "IPC_LOCK";
    /**
     * CAP_IPC_OWNER
     */
    Capability["IPC_OWNER"] = "IPC_OWNER";
    /**
     * CAP_KILL
     */
    Capability["KILL"] = "KILL";
    /**
     * CAP_LEASE
     */
    Capability["LEASE"] = "LEASE";
    /**
     * CAP_LINUX_IMMUTABLE
     */
    Capability["LINUX_IMMUTABLE"] = "LINUX_IMMUTABLE";
    /**
     * CAP_MAC_ADMIN
     */
    Capability["MAC_ADMIN"] = "MAC_ADMIN";
    /**
     * CAP_MAC_OVERRIDE
     */
    Capability["MAC_OVERRIDE"] = "MAC_OVERRIDE";
    /**
     * CAP_MKNOD
     */
    Capability["MKNOD"] = "MKNOD";
    /**
     * CAP_NET_ADMIN
     */
    Capability["NET_ADMIN"] = "NET_ADMIN";
    /**
     * CAP_NET_BIND_SERVICE
     */
    Capability["NET_BIND_SERVICE"] = "NET_BIND_SERVICE";
    /**
     * CAP_NET_BROADCAST
     */
    Capability["NET_BROADCAST"] = "NET_BROADCAST";
    /**
     * CAP_NET_RAW
     */
    Capability["NET_RAW"] = "NET_RAW";
    /**
     * CAP_PERFMON
     */
    Capability["PERFMON"] = "PERFMON";
    /**
     * CAP_SETGID
     */
    Capability["SETGID"] = "SETGID";
    /**
     * CAP_SETFCAP
     */
    Capability["SETFCAP"] = "SETFCAP";
    /**
     * CAP_SETPCAP
     */
    Capability["SETPCAP"] = "SETPCAP";
    /**
     * CAP_SETUID
     */
    Capability["SETUID"] = "SETUID";
    /**
     * CAP_SYS_ADMIN
     */
    Capability["SYS_ADMIN"] = "SYS_ADMIN";
    /**
     * CAP_SYS_BOOT
     */
    Capability["SYS_BOOT"] = "SYS_BOOT";
    /**
     * CAP_SYS_CHROOT
     */
    Capability["SYS_CHROOT"] = "SYS_CHROOT";
    /**
     * CAP_SYS_MODULE
     */
    Capability["SYS_MODULE"] = "SYS_MODULE";
    /**
     * CAP_SYS_NICE
     */
    Capability["SYS_NICE"] = "SYS_NICE";
    /**
     * CAP_SYS_PACCT
     */
    Capability["SYS_PACCT"] = "SYS_PACCT";
    /**
     * CAP_SYS_PTRACE
     */
    Capability["SYS_PTRACE"] = "SYS_PTRACE";
    /**
     * CAP_SYS_RAWIO
     */
    Capability["SYS_RAWIO"] = "SYS_RAWIO";
    /**
     * CAP_SYS_RESOURCE
     */
    Capability["SYS_RESOURCE"] = "SYS_RESOURCE";
    /**
     * CAP_SYS_TIME
     */
    Capability["SYS_TIME"] = "SYS_TIME";
    /**
     * CAP_SYS_TTY_CONFIG
     */
    Capability["SYS_TTY_CONFIG"] = "SYS_TTY_CONFIG";
    /**
     * CAP_SYSLOG
     */
    Capability["SYSLOG"] = "SYSLOG";
    /**
     * CAP_WAKE_ALARM
     */
    Capability["WAKE_ALARM"] = "WAKE_ALARM";
})(Capability = exports.Capability || (exports.Capability = {}));
var SeccompProfileType;
(function (SeccompProfileType) {
    /**
    * A profile defined in a file on the node should be used
    */
    SeccompProfileType["LOCALHOST"] = "Localhost";
    /**
    * The container runtime default profile should be used
    */
    SeccompProfileType["RUNTIME_DEFAULT"] = "RuntimeDefault";
    /**
    * No profile should be applied
    */
    SeccompProfileType["UNCONFINED"] = "Unconfined";
})(SeccompProfileType = exports.SeccompProfileType || (exports.SeccompProfileType = {}));
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        if (props.seccompProfile?.localhostProfile && props.seccompProfile.type != SeccompProfileType.LOCALHOST) {
            throw new Error('localhostProfile must only be set if type is "Localhost"');
        }
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? true;
        this.user = props.user;
        this.group = props.group;
        this.allowPrivilegeEscalation = props.allowPrivilegeEscalation ?? false;
        this.capabilities = props.capabilities;
        this.seccompProfile = props.seccompProfile;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
            allowPrivilegeEscalation: this.allowPrivilegeEscalation,
            capabilities: this.capabilities,
            seccompProfile: this.seccompProfile,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-28.ContainerSecurityContext", version: "2.5.1" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-28.EnvValue", version: "2.5.1" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * RestartPolicy defines the restart behavior of individual containers in a pod.
 * This field may only be set for init containers, and the only allowed value is "Always".
 * For non-init containers or when this field is not specified,
 * the restart behavior is defined by the Pod's restart policy and the container type.
 * Setting the RestartPolicy as "Always" for the init container will have the following effect:
 * this init container will be continually restarted on exit until all regular containers have terminated.
 * Once all regular containers have completed, all init containers with restartPolicy "Always" will be shut down.
 * This lifecycle differs from normal init containers and is often referred to as a "sidecar" container.
 *
 * @see https://kubernetes.io/docs/concepts/workloads/pods/sidecar-containers/
 */
var ContainerRestartPolicy;
(function (ContainerRestartPolicy) {
    /**
     * If an init container is created with its restartPolicy set to Always,
     * it will start and remain running during the entire life of the Pod.
     * For regular containers, this is ignored by Kubernetes.
     */
    ContainerRestartPolicy["ALWAYS"] = "Always";
})(ContainerRestartPolicy = exports.ContainerRestartPolicy || (exports.ContainerRestartPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        this.restartPolicy = props.restartPolicy;
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numberProtocols = this._ports.map(p => `${p.number}/${p.protocol || Protocol.TCP}`);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        const protocol = `${port.number}/${port.protocol || Protocol.TCP}`;
        if (numberProtocols.includes(protocol)) {
            throw new Error(`Port with number ${port.number} and protocol ${port.protocol || Protocol.TCP} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol?.toString(),
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            restartPolicy: this.restartPolicy,
            securityContext: this.securityContext._toKube(),
        };
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-28.Container", version: "2.5.1" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-28.Cpu", version: "2.5.1" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-28.EnvFrom", version: "2.5.1" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-28.Env", version: "2.5.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29udGFpbmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2NvbnRhaW5lci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLGlDQUE2QjtBQUc3QixxQ0FBcUM7QUFDckMsaUNBQWlDO0FBRWpDLG1DQUEyQztBQUczQzs7R0FFRztBQUNILElBQVksVUF5S1g7QUF6S0QsV0FBWSxVQUFVO0lBQ3BCOztPQUVHO0lBQ0gseUJBQVcsQ0FBQTtJQUNYOztPQUVHO0lBQ0gsNkNBQStCLENBQUE7SUFDL0I7O09BRUc7SUFDSCx1Q0FBeUIsQ0FBQTtJQUN6Qjs7T0FFRztJQUNILHlDQUEyQixDQUFBO0lBQzNCOztPQUVHO0lBQ0gsNkNBQStCLENBQUE7SUFDL0I7O09BRUc7SUFDSCx5QkFBVyxDQUFBO0lBQ1g7O09BRUc7SUFDSCx1REFBeUMsQ0FBQTtJQUN6Qzs7T0FFRztJQUNILDZCQUFlLENBQUE7SUFDZjs7T0FFRztJQUNILDJDQUE2QixDQUFBO0lBQzdCOztPQUVHO0lBQ0gsaURBQW1DLENBQUE7SUFDbkM7O09BRUc7SUFDSCwrQkFBaUIsQ0FBQTtJQUNqQjs7T0FFRztJQUNILCtCQUFpQixDQUFBO0lBQ2pCOztPQUVHO0lBQ0gsbUNBQXFCLENBQUE7SUFDckI7O09BRUc7SUFDSCxxQ0FBdUIsQ0FBQTtJQUN2Qjs7T0FFRztJQUNILDJCQUFhLENBQUE7SUFDYjs7T0FFRztJQUNILDZCQUFlLENBQUE7SUFDZjs7T0FFRztJQUNILGlEQUFtQyxDQUFBO0lBQ25DOztPQUVHO0lBQ0gscUNBQXVCLENBQUE7SUFDdkI7O09BRUc7SUFDSCwyQ0FBNkIsQ0FBQTtJQUM3Qjs7T0FFRztJQUNILDZCQUFlLENBQUE7SUFDZjs7T0FFRztJQUNILHFDQUF1QixDQUFBO0lBQ3ZCOztPQUVHO0lBQ0gsbURBQXFDLENBQUE7SUFDckM7O09BRUc7SUFDSCw2Q0FBK0IsQ0FBQTtJQUMvQjs7T0FFRztJQUNILGlDQUFtQixDQUFBO0lBQ25COztPQUVHO0lBQ0gsaUNBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCwrQkFBaUIsQ0FBQTtJQUNqQjs7T0FFRztJQUNILGlDQUFtQixDQUFBO0lBQ25COztPQUVHO0lBQ0gsaUNBQW1CLENBQUE7SUFDbkI7O09BRUc7SUFDSCwrQkFBaUIsQ0FBQTtJQUNqQjs7T0FFRztJQUNILHFDQUF1QixDQUFBO0lBQ3ZCOztPQUVHO0lBQ0gsbUNBQXFCLENBQUE7SUFDckI7O09BRUc7SUFDSCx1Q0FBeUIsQ0FBQTtJQUN6Qjs7T0FFRztJQUNILHVDQUF5QixDQUFBO0lBQ3pCOztPQUVHO0lBQ0gsbUNBQXFCLENBQUE7SUFDckI7O09BRUc7SUFDSCxxQ0FBdUIsQ0FBQTtJQUN2Qjs7T0FFRztJQUNILHVDQUF5QixDQUFBO0lBQ3pCOztPQUVHO0lBQ0gscUNBQXVCLENBQUE7SUFDdkI7O09BRUc7SUFDSCwyQ0FBNkIsQ0FBQTtJQUM3Qjs7T0FFRztJQUNILG1DQUFxQixDQUFBO0lBQ3JCOztPQUVHO0lBQ0gsK0NBQWlDLENBQUE7SUFDakM7O09BRUc7SUFDSCwrQkFBaUIsQ0FBQTtJQUNqQjs7T0FFRztJQUNILHVDQUF5QixDQUFBO0FBQzNCLENBQUMsRUF6S1csVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUF5S3JCO0FBRUQsSUFBWSxrQkFhWDtBQWJELFdBQVksa0JBQWtCO0lBQzVCOztNQUVFO0lBQ0YsNkNBQXVCLENBQUE7SUFDdkI7O01BRUU7SUFDRix3REFBa0MsQ0FBQTtJQUNsQzs7TUFFRTtJQUNGLCtDQUF5QixDQUFBO0FBQzNCLENBQUMsRUFiVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQWE3QjtBQTZJRDs7R0FFRztBQUNILElBQVksUUFnQlg7QUFoQkQsV0FBWSxRQUFRO0lBRWxCOztPQUVHO0lBQ0gsdUJBQVcsQ0FBQTtJQUVYOztPQUVHO0lBQ0gsdUJBQVcsQ0FBQTtJQUVYOztPQUVHO0lBQ0gseUJBQWEsQ0FBQTtBQUNmLENBQUMsRUFoQlcsUUFBUSxHQUFSLGdCQUFRLEtBQVIsZ0JBQVEsUUFnQm5CO0FBR0Q7O0dBRUc7QUFDSCxNQUFhLHdCQUF3QjtJQVduQyxZQUFZLFFBQXVDLEVBQUU7UUFDbkQsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFLGdCQUFnQixJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBSSxJQUFJLGtCQUFrQixDQUFDLFNBQVMsRUFBRTtZQUN2RyxNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxDQUFDLENBQUM7U0FDN0U7UUFDRCxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDO1FBQ2pELElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUM7UUFDNUMsSUFBSSxDQUFDLHNCQUFzQixHQUFHLEtBQUssQ0FBQyxzQkFBc0IsSUFBSSxJQUFJLENBQUM7UUFDbkUsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztRQUN6QixJQUFJLENBQUMsd0JBQXdCLEdBQUcsS0FBSyxDQUFDLHdCQUF3QixJQUFJLEtBQUssQ0FBQztRQUN4RSxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDdkMsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPO1lBQ0wsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLO1lBQ3RCLFNBQVMsRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNwQixZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWE7WUFDaEMsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO1lBQzNCLHNCQUFzQixFQUFFLElBQUksQ0FBQyxzQkFBc0I7WUFDbkQsd0JBQXdCLEVBQUUsSUFBSSxDQUFDLHdCQUF3QjtZQUN2RCxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO1NBQ3BDLENBQUM7SUFDSixDQUFDOztBQXZDSCw0REF5Q0M7OztBQUVELElBQVksYUFrRFg7QUFsREQsV0FBWSxhQUFhO0lBQ3ZCOztPQUVHO0lBQ0gsMkNBQTBCLENBQUE7SUFFMUI7O09BRUc7SUFDSCxxREFBb0MsQ0FBQTtJQUVwQzs7T0FFRztJQUNILHlDQUF3QixDQUFBO0lBRXhCOztPQUVHO0lBQ0gsOENBQTZCLENBQUE7SUFFN0I7O09BRUc7SUFDSCx3REFBdUMsQ0FBQTtJQUV2Qzs7T0FFRztJQUNILHdDQUF1QixDQUFBO0lBRXZCOztPQUVHO0lBQ0gsaUVBQWdELENBQUE7SUFFaEQ7O09BRUc7SUFDSCw0Q0FBMkIsQ0FBQTtJQUUzQjs7T0FFRztJQUNILDBDQUF5QixDQUFBO0lBRXpCOztPQUVHO0lBQ0gsMENBQXlCLENBQUE7QUFDM0IsQ0FBQyxFQWxEVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQWtEeEI7QUFFRCxJQUFZLGtCQThCWDtBQTlCRCxXQUFZLGtCQUFrQjtJQUM1Qjs7T0FFRztJQUNILDhDQUF3QixDQUFBO0lBRXhCOztPQUVHO0lBQ0gsb0RBQThCLENBQUE7SUFFOUI7O09BRUc7SUFDSCxrREFBNEIsQ0FBQTtJQUU1Qjs7T0FFRztJQUNILHdEQUFrQyxDQUFBO0lBRWxDOztPQUVHO0lBQ0gsZ0VBQTBDLENBQUE7SUFFMUM7O09BRUc7SUFDSCxvRUFBOEMsQ0FBQTtBQUNoRCxDQUFDLEVBOUJXLGtCQUFrQixHQUFsQiwwQkFBa0IsS0FBbEIsMEJBQWtCLFFBOEI3QjtBQXlFRDs7R0FFRztBQUNILE1BQWEsUUFBUTtJQStHbkIsWUFBb0MsS0FBVyxFQUFrQixTQUFlO1FBQTVDLFVBQUssR0FBTCxLQUFLLENBQU07UUFBa0IsY0FBUyxHQUFULFNBQVMsQ0FBTTtJQUFJLENBQUM7SUE3R3JGOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsU0FBK0IsRUFBRSxHQUFXLEVBQUUsVUFBd0MsRUFBRTtRQUVsSCxNQUFNLE1BQU0sR0FBcUI7WUFDL0IsZUFBZSxFQUFFO2dCQUNmLElBQUksRUFBRSxTQUFTLENBQUMsSUFBSTtnQkFDcEIsR0FBRztnQkFDSCxRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVE7YUFDM0I7U0FDRixDQUFDO1FBQ0YsT0FBTyxJQUFJLFFBQVEsQ0FBQyxTQUFTLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGVBQWUsQ0FBQyxXQUErQixFQUFFLFVBQXFDLEVBQUU7UUFDcEcsTUFBTSxNQUFNLEdBQXFCO1lBQy9CLFlBQVksRUFBRTtnQkFDWixJQUFJLEVBQUUsV0FBVyxDQUFDLE1BQU0sQ0FBQyxJQUFJO2dCQUM3QixHQUFHLEVBQUUsV0FBVyxDQUFDLEdBQUc7Z0JBQ3BCLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUTthQUMzQjtTQUNGLENBQUM7UUFFRixPQUFPLElBQUksUUFBUSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsS0FBYTtRQUNuQyxPQUFPLElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLFNBQXdCLEVBQUUsVUFBdUMsRUFBRTtRQUM1RixJQUFJLFFBQVEsR0FBWSxLQUFLLENBQUM7UUFFOUIsSUFBSSxTQUFTLEtBQUssYUFBYSxDQUFDLFNBQVMsSUFBSSxTQUFTLEtBQUssYUFBYSxDQUFDLGNBQWMsRUFBRTtZQUN2RixRQUFRLEdBQUcsSUFBSSxDQUFDO1NBQ2pCO1FBRUQsSUFBSSxRQUFRLElBQUksT0FBTyxDQUFDLEdBQUcsS0FBSyxTQUFTLEVBQUU7WUFDekMsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLFNBQVMsaUJBQWlCLENBQUMsQ0FBQztTQUNoRDtRQUVELE1BQU0sTUFBTSxHQUFxQjtZQUMvQixRQUFRLEVBQUU7Z0JBQ1IsU0FBUyxFQUFFLENBQUMsR0FBRyxTQUFTLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsS0FBSyxPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO2dCQUNwRSxHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsRUFBRSxVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDbEU7U0FDRixDQUFDO1FBRUYsT0FBTyxJQUFJLFFBQVEsQ0FBQyxTQUFTLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUE0QixFQUFFLFVBQXVDLEVBQUU7UUFDaEcsTUFBTSxNQUFNLEdBQXFCO1lBQy9CLGdCQUFnQixFQUFFO2dCQUNoQixRQUFRLEVBQUUsR0FBRyxRQUFRLEVBQUU7Z0JBQ3ZCLEdBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUNyRixHQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxhQUFhLEVBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2FBQ3pFO1NBQ0YsQ0FBQztRQUVGLE9BQU8sSUFBSSxRQUFRLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsR0FBVyxFQUFFLFVBQXNDLEVBQUU7UUFFN0UsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUUvQixJQUFJLE9BQU8sQ0FBQyxRQUFRLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxXQUFXLEdBQUcsZUFBZSxDQUFDLENBQUM7U0FDaEQ7UUFFRCxPQUFPLFFBQVEsQ0FBQyxTQUFTLENBQUMsS0FBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQzs7QUE3R0gsNEJBZ0hDOzs7QUFFRCxJQUFZLGVBd0JYO0FBeEJELFdBQVksZUFBZTtJQUN6Qjs7Ozs7Ozs7T0FRRztJQUNILG9DQUFpQixDQUFBO0lBRWpCOzs7OztPQUtHO0lBQ0gsa0RBQStCLENBQUE7SUFFL0I7O09BRUc7SUFDSCxrQ0FBZSxDQUFBO0FBQ2pCLENBQUMsRUF4QlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUF3QjFCO0FBZ0NEOzs7Ozs7Ozs7OztHQVdHO0FBQ0gsSUFBWSxzQkFRWDtBQVJELFdBQVksc0JBQXNCO0lBRWhDOzs7O09BSUc7SUFDSCwyQ0FBaUIsQ0FBQTtBQUNuQixDQUFDLEVBUlcsc0JBQXNCLEdBQXRCLDhCQUFzQixLQUF0Qiw4QkFBc0IsUUFRakM7QUFtTEQ7O0dBRUc7QUFDSCxNQUFhLFNBQVM7SUErRHBCLFlBQVksS0FBcUI7UUFOaEIsV0FBTSxHQUFvQixFQUFFLENBQUM7UUFPNUMsSUFBSSxLQUFLLFlBQVksU0FBUyxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztTQUNoRjtRQUVELE1BQU0sbUJBQW1CLEdBQXVCO1lBQzlDLEdBQUcsRUFBRTtnQkFDSCxPQUFPLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUM7Z0JBQ3pCLEtBQUssRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQzthQUN4QjtZQUNELE1BQU0sRUFBRTtnQkFDTixPQUFPLEVBQUUsWUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUM7Z0JBQzVCLEtBQUssRUFBRSxZQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQzthQUM1QjtTQUNGLENBQUM7UUFFRixJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNsQyxNQUFNLElBQUksS0FBSyxDQUFDLCtGQUErRixDQUFDLENBQUM7U0FDbEg7UUFFRCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDbEQsTUFBTSx5QkFBeUIsR0FBZ0IsS0FBSyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUUvRixJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksTUFBTSxDQUFDO1FBQ2pDLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztRQUN6QixJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztRQUM3QixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDOUIsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQ3hCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDaEMsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzNGLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNsQyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksbUJBQW1CLENBQUM7UUFDeEQsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ25DLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLFlBQVksSUFBSSxFQUFFLENBQUM7UUFDdkMsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxJQUFJLGVBQWUsQ0FBQyxNQUFNLENBQUM7UUFDdkUsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLHdCQUF3QixDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUMzRSxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxFQUFFLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBRXpDLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixJQUFJLENBQUMsT0FBTyxDQUFDO2dCQUNYLE1BQU0sRUFBRSxJQUFJLENBQUMsVUFBVTthQUN4QixDQUFDLENBQUM7U0FDSjtRQUVELEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxDQUFDLEtBQUssSUFBSSxFQUFFLEVBQUU7WUFDcEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNwQjtJQUNILENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsSUFBSTtRQUNiLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQztJQUN6QixDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyxLQUFLO1FBQ2QsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzFCLENBQUM7SUFFRDs7O09BR0c7SUFDSCxJQUFXLE9BQU87UUFDaEIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDeEQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLElBQUk7UUFDYixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksS0FBSyxDQUFDLElBQVksRUFBRSxPQUF3QixFQUFFLFVBQXdCLEVBQUU7UUFDN0UsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLE9BQU8sQ0FBQyxRQUFRLEVBQUUsRUFBRSxHQUFHLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDckUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTyxDQUFDLElBQW1CO1FBRWhDLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzFELE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFFMUYsSUFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsa0JBQWtCLElBQUksQ0FBQyxJQUFJLGlCQUFpQixDQUFDLENBQUM7U0FDL0Q7UUFFRCxNQUFNLFFBQVEsR0FBRyxHQUFHLElBQUksQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDbkUsSUFBSSxlQUFlLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ3RDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLElBQUksQ0FBQyxNQUFNLGlCQUFpQixJQUFJLENBQUMsUUFBUSxJQUFJLFFBQVEsQ0FBQyxHQUFHLGlCQUFpQixDQUFDLENBQUM7U0FDakg7UUFFRCxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN6QixDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osTUFBTSxZQUFZLEdBQXNCLEVBQUUsQ0FBQztRQUUzQyxLQUFLLE1BQU0sS0FBSyxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDL0IsWUFBWSxDQUFDLElBQUksQ0FBQztnQkFDaEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSTtnQkFDdkIsU0FBUyxFQUFFLEtBQUssQ0FBQyxJQUFJO2dCQUNyQixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7Z0JBQ3hCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxXQUFXO2dCQUNuQyxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ3RCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVzthQUMvQixDQUFDLENBQUM7U0FDSjtRQUVELE1BQU0sS0FBSyxHQUFHLElBQUksS0FBSyxFQUFxQixDQUFDO1FBRTdDLEtBQUssTUFBTSxJQUFJLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtZQUM3QixLQUFLLENBQUMsSUFBSSxDQUFDO2dCQUNULGFBQWEsRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDMUIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRLEVBQUUsUUFBUSxFQUFFO2dCQUNuQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2dCQUN2QixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07YUFDcEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCwrQkFBK0I7UUFDL0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQztRQUNwRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxPQUFPLEVBQUUsTUFBTSxDQUFDO1FBQ3hELE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxTQUFTLEVBQUUsTUFBTSxFQUFFLEtBQUssQ0FBQztRQUNsRCxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUM7UUFDdEQsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFLGdCQUFnQixFQUFFLEtBQUssQ0FBQztRQUN0RSxNQUFNLHVCQUF1QixHQUFHLElBQUksQ0FBQyxTQUFTLEVBQUUsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDO1FBRTFFLE1BQU0sTUFBTSxHQUFvQyxFQUFFLENBQUM7UUFDbkQsTUFBTSxRQUFRLEdBQW9DLEVBQUUsQ0FBQztRQUVyRCxJQUFJLFFBQVEsRUFBRTtZQUNaLE1BQU0sQ0FBQyxHQUFHLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDaEQ7UUFDRCxJQUFJLFdBQVcsRUFBRTtZQUNmLE1BQU0sQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRSxDQUFDLFFBQVEsRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDO1NBQ3RGO1FBQ0QsSUFBSSxxQkFBcUIsRUFBRTtZQUN6QixNQUFNLENBQUMsbUJBQW1CLENBQUMsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxRQUFRLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztTQUM5RztRQUNELElBQUksVUFBVSxFQUFFO1lBQ2QsUUFBUSxDQUFDLEdBQUcsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNwRDtRQUNELElBQUksYUFBYSxFQUFFO1lBQ2pCLFFBQVEsQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsRUFBRSxDQUFDLFFBQVEsRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDO1NBQzFGO1FBQ0QsSUFBSSx1QkFBdUIsRUFBRTtZQUMzQixRQUFRLENBQUMsbUJBQW1CLENBQUMsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyx1QkFBdUIsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxRQUFRLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztTQUNsSDtRQUVELElBQUksb0JBQW9CLEdBQXlDLFNBQVMsQ0FBQztRQUMzRSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDdEUsb0JBQW9CLEdBQUc7Z0JBQ3JCLE1BQU0sRUFBRSx3QkFBZ0IsQ0FBQyxNQUFNLENBQUM7Z0JBQ2hDLFFBQVEsRUFBRSx3QkFBZ0IsQ0FBQyxRQUFRLENBQUM7YUFDckMsQ0FBQztTQUNIO1FBRUQsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUUvQixPQUFPO1lBQ0wsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1lBQ2YsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLO1lBQ2pCLGVBQWUsRUFBRSxJQUFJLENBQUMsZUFBZTtZQUNyQyxLQUFLLEVBQUUsd0JBQWdCLENBQUMsS0FBSyxDQUFDO1lBQzlCLFlBQVksRUFBRSx3QkFBZ0IsQ0FBQyxZQUFZLENBQUM7WUFDNUMsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPO1lBQ3JCLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNmLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUMzQixHQUFHLEVBQUUsR0FBRyxDQUFDLFNBQVM7WUFDbEIsT0FBTyxFQUFFLEdBQUcsQ0FBQyxJQUFJO1lBQ2pCLGNBQWMsRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUM7WUFDOUMsYUFBYSxFQUFFLElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQztZQUM1QyxZQUFZLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDO1lBQzFDLFNBQVMsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztnQkFDM0IsU0FBUyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUM7Z0JBQ25ELE9BQU8sRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDO2FBQ2hELENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDYixTQUFTLEVBQUUsb0JBQW9CO1lBQy9CLGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYTtZQUNqQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxPQUFPLEVBQUU7U0FDaEQsQ0FBQztJQUNKLENBQUM7O0FBNVFILDhCQTZRQzs7O0FBNkRELElBQVksZ0JBZ0RYO0FBaERELFdBQVksZ0JBQWdCO0lBQzFCOzs7Ozs7Ozs7T0FTRztJQUNILGlDQUFhLENBQUE7SUFFYjs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0gseURBQXFDLENBQUE7SUFFckM7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0gsbURBQStCLENBQUE7QUFDakMsQ0FBQyxFQWhEVyxnQkFBZ0IsR0FBaEIsd0JBQWdCLEtBQWhCLHdCQUFnQixRQWdEM0I7QUFtQkQ7OztHQUdHO0FBQ0gsTUFBYSxHQUFHO0lBUWQsWUFBb0IsTUFBYztRQUNoQyxJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztJQUN2QixDQUFDO0lBVEQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFjO1FBQzFCLE9BQU8sSUFBSSxHQUFHLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFDRCxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQWM7UUFDekIsT0FBTyxJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztJQUNwQyxDQUFDOztBQU5ILGtCQVdDOzs7QUFrQkQ7O0dBRUc7QUFDSCxNQUFhLE9BQU87SUFFbEIsWUFDbUIsU0FBZ0MsRUFDaEMsTUFBZSxFQUNmLEdBQW9CO1FBRnBCLGNBQVMsR0FBVCxTQUFTLENBQXVCO1FBQ2hDLFdBQU0sR0FBTixNQUFNLENBQVM7UUFDZixRQUFHLEdBQUgsR0FBRyxDQUFpQjtJQUFJLENBQUM7SUFBQSxDQUFDO0lBRTdDOztPQUVHO0lBQ0ksT0FBTztRQUNaLE9BQU87WUFDTCxZQUFZLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7Z0JBQzdCLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUk7YUFDMUIsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNiLFNBQVMsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDcEIsSUFBSSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSTthQUNwQixDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2IsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO1NBQ3BCLENBQUM7SUFDSixDQUFDOztBQXBCSCwwQkFzQkM7OztBQUVELFNBQWdCLHFCQUFxQixDQUFDLFFBQWM7SUFFbEQsSUFBSSxDQUFDLFFBQVEsRUFBRTtRQUFFLE9BQU8sRUFBRSxDQUFDO0tBQUU7SUFFN0IsNERBQTREO0lBQzVELDBCQUEwQjtJQUMxQixNQUFNLFVBQVUsR0FBaUIsUUFBZ0IsQ0FBQyxVQUFVLElBQUksRUFBRSxDQUFDO0lBRW5FLE9BQU8sVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUMxQyxDQUFDO0FBVEQsc0RBU0M7QUFFRDs7R0FFRztBQUNILE1BQWEsR0FBRztJQXVCZCxZQUFtQixPQUFrQixFQUFFLFNBQXVDO1FBQzVFLElBQUksQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxVQUFVLEdBQUcsU0FBUyxDQUFDO0lBQzlCLENBQUM7SUF4QkQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsU0FBK0IsRUFBRSxNQUFlO1FBQzFFLE9BQU8sSUFBSSxPQUFPLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBb0I7UUFDM0MsT0FBTyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFVRDs7OztPQUlHO0lBQ0ksV0FBVyxDQUFDLElBQVksRUFBRSxLQUFlO1FBQzlDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7O09BR0c7SUFDSCxJQUFXLFNBQVM7UUFDbEIsT0FBTyxFQUFFLEdBQUcsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxRQUFRLENBQUMsSUFBYTtRQUMzQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMzQixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSCxJQUFXLE9BQU87UUFDaEIsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFTyxTQUFTLENBQUMsR0FBaUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQWMsQ0FBQztRQUN2QyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUMzQyxNQUFNLENBQUMsSUFBSSxDQUFDO2dCQUNWLElBQUk7Z0JBQ0osS0FBSyxFQUFFLENBQUMsQ0FBQyxLQUFLO2dCQUNkLFNBQVMsRUFBRSxDQUFDLENBQUMsU0FBUzthQUN2QixDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPO1lBQ0wsSUFBSSxFQUFFLHdCQUFnQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDM0QsU0FBUyxFQUFFLHdCQUFnQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdELENBQUM7SUFDSixDQUFDOztBQW5GSCxrQkFvRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBTaXplIH0gZnJvbSAnY2RrOHMnO1xuaW1wb3J0ICogYXMgY29uZmlnbWFwIGZyb20gJy4vY29uZmlnLW1hcCc7XG5pbXBvcnQgKiBhcyBoYW5kbGVyIGZyb20gJy4vaGFuZGxlcic7XG5pbXBvcnQgKiBhcyBrOHMgZnJvbSAnLi9pbXBvcnRzL2s4cyc7XG5pbXBvcnQgKiBhcyBwcm9iZSBmcm9tICcuL3Byb2JlJztcbmltcG9ydCAqIGFzIHNlY3JldCBmcm9tICcuL3NlY3JldCc7XG5pbXBvcnQgeyB1bmRlZmluZWRJZkVtcHR5IH0gZnJvbSAnLi91dGlscyc7XG5pbXBvcnQgKiBhcyB2b2x1bWUgZnJvbSAnLi92b2x1bWUnO1xuXG4vKipcbiAqIENhcGFiaWxpdHkgLSBjb21wbGV0ZSBsaXN0IG9mIFBPU0lYIGNhcGFiaWxpdGllc1xuICovXG5leHBvcnQgZW51bSBDYXBhYmlsaXR5IHtcbiAgLyoqXG4gICAqIEFMTFxuICAgKi9cbiAgQUxMID0gJ0FMTCcsXG4gIC8qKlxuICAgKiBDQVBfQVVESVRfQ09OVFJPTFxuICAgKi9cbiAgQVVESVRfQ09OVFJPTCA9ICdBVURJVF9DT05UUk9MJyxcbiAgLyoqXG4gICAqIENBUF9BVURJVF9SRUFEXG4gICAqL1xuICBBVURJVF9SRUFEID0gJ0FVRElUX1JFQUQnLFxuICAvKipcbiAgICogQ0FQX0FVRElUX1dSSVRFXG4gICAqL1xuICBBVURJVF9XUklURSA9ICdBVURJVF9XUklURScsXG4gIC8qKlxuICAgKiBDQVBfQkxPQ0tfU1VTUEVORFxuICAgKi9cbiAgQkxPQ0tfU1VTUEVORCA9ICdCTE9DS19TVVNQRU5EJyxcbiAgLyoqXG4gICAqIENBUF9CUEZcbiAgICovXG4gIEJQRiA9ICdCUEYnLFxuICAvKipcbiAgICogQ0FQX0NIRUNLUE9JTlRfUkVTVE9SRVxuICAgKi9cbiAgQ0hFQ0tQT0lOVF9SRVNUT1JFID0gJ0NIRUNLUE9JTlRfUkVTVE9SRScsXG4gIC8qKlxuICAgKiBDQVBfQ0hPV05cbiAgICovXG4gIENIT1dOID0gJ0NIT1dOJyxcbiAgLyoqXG4gICAqIENBUF9EQUNfT1ZFUlJJREVcbiAgICovXG4gIERBQ19PVkVSUklERSA9ICdEQUNfT1ZFUlJJREUnLFxuICAvKipcbiAgICogQ0FQX0RBQ19SRUFEX1NFQVJDSFxuICAgKi9cbiAgREFDX1JFQURfU0VBUkNIID0gJ0RBQ19SRUFEX1NFQVJDSCcsXG4gIC8qKlxuICAgKiBDQVBfRk9XTkVSXG4gICAqL1xuICBGT1dORVIgPSAnRk9XTkVSJyxcbiAgLyoqXG4gICAqIENBUF9GU0VUSURcbiAgICovXG4gIEZTRVRJRCA9ICdGU0VUSUQnLFxuICAvKipcbiAgICogQ0FQX0lQQ19MT0NLXG4gICAqL1xuICBJUENfTE9DSyA9ICdJUENfTE9DSycsXG4gIC8qKlxuICAgKiBDQVBfSVBDX09XTkVSXG4gICAqL1xuICBJUENfT1dORVIgPSAnSVBDX09XTkVSJyxcbiAgLyoqXG4gICAqIENBUF9LSUxMXG4gICAqL1xuICBLSUxMID0gJ0tJTEwnLFxuICAvKipcbiAgICogQ0FQX0xFQVNFXG4gICAqL1xuICBMRUFTRSA9ICdMRUFTRScsXG4gIC8qKlxuICAgKiBDQVBfTElOVVhfSU1NVVRBQkxFXG4gICAqL1xuICBMSU5VWF9JTU1VVEFCTEUgPSAnTElOVVhfSU1NVVRBQkxFJyxcbiAgLyoqXG4gICAqIENBUF9NQUNfQURNSU5cbiAgICovXG4gIE1BQ19BRE1JTiA9ICdNQUNfQURNSU4nLFxuICAvKipcbiAgICogQ0FQX01BQ19PVkVSUklERVxuICAgKi9cbiAgTUFDX09WRVJSSURFID0gJ01BQ19PVkVSUklERScsXG4gIC8qKlxuICAgKiBDQVBfTUtOT0RcbiAgICovXG4gIE1LTk9EID0gJ01LTk9EJyxcbiAgLyoqXG4gICAqIENBUF9ORVRfQURNSU5cbiAgICovXG4gIE5FVF9BRE1JTiA9ICdORVRfQURNSU4nLFxuICAvKipcbiAgICogQ0FQX05FVF9CSU5EX1NFUlZJQ0VcbiAgICovXG4gIE5FVF9CSU5EX1NFUlZJQ0UgPSAnTkVUX0JJTkRfU0VSVklDRScsXG4gIC8qKlxuICAgKiBDQVBfTkVUX0JST0FEQ0FTVFxuICAgKi9cbiAgTkVUX0JST0FEQ0FTVCA9ICdORVRfQlJPQURDQVNUJyxcbiAgLyoqXG4gICAqIENBUF9ORVRfUkFXXG4gICAqL1xuICBORVRfUkFXID0gJ05FVF9SQVcnLFxuICAvKipcbiAgICogQ0FQX1BFUkZNT05cbiAgICovXG4gIFBFUkZNT04gPSAnUEVSRk1PTicsXG4gIC8qKlxuICAgKiBDQVBfU0VUR0lEXG4gICAqL1xuICBTRVRHSUQgPSAnU0VUR0lEJyxcbiAgLyoqXG4gICAqIENBUF9TRVRGQ0FQXG4gICAqL1xuICBTRVRGQ0FQID0gJ1NFVEZDQVAnLFxuICAvKipcbiAgICogQ0FQX1NFVFBDQVBcbiAgICovXG4gIFNFVFBDQVAgPSAnU0VUUENBUCcsXG4gIC8qKlxuICAgKiBDQVBfU0VUVUlEXG4gICAqL1xuICBTRVRVSUQgPSAnU0VUVUlEJyxcbiAgLyoqXG4gICAqIENBUF9TWVNfQURNSU5cbiAgICovXG4gIFNZU19BRE1JTiA9ICdTWVNfQURNSU4nLFxuICAvKipcbiAgICogQ0FQX1NZU19CT09UXG4gICAqL1xuICBTWVNfQk9PVCA9ICdTWVNfQk9PVCcsXG4gIC8qKlxuICAgKiBDQVBfU1lTX0NIUk9PVFxuICAgKi9cbiAgU1lTX0NIUk9PVCA9ICdTWVNfQ0hST09UJyxcbiAgLyoqXG4gICAqIENBUF9TWVNfTU9EVUxFXG4gICAqL1xuICBTWVNfTU9EVUxFID0gJ1NZU19NT0RVTEUnLFxuICAvKipcbiAgICogQ0FQX1NZU19OSUNFXG4gICAqL1xuICBTWVNfTklDRSA9ICdTWVNfTklDRScsXG4gIC8qKlxuICAgKiBDQVBfU1lTX1BBQ0NUXG4gICAqL1xuICBTWVNfUEFDQ1QgPSAnU1lTX1BBQ0NUJyxcbiAgLyoqXG4gICAqIENBUF9TWVNfUFRSQUNFXG4gICAqL1xuICBTWVNfUFRSQUNFID0gJ1NZU19QVFJBQ0UnLFxuICAvKipcbiAgICogQ0FQX1NZU19SQVdJT1xuICAgKi9cbiAgU1lTX1JBV0lPID0gJ1NZU19SQVdJTycsXG4gIC8qKlxuICAgKiBDQVBfU1lTX1JFU09VUkNFXG4gICAqL1xuICBTWVNfUkVTT1VSQ0UgPSAnU1lTX1JFU09VUkNFJyxcbiAgLyoqXG4gICAqIENBUF9TWVNfVElNRVxuICAgKi9cbiAgU1lTX1RJTUUgPSAnU1lTX1RJTUUnLFxuICAvKipcbiAgICogQ0FQX1NZU19UVFlfQ09ORklHXG4gICAqL1xuICBTWVNfVFRZX0NPTkZJRyA9ICdTWVNfVFRZX0NPTkZJRycsXG4gIC8qKlxuICAgKiBDQVBfU1lTTE9HXG4gICAqL1xuICBTWVNMT0cgPSAnU1lTTE9HJyxcbiAgLyoqXG4gICAqIENBUF9XQUtFX0FMQVJNXG4gICAqL1xuICBXQUtFX0FMQVJNID0gJ1dBS0VfQUxBUk0nLFxufVxuXG5leHBvcnQgZW51bSBTZWNjb21wUHJvZmlsZVR5cGUge1xuICAvKipcbiAgKiBBIHByb2ZpbGUgZGVmaW5lZCBpbiBhIGZpbGUgb24gdGhlIG5vZGUgc2hvdWxkIGJlIHVzZWRcbiAgKi9cbiAgTE9DQUxIT1NUID0gJ0xvY2FsaG9zdCcsXG4gIC8qKlxuICAqIFRoZSBjb250YWluZXIgcnVudGltZSBkZWZhdWx0IHByb2ZpbGUgc2hvdWxkIGJlIHVzZWRcbiAgKi9cbiAgUlVOVElNRV9ERUZBVUxUID0gJ1J1bnRpbWVEZWZhdWx0JyxcbiAgLyoqXG4gICogTm8gcHJvZmlsZSBzaG91bGQgYmUgYXBwbGllZFxuICAqL1xuICBVTkNPTkZJTkVEID0gJ1VuY29uZmluZWQnLFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFNlY2NvbXBQcm9maWxlIHtcbiAgLyoqXG4gICAqIGxvY2FsaG9zdFByb2ZpbGUgaW5kaWNhdGVzIGEgcHJvZmlsZSBkZWZpbmVkIGluIGEgZmlsZSBvbiB0aGUgbm9kZSBzaG91bGQgYmUgdXNlZC5cbiAgICogVGhlIHByb2ZpbGUgbXVzdCBiZSBwcmVjb25maWd1cmVkIG9uIHRoZSBub2RlIHRvIHdvcmsuIE11c3QgYmUgYSBkZXNjZW5kaW5nIHBhdGgsXG4gICAqIHJlbGF0aXZlIHRvIHRoZSBrdWJlbGV0J3MgY29uZmlndXJlZCBzZWNjb21wIHByb2ZpbGUgbG9jYXRpb24uXG4gICAqIE11c3Qgb25seSBiZSBzZXQgaWYgdHlwZSBpcyBcIkxvY2FsaG9zdFwiLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGVtcHR5IHN0cmluZ1xuICAgKi9cbiAgcmVhZG9ubHkgbG9jYWxob3N0UHJvZmlsZT86IHN0cmluZztcbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGljaCBraW5kIG9mIHNlY2NvbXAgcHJvZmlsZSB3aWxsIGJlIGFwcGxpZWRcbiAgICovXG4gIHJlYWRvbmx5IHR5cGU6IFNlY2NvbXBQcm9maWxlVHlwZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDb250YWluZXJTZWN1dGlyeUNvbnRleHRDYXBhYmlsaXRpZXMge1xuICAvKipcbiAgICogQWRkZWQgY2FwYWJpbGl0aWVzXG4gICAqL1xuICByZWFkb25seSBhZGQ/OiBDYXBhYmlsaXR5W107XG5cbiAgLyoqXG4gICAqIFJlbW92ZWQgY2FwYWJpbGl0aWVzXG4gICAqL1xuICByZWFkb25seSBkcm9wPzogQ2FwYWJpbGl0eVtdO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBDb250YWluZXJTZWN1cml0eUNvbnRleHRgXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyU2VjdXJpdHlDb250ZXh0UHJvcHMge1xuXG4gIC8qKlxuICAgICogVGhlIFVJRCB0byBydW4gdGhlIGVudHJ5cG9pbnQgb2YgdGhlIGNvbnRhaW5lciBwcm9jZXNzLlxuICAgICpcbiAgICAqIEBkZWZhdWx0IC0gMjUwMDAuIEFuIGFyYml0cmFyeSBudW1iZXIgYmlnZ2VyIHRoYW4gOTk5OSBpcyBzZWxlY3RlZCBoZXJlLlxuICAgICogVGhpcyBpcyBzbyB0aGF0IHRoZSBjb250YWluZXIgaXMgYmxvY2tlZCB0byBhY2Nlc3MgaG9zdCBmaWxlcyBldmVuIGlmXG4gICAgKiBzb21laG93IGl0IG1hbmFnZXMgdG8gZ2V0IGFjY2VzcyB0byBob3N0IGZpbGUgc3lzdGVtLlxuICAgICovXG4gIHJlYWRvbmx5IHVzZXI/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAgKiBUaGUgR0lEIHRvIHJ1biB0aGUgZW50cnlwb2ludCBvZiB0aGUgY29udGFpbmVyIHByb2Nlc3MuXG4gICAgKlxuICAgICogQGRlZmF1bHQgLSAyNjAwMC4gQW4gYXJiaXRyYXJ5IG51bWJlciBiaWdnZXIgdGhhbiA5OTk5IGlzIHNlbGVjdGVkIGhlcmUuXG4gICAgKiBUaGlzIGlzIHNvIHRoYXQgdGhlIGNvbnRhaW5lciBpcyBibG9ja2VkIHRvIGFjY2VzcyBob3N0IGZpbGVzIGV2ZW4gaWZcbiAgICAqIHNvbWVob3cgaXQgbWFuYWdlcyB0byBnZXQgYWNjZXNzIHRvIGhvc3QgZmlsZSBzeXN0ZW0uXG4gICAgKi9cbiAgcmVhZG9ubHkgZ3JvdXA/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAgKiBJbmRpY2F0ZXMgdGhhdCB0aGUgY29udGFpbmVyIG11c3QgcnVuIGFzIGEgbm9uLXJvb3QgdXNlci5cbiAgICAqIElmIHRydWUsIHRoZSBLdWJlbGV0IHdpbGwgdmFsaWRhdGUgdGhlIGltYWdlIGF0IHJ1bnRpbWUgdG8gZW5zdXJlIHRoYXQgaXQgZG9lc1xuICAgICogbm90IHJ1biBhcyBVSUQgMCAocm9vdCkgYW5kIGZhaWwgdG8gc3RhcnQgdGhlIGNvbnRhaW5lciBpZiBpdCBkb2VzLlxuICAgICpcbiAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAqL1xuICByZWFkb25seSBlbnN1cmVOb25Sb290PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUnVuIGNvbnRhaW5lciBpbiBwcml2aWxlZ2VkIG1vZGUuIFByb2Nlc3NlcyBpbiBwcml2aWxlZ2VkIGNvbnRhaW5lcnMgYXJlIGVzc2VudGlhbGx5IGVxdWl2YWxlbnQgdG8gcm9vdCBvbiB0aGUgaG9zdC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHByaXZpbGVnZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgY29udGFpbmVyIGhhcyBhIHJlYWQtb25seSByb290IGZpbGVzeXN0ZW0uXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHJlYWRPbmx5Um9vdEZpbGVzeXN0ZW0/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIGEgcHJvY2VzcyBjYW4gZ2FpbiBtb3JlIHByaXZpbGVnZXMgdGhhbiBpdHMgcGFyZW50IHByb2Nlc3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBhbGxvd1ByaXZpbGVnZUVzY2FsYXRpb24/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBQT1NJWCBjYXBhYmlsaXRpZXMgZm9yIHJ1bm5pbmcgY29udGFpbmVyc1xuICAgKlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBjYXBhYmlsaXRpZXM/OiBDb250YWluZXJTZWN1dGlyeUNvbnRleHRDYXBhYmlsaXRpZXM7XG5cbiAgLyoqXG4gICogQ29udGFpbmVyJ3Mgc2VjY29tcCBwcm9maWxlIHNldHRpbmdzLiBPbmx5IG9uZSBwcm9maWxlIHNvdXJjZSBtYXkgYmUgc2V0XG4gICpcbiAgKiBAZGVmYXVsdCBub25lXG4gICovXG4gIHJlYWRvbmx5IHNlY2NvbXBQcm9maWxlPzogU2VjY29tcFByb2ZpbGU7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIG5ldHdvcmsgcG9ydCBpbiBhIHNpbmdsZSBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyUG9ydCB7XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBwb3J0IHRvIGV4cG9zZSBvbiB0aGUgcG9kJ3MgSVAgYWRkcmVzcy4gVGhpcyBtdXN0IGJlIGEgdmFsaWQgcG9ydCBudW1iZXIsIDAgPCB4IDwgNjU1MzYuXG4gICAqL1xuICByZWFkb25seSBudW1iZXI6IG51bWJlcjtcblxuICAvKipcbiAgICogV2hhdCBob3N0IElQIHRvIGJpbmQgdGhlIGV4dGVybmFsIHBvcnQgdG8uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gMTI3LjAuMC4xLlxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdElwPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBOdW1iZXIgb2YgcG9ydCB0byBleHBvc2Ugb24gdGhlIGhvc3QuIElmIHNwZWNpZmllZCwgdGhpcyBtdXN0IGJlIGEgdmFsaWQgcG9ydCBudW1iZXIsIDAgPCB4IDwgNjU1MzYuXG4gICAqIE1vc3QgY29udGFpbmVycyBkbyBub3QgbmVlZCB0aGlzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGF1dG8gZ2VuZXJhdGVkIGJ5IGt1YmVybmV0ZXMgYW5kIG1pZ2h0IGNoYW5nZSBvbiByZXN0YXJ0cy5cbiAgICovXG4gIHJlYWRvbmx5IGhvc3RQb3J0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBJZiBzcGVjaWZpZWQsIHRoaXMgbXVzdCBiZSBhbiBJQU5BX1NWQ19OQU1FIGFuZCB1bmlxdWUgd2l0aGluIHRoZSBwb2QuXG4gICAqIEVhY2ggbmFtZWQgcG9ydCBpbiBhIHBvZCBtdXN0IGhhdmUgYSB1bmlxdWUgbmFtZS5cbiAgICogTmFtZSBmb3IgdGhlIHBvcnQgdGhhdCBjYW4gYmUgcmVmZXJyZWQgdG8gYnkgc2VydmljZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcG9ydCBpcyBub3QgbmFtZWQuXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQcm90b2NvbCBmb3IgcG9ydC4gTXVzdCBiZSBVRFAsIFRDUCwgb3IgU0NUUC4gRGVmYXVsdHMgdG8gXCJUQ1BcIi5cbiAgICpcbiAgICogQGRlZmF1bHQgUHJvdG9jb2wuVENQXG4gICAqL1xuICByZWFkb25seSBwcm90b2NvbD86IFByb3RvY29sO1xuXG59XG5cbi8qKlxuICogTmV0d29yayBwcm90b2NvbHMuXG4gKi9cbmV4cG9ydCBlbnVtIFByb3RvY29sIHtcblxuICAvKipcbiAgICogVENQLlxuICAgKi9cbiAgVENQID0gJ1RDUCcsXG5cbiAgLyoqXG4gICAqIFVEUC5cbiAgICovXG4gIFVEUCA9ICdVRFAnLFxuXG4gIC8qKlxuICAgKiBTQ1RQLlxuICAgKi9cbiAgU0NUUCA9ICdTQ1RQJ1xufVxuXG5cbi8qKlxuICogQ29udGFpbmVyIHNlY3VyaXR5IGF0dHJpYnV0ZXMgYW5kIHNldHRpbmdzLlxuICovXG5leHBvcnQgY2xhc3MgQ29udGFpbmVyU2VjdXJpdHlDb250ZXh0IHtcblxuICBwdWJsaWMgcmVhZG9ubHkgZW5zdXJlTm9uUm9vdDogYm9vbGVhbjtcbiAgcHVibGljIHJlYWRvbmx5IHByaXZpbGVnZWQ6IGJvb2xlYW47XG4gIHB1YmxpYyByZWFkb25seSByZWFkT25seVJvb3RGaWxlc3lzdGVtOiBib29sZWFuO1xuICBwdWJsaWMgcmVhZG9ubHkgdXNlcj86IG51bWJlcjtcbiAgcHVibGljIHJlYWRvbmx5IGdyb3VwPzogbnVtYmVyO1xuICBwdWJsaWMgcmVhZG9ubHkgYWxsb3dQcml2aWxlZ2VFc2NhbGF0aW9uPzogYm9vbGVhbjtcbiAgcHVibGljIHJlYWRvbmx5IGNhcGFiaWxpdGllcz86IENvbnRhaW5lclNlY3V0aXJ5Q29udGV4dENhcGFiaWxpdGllcztcbiAgcHVibGljIHJlYWRvbmx5IHNlY2NvbXBQcm9maWxlPzogU2VjY29tcFByb2ZpbGU7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IENvbnRhaW5lclNlY3VyaXR5Q29udGV4dFByb3BzID0ge30pIHtcbiAgICBpZiAocHJvcHMuc2VjY29tcFByb2ZpbGU/LmxvY2FsaG9zdFByb2ZpbGUgJiYgcHJvcHMuc2VjY29tcFByb2ZpbGUudHlwZSAhPSBTZWNjb21wUHJvZmlsZVR5cGUuTE9DQUxIT1NUKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2xvY2FsaG9zdFByb2ZpbGUgbXVzdCBvbmx5IGJlIHNldCBpZiB0eXBlIGlzIFwiTG9jYWxob3N0XCInKTtcbiAgICB9XG4gICAgdGhpcy5lbnN1cmVOb25Sb290ID0gcHJvcHMuZW5zdXJlTm9uUm9vdCA/PyB0cnVlO1xuICAgIHRoaXMucHJpdmlsZWdlZCA9IHByb3BzLnByaXZpbGVnZWQgPz8gZmFsc2U7XG4gICAgdGhpcy5yZWFkT25seVJvb3RGaWxlc3lzdGVtID0gcHJvcHMucmVhZE9ubHlSb290RmlsZXN5c3RlbSA/PyB0cnVlO1xuICAgIHRoaXMudXNlciA9IHByb3BzLnVzZXI7XG4gICAgdGhpcy5ncm91cCA9IHByb3BzLmdyb3VwO1xuICAgIHRoaXMuYWxsb3dQcml2aWxlZ2VFc2NhbGF0aW9uID0gcHJvcHMuYWxsb3dQcml2aWxlZ2VFc2NhbGF0aW9uID8/IGZhbHNlO1xuICAgIHRoaXMuY2FwYWJpbGl0aWVzID0gcHJvcHMuY2FwYWJpbGl0aWVzO1xuICAgIHRoaXMuc2VjY29tcFByb2ZpbGUgPSBwcm9wcy5zZWNjb21wUHJvZmlsZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5TZWN1cml0eUNvbnRleHQge1xuICAgIHJldHVybiB7XG4gICAgICBydW5Bc0dyb3VwOiB0aGlzLmdyb3VwLFxuICAgICAgcnVuQXNVc2VyOiB0aGlzLnVzZXIsXG4gICAgICBydW5Bc05vblJvb3Q6IHRoaXMuZW5zdXJlTm9uUm9vdCxcbiAgICAgIHByaXZpbGVnZWQ6IHRoaXMucHJpdmlsZWdlZCxcbiAgICAgIHJlYWRPbmx5Um9vdEZpbGVzeXN0ZW06IHRoaXMucmVhZE9ubHlSb290RmlsZXN5c3RlbSxcbiAgICAgIGFsbG93UHJpdmlsZWdlRXNjYWxhdGlvbjogdGhpcy5hbGxvd1ByaXZpbGVnZUVzY2FsYXRpb24sXG4gICAgICBjYXBhYmlsaXRpZXM6IHRoaXMuY2FwYWJpbGl0aWVzLFxuICAgICAgc2VjY29tcFByb2ZpbGU6IHRoaXMuc2VjY29tcFByb2ZpbGUsXG4gICAgfTtcbiAgfVxuXG59XG5cbmV4cG9ydCBlbnVtIEVudkZpZWxkUGF0aHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHBvZC5cbiAgICovXG4gIFBPRF9OQU1FID0gJ21ldGFkYXRhLm5hbWUnLFxuXG4gIC8qKlxuICAgKiBUaGUgbmFtZXNwYWNlIG9mIHRoZSBwb2QuXG4gICAqL1xuICBQT0RfTkFNRVNQQUNFID0gJ21ldGFkYXRhLm5hbWVzcGFjZScsXG5cbiAgLyoqXG4gICAqIFRoZSB1aWQgb2YgdGhlIHBvZC5cbiAgICovXG4gIFBPRF9VSUQgPSAnbWV0YWRhdGEudWlkJyxcblxuICAvKipcbiAgICogVGhlIGxhYmVscyBvZiB0aGUgcG9kLlxuICAgKi9cbiAgUE9EX0xBQkVMID0gJ21ldGFkYXRhLmxhYmVscycsXG5cbiAgLyoqXG4gICAqIFRoZSBhbm5vdGF0aW9ucyBvZiB0aGUgcG9kLlxuICAgKi9cbiAgUE9EX0FOTk9UQVRJT04gPSAnbWV0YWRhdGEuYW5ub3RhdGlvbnMnLFxuXG4gIC8qKlxuICAgKiBUaGUgaXBBZGRyZXNzIG9mIHRoZSBwb2QuXG4gICAqL1xuICBQT0RfSVAgPSAnc3RhdHVzLnBvZElQJyxcblxuICAvKipcbiAgICogVGhlIHNlcnZpY2UgYWNjb3VudCBuYW1lIG9mIHRoZSBwb2QuXG4gICAqL1xuICBTRVJWSUNFX0FDQ09VTlRfTkFNRSA9ICdzcGVjLnNlcnZpY2VBY2NvdW50TmFtZScsXG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBub2RlLlxuICAgKi9cbiAgTk9ERV9OQU1FID0gJ3NwZWMubm9kZU5hbWUnLFxuXG4gIC8qKlxuICAgKiBUaGUgaXBBZGRyZXNzIG9mIHRoZSBub2RlLlxuICAgKi9cbiAgTk9ERV9JUCA9ICdzdGF0dXMuaG9zdElQJyxcblxuICAvKipcbiAgICogVGhlIGlwQWRkcmVzZXNzIG9mIHRoZSBwb2QuXG4gICAqL1xuICBQT0RfSVBTID0gJ3N0YXR1cy5wb2RJUHMnLFxufVxuXG5leHBvcnQgZW51bSBSZXNvdXJjZUZpZWxkUGF0aHMge1xuICAvKipcbiAgICogQ1BVIGxpbWl0IG9mIHRoZSBjb250YWluZXIuXG4gICAqL1xuICBDUFVfTElNSVQgPSAnbGltaXRzLmNwdScsXG5cbiAgLyoqXG4gICAqIE1lbW9yeSBsaW1pdCBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgTUVNT1JZX0xJTUlUID0gJ2xpbWl0cy5tZW1vcnknLFxuXG4gIC8qKlxuICAgKiBDUFUgcmVxdWVzdCBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgQ1BVX1JFUVVFU1QgPSAncmVxdWVzdHMuY3B1JyxcblxuICAvKipcbiAgICogTWVtb3J5IHJlcXVlc3Qgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIE1FTU9SWV9SRVFVRVNUID0gJ3JlcXVlc3RzLm1lbW9yeScsXG5cbiAgLyoqXG4gICAqIEVwaGVtZXJhbCBzdG9yYWdlIGxpbWl0IG9mIHRoZSBjb250YWluZXIuXG4gICAqL1xuICBTVE9SQUdFX0xJTUlUID0gJ2xpbWl0cy5lcGhlbWVyYWwtc3RvcmFnZScsXG5cbiAgLyoqXG4gICAqIEVwaGVtZXJhbCBzdG9yYWdlIHJlcXVlc3Qgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIFNUT1JBR0VfUkVRVUVTVCA9ICdyZXF1ZXN0cy5lcGhlbWVyYWwtc3RvcmFnZScsXG59XG5cbi8qKlxuICogT3B0aW9ucyB0byBzcGVjaWZ5IGFuIGVudmlvbm1lbnQgdmFyaWFibGUgdmFsdWUgZnJvbSBhIENvbmZpZ01hcCBrZXkuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRW52VmFsdWVGcm9tQ29uZmlnTWFwT3B0aW9ucyB7XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgd2hldGhlciB0aGUgQ29uZmlnTWFwIG9yIGl0cyBrZXkgbXVzdCBiZSBkZWZpbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgb3B0aW9uYWw/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gc3BlY2lmeSBhbiBlbnZpcm9ubWVudCB2YXJpYWJsZSB2YWx1ZSBmcm9tIGEgU2VjcmV0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVudlZhbHVlRnJvbVNlY3JldE9wdGlvbnMge1xuXG4gIC8qKlxuICAgKiBTcGVjaWZ5IHdoZXRoZXIgdGhlIFNlY3JldCBvciBpdHMga2V5IG11c3QgYmUgZGVmaW5lZC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9wdGlvbmFsPzogYm9vbGVhbjtcblxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gc3BlY2lmeSBhbiBlbnZpcm9ubWVudCB2YXJpYWJsZSB2YWx1ZSBmcm9tIHRoZSBwcm9jZXNzIGVudmlyb25tZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVudlZhbHVlRnJvbVByb2Nlc3NPcHRpb25zIHtcblxuICAvKipcbiAgICogU3BlY2lmeSB3aGV0aGVyIHRoZSBrZXkgbXVzdCBleGlzdCBpbiB0aGUgZW52aXJvbm1lbnQuXG4gICAqIElmIHRoaXMgaXMgc2V0IHRvIHRydWUsIGFuZCB0aGUga2V5IGRvZXMgbm90IGV4aXN0LCBhbiBlcnJvciB3aWxsIHRocm93bi5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlcXVpcmVkPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIHRvIHNwZWNpZnkgYW4gZW52aXJvbm1lbnQgdmFyaWFibGUgdmFsdWUgZnJvbSBhIGZpZWxkIHJlZmVyZW5jZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbnZWYWx1ZUZyb21GaWVsZFJlZk9wdGlvbnMge1xuICAvKipcbiAgICogVmVyc2lvbiBvZiB0aGUgc2NoZW1hIHRoZSBGaWVsZFBhdGggaXMgd3JpdHRlbiBpbiB0ZXJtcyBvZi5cbiAgICovXG4gIHJlYWRvbmx5IGFwaVZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBrZXkgdG8gc2VsZWN0IHRoZSBwb2QgbGFiZWwgb3IgYW5ub3RhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGtleT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBPcHRpb25zIHRvIHNwZWNpZnkgYW4gZW52aXJvbm1lbnQgdmFyaWFibGUgdmFsdWUgZnJvbSBhIHJlc291cmNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVudlZhbHVlRnJvbVJlc291cmNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgY29udGFpbmVyIHRvIHNlbGVjdCB0aGUgdmFsdWUgZnJvbS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnRhaW5lcj86IENvbnRhaW5lcjtcblxuICAvKipcbiAgICogVGhlIG91dHB1dCBmb3JtYXQgb2YgdGhlIGV4cG9zZWQgcmVzb3VyY2UuXG4gICAqL1xuICByZWFkb25seSBkaXZpc29yPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFV0aWxpdHkgY2xhc3MgZm9yIGNyZWF0aW5nIHJlYWRpbmcgZW52IHZhbHVlcyBmcm9tIHZhcmlvdXMgc291cmNlcy5cbiAqL1xuZXhwb3J0IGNsYXNzIEVudlZhbHVlIHtcblxuICAvKipcbiAgICogQ3JlYXRlIGEgdmFsdWUgYnkgcmVhZGluZyBhIHNwZWNpZmljIGtleSBpbnNpZGUgYSBjb25maWcgbWFwLlxuICAgKlxuICAgKiBAcGFyYW0gY29uZmlnTWFwIC0gVGhlIGNvbmZpZyBtYXAuXG4gICAqIEBwYXJhbSBrZXkgLSBUaGUga2V5IHRvIGV4dHJhY3QgdGhlIHZhbHVlIGZyb20uXG4gICAqIEBwYXJhbSBvcHRpb25zIC0gQWRkaXRpb25hbCBvcHRpb25zLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQ29uZmlnTWFwKGNvbmZpZ01hcDogY29uZmlnbWFwLklDb25maWdNYXAsIGtleTogc3RyaW5nLCBvcHRpb25zOiBFbnZWYWx1ZUZyb21Db25maWdNYXBPcHRpb25zID0ge30pOiBFbnZWYWx1ZSB7XG5cbiAgICBjb25zdCBzb3VyY2U6IGs4cy5FbnZWYXJTb3VyY2UgPSB7XG4gICAgICBjb25maWdNYXBLZXlSZWY6IHtcbiAgICAgICAgbmFtZTogY29uZmlnTWFwLm5hbWUsXG4gICAgICAgIGtleSxcbiAgICAgICAgb3B0aW9uYWw6IG9wdGlvbnMub3B0aW9uYWwsXG4gICAgICB9LFxuICAgIH07XG4gICAgcmV0dXJuIG5ldyBFbnZWYWx1ZSh1bmRlZmluZWQsIHNvdXJjZSk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBhbiBlbnZpcm9ubWVudCB2YWx1ZSBmcm9tIGEgc2VjcmV0IEpTT04gdmFsdWUuXG4gICAqXG4gICAqIEBwYXJhbSBzZWNyZXRWYWx1ZSBUaGUgc2VjcmV0IHZhbHVlIChzZWNyZW50ICsga2V5KVxuICAgKiBAcGFyYW0gb3B0aW9ucyBBZGRpdGlvbmFsIG9wdGlvbnNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVNlY3JldFZhbHVlKHNlY3JldFZhbHVlOiBzZWNyZXQuU2VjcmV0VmFsdWUsIG9wdGlvbnM6IEVudlZhbHVlRnJvbVNlY3JldE9wdGlvbnMgPSB7fSk6IEVudlZhbHVlIHtcbiAgICBjb25zdCBzb3VyY2U6IGs4cy5FbnZWYXJTb3VyY2UgPSB7XG4gICAgICBzZWNyZXRLZXlSZWY6IHtcbiAgICAgICAgbmFtZTogc2VjcmV0VmFsdWUuc2VjcmV0Lm5hbWUsXG4gICAgICAgIGtleTogc2VjcmV0VmFsdWUua2V5LFxuICAgICAgICBvcHRpb25hbDogb3B0aW9ucy5vcHRpb25hbCxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIHJldHVybiBuZXcgRW52VmFsdWUodW5kZWZpbmVkLCBzb3VyY2UpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHZhbHVlIGZyb20gdGhlIGdpdmVuIGFyZ3VtZW50LlxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgLSBUaGUgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21WYWx1ZSh2YWx1ZTogc3RyaW5nKTogRW52VmFsdWUge1xuICAgIHJldHVybiBuZXcgRW52VmFsdWUodmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqXG4gICAqIENyZWF0ZSBhIHZhbHVlIGZyb20gYSBmaWVsZCByZWZlcmVuY2UuXG4gICAqXG4gICAqIEBwYXJhbSBmaWVsZFBhdGg6IFRoZSBmaWVsZCByZWZlcmVuY2UuXG4gICAqIEBwYXJhbSBvcHRpb25zOiBBZGRpdGlvbmFsIG9wdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21GaWVsZFJlZihmaWVsZFBhdGg6IEVudkZpZWxkUGF0aHMsIG9wdGlvbnM6IEVudlZhbHVlRnJvbUZpZWxkUmVmT3B0aW9ucyA9IHt9KTogRW52VmFsdWUge1xuICAgIGxldCBuZWVkc0tleTogYm9vbGVhbiA9IGZhbHNlO1xuXG4gICAgaWYgKGZpZWxkUGF0aCA9PT0gRW52RmllbGRQYXRocy5QT0RfTEFCRUwgfHwgZmllbGRQYXRoID09PSBFbnZGaWVsZFBhdGhzLlBPRF9BTk5PVEFUSU9OKSB7XG4gICAgICBuZWVkc0tleSA9IHRydWU7XG4gICAgfVxuXG4gICAgaWYgKG5lZWRzS2V5ICYmIG9wdGlvbnMua2V5ID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtmaWVsZFBhdGh9IHJlcXVpcmVzIGEga2V5YCk7XG4gICAgfVxuXG4gICAgY29uc3Qgc291cmNlOiBrOHMuRW52VmFyU291cmNlID0ge1xuICAgICAgZmllbGRSZWY6IHtcbiAgICAgICAgZmllbGRQYXRoOiAoYCR7ZmllbGRQYXRofWAgKyAobmVlZHNLZXkgPyBgWycke29wdGlvbnMua2V5fSddYCA6ICcnKSksXG4gICAgICAgIC4uLihvcHRpb25zLmFwaVZlcnNpb24gPyB7IGFwaVZlcnNpb246IG9wdGlvbnMuYXBpVmVyc2lvbiB9IDoge30pLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgcmV0dXJuIG5ldyBFbnZWYWx1ZSh1bmRlZmluZWQsIHNvdXJjZSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgdmFsdWUgZnJvbSBhIHJlc291cmNlLlxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2U6IFJlc291cmNlIHRvIHNlbGVjdCB0aGUgdmFsdWUgZnJvbS5cbiAgICogQHBhcmFtIG9wdGlvbnM6IEFkZGl0aW9uYWwgb3B0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVJlc291cmNlKHJlc291cmNlOiBSZXNvdXJjZUZpZWxkUGF0aHMsIG9wdGlvbnM6IEVudlZhbHVlRnJvbVJlc291cmNlT3B0aW9ucyA9IHt9KTogRW52VmFsdWUge1xuICAgIGNvbnN0IHNvdXJjZTogazhzLkVudlZhclNvdXJjZSA9IHtcbiAgICAgIHJlc291cmNlRmllbGRSZWY6IHtcbiAgICAgICAgcmVzb3VyY2U6IGAke3Jlc291cmNlfWAsXG4gICAgICAgIC4uLiAob3B0aW9ucy5kaXZpc29yID8geyBkaXZpc29yOiBrOHMuSW50T3JTdHJpbmcuZnJvbVN0cmluZyhvcHRpb25zLmRpdmlzb3IpIH0gOiB7fSksXG4gICAgICAgIC4uLiAob3B0aW9ucy5jb250YWluZXIgPyB7IGNvbnRhaW5lck5hbWU6IG9wdGlvbnMuY29udGFpbmVyLm5hbWUgfSA6IHt9KSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIHJldHVybiBuZXcgRW52VmFsdWUodW5kZWZpbmVkLCBzb3VyY2UpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHZhbHVlIGZyb20gYSBrZXkgaW4gdGhlIGN1cnJlbnQgcHJvY2VzcyBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQHBhcmFtIGtleSAtIFRoZSBrZXkgdG8gcmVhZC5cbiAgICogQHBhcmFtIG9wdGlvbnMgLSBBZGRpdGlvbmFsIG9wdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Qcm9jZXNzKGtleTogc3RyaW5nLCBvcHRpb25zOiBFbnZWYWx1ZUZyb21Qcm9jZXNzT3B0aW9ucyA9IHt9KTogRW52VmFsdWUge1xuXG4gICAgY29uc3QgdmFsdWUgPSBwcm9jZXNzLmVudltrZXldO1xuXG4gICAgaWYgKG9wdGlvbnMucmVxdWlyZWQgJiYgIXZhbHVlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYE1pc3NpbmcgJHtrZXl9IGVudiB2YXJpYWJsZWApO1xuICAgIH1cblxuICAgIHJldHVybiBFbnZWYWx1ZS5mcm9tVmFsdWUodmFsdWUhKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHZhbHVlPzogYW55LCBwdWJsaWMgcmVhZG9ubHkgdmFsdWVGcm9tPzogYW55KSB7IH1cbn1cblxuZXhwb3J0IGVudW0gSW1hZ2VQdWxsUG9saWN5IHtcbiAgLyoqXG4gICAqIEV2ZXJ5IHRpbWUgdGhlIGt1YmVsZXQgbGF1bmNoZXMgYSBjb250YWluZXIsIHRoZSBrdWJlbGV0IHF1ZXJpZXMgdGhlIGNvbnRhaW5lciBpbWFnZSByZWdpc3RyeVxuICAgKiB0byByZXNvbHZlIHRoZSBuYW1lIHRvIGFuIGltYWdlIGRpZ2VzdC4gSWYgdGhlIGt1YmVsZXQgaGFzIGEgY29udGFpbmVyIGltYWdlIHdpdGggdGhhdCBleGFjdFxuICAgKiBkaWdlc3QgY2FjaGVkIGxvY2FsbHksIHRoZSBrdWJlbGV0IHVzZXMgaXRzIGNhY2hlZCBpbWFnZTsgb3RoZXJ3aXNlLCB0aGUga3ViZWxldCBkb3dubG9hZHNcbiAgICogKHB1bGxzKSB0aGUgaW1hZ2Ugd2l0aCB0aGUgcmVzb2x2ZWQgZGlnZXN0LCBhbmQgdXNlcyB0aGF0IGltYWdlIHRvIGxhdW5jaCB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBEZWZhdWx0IGlzIEFsd2F5cyBpZiBJbWFnZVB1bGxQb2xpY3kgaXMgb21pdHRlZCBhbmQgZWl0aGVyIHRoZSBpbWFnZSB0YWcgaXMgOmxhdGVzdCBvclxuICAgKiB0aGUgaW1hZ2UgdGFnIGlzIG9taXR0ZWQuXG4gICAqL1xuICBBTFdBWVMgPSAnQWx3YXlzJyxcblxuICAvKipcbiAgICogVGhlIGltYWdlIGlzIHB1bGxlZCBvbmx5IGlmIGl0IGlzIG5vdCBhbHJlYWR5IHByZXNlbnQgbG9jYWxseS5cbiAgICpcbiAgICogRGVmYXVsdCBpcyBJZk5vdFByZXNlbnQgaWYgSW1hZ2VQdWxsUG9saWN5IGlzIG9taXR0ZWQgYW5kIHRoZSBpbWFnZSB0YWcgaXMgcHJlc2VudCBidXRcbiAgICogbm90IDpsYXRlc3RcbiAgICovXG4gIElGX05PVF9QUkVTRU5UID0gJ0lmTm90UHJlc2VudCcsXG5cbiAgLyoqXG4gICAqIFRoZSBpbWFnZSBpcyBhc3N1bWVkIHRvIGV4aXN0IGxvY2FsbHkuIE5vIGF0dGVtcHQgaXMgbWFkZSB0byBwdWxsIHRoZSBpbWFnZS5cbiAgICovXG4gIE5FVkVSID0gJ05ldmVyJyxcbn1cblxuLyoqXG4gKiBDb250YWluZXIgbGlmZWN5Y2xlIHByb3BlcnRpZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyTGlmZWN5Y2xlIHtcblxuICAvKipcbiAgICogVGhpcyBob29rIGlzIGV4ZWN1dGVkIGltbWVkaWF0ZWx5IGFmdGVyIGEgY29udGFpbmVyIGlzIGNyZWF0ZWQuIEhvd2V2ZXIsXG4gICAqIHRoZXJlIGlzIG5vIGd1YXJhbnRlZSB0aGF0IHRoZSBob29rIHdpbGwgZXhlY3V0ZSBiZWZvcmUgdGhlIGNvbnRhaW5lciBFTlRSWVBPSU5ULlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHBvc3Qgc3RhcnQgaGFuZGxlci5cbiAgICovXG4gIHJlYWRvbmx5IHBvc3RTdGFydD86IGhhbmRsZXIuSGFuZGxlcjtcblxuICAvKipcbiAgICogVGhpcyBob29rIGlzIGNhbGxlZCBpbW1lZGlhdGVseSBiZWZvcmUgYSBjb250YWluZXIgaXMgdGVybWluYXRlZCBkdWUgdG8gYW4gQVBJIHJlcXVlc3Qgb3IgbWFuYWdlbWVudFxuICAgKiBldmVudCBzdWNoIGFzIGEgbGl2ZW5lc3Mvc3RhcnR1cCBwcm9iZSBmYWlsdXJlLCBwcmVlbXB0aW9uLCByZXNvdXJjZSBjb250ZW50aW9uIGFuZCBvdGhlcnMuXG4gICAqIEEgY2FsbCB0byB0aGUgUHJlU3RvcCBob29rIGZhaWxzIGlmIHRoZSBjb250YWluZXIgaXMgYWxyZWFkeSBpbiBhIHRlcm1pbmF0ZWQgb3IgY29tcGxldGVkIHN0YXRlXG4gICAqIGFuZCB0aGUgaG9vayBtdXN0IGNvbXBsZXRlIGJlZm9yZSB0aGUgVEVSTSBzaWduYWwgdG8gc3RvcCB0aGUgY29udGFpbmVyIGNhbiBiZSBzZW50LlxuICAgKiBUaGUgUG9kJ3MgdGVybWluYXRpb24gZ3JhY2UgcGVyaW9kIGNvdW50ZG93biBiZWdpbnMgYmVmb3JlIHRoZSBQcmVTdG9wIGhvb2sgaXMgZXhlY3V0ZWQsXG4gICAqIHNvIHJlZ2FyZGxlc3Mgb2YgdGhlIG91dGNvbWUgb2YgdGhlIGhhbmRsZXIsIHRoZSBjb250YWluZXIgd2lsbCBldmVudHVhbGx5IHRlcm1pbmF0ZVxuICAgKiB3aXRoaW4gdGhlIFBvZCdzIHRlcm1pbmF0aW9uIGdyYWNlIHBlcmlvZC4gTm8gcGFyYW1ldGVycyBhcmUgcGFzc2VkIHRvIHRoZSBoYW5kbGVyLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3dvcmtsb2Fkcy9wb2RzL3BvZC1saWZlY3ljbGUvI3BvZC10ZXJtaW5hdGlvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHByZSBzdG9wIGhhbmRsZXIuXG4gICAqL1xuICByZWFkb25seSBwcmVTdG9wPzogaGFuZGxlci5IYW5kbGVyO1xuXG59XG5cbi8qKlxuICogUmVzdGFydFBvbGljeSBkZWZpbmVzIHRoZSByZXN0YXJ0IGJlaGF2aW9yIG9mIGluZGl2aWR1YWwgY29udGFpbmVycyBpbiBhIHBvZC5cbiAqIFRoaXMgZmllbGQgbWF5IG9ubHkgYmUgc2V0IGZvciBpbml0IGNvbnRhaW5lcnMsIGFuZCB0aGUgb25seSBhbGxvd2VkIHZhbHVlIGlzIFwiQWx3YXlzXCIuXG4gKiBGb3Igbm9uLWluaXQgY29udGFpbmVycyBvciB3aGVuIHRoaXMgZmllbGQgaXMgbm90IHNwZWNpZmllZCxcbiAqIHRoZSByZXN0YXJ0IGJlaGF2aW9yIGlzIGRlZmluZWQgYnkgdGhlIFBvZCdzIHJlc3RhcnQgcG9saWN5IGFuZCB0aGUgY29udGFpbmVyIHR5cGUuXG4gKiBTZXR0aW5nIHRoZSBSZXN0YXJ0UG9saWN5IGFzIFwiQWx3YXlzXCIgZm9yIHRoZSBpbml0IGNvbnRhaW5lciB3aWxsIGhhdmUgdGhlIGZvbGxvd2luZyBlZmZlY3Q6XG4gKiB0aGlzIGluaXQgY29udGFpbmVyIHdpbGwgYmUgY29udGludWFsbHkgcmVzdGFydGVkIG9uIGV4aXQgdW50aWwgYWxsIHJlZ3VsYXIgY29udGFpbmVycyBoYXZlIHRlcm1pbmF0ZWQuXG4gKiBPbmNlIGFsbCByZWd1bGFyIGNvbnRhaW5lcnMgaGF2ZSBjb21wbGV0ZWQsIGFsbCBpbml0IGNvbnRhaW5lcnMgd2l0aCByZXN0YXJ0UG9saWN5IFwiQWx3YXlzXCIgd2lsbCBiZSBzaHV0IGRvd24uXG4gKiBUaGlzIGxpZmVjeWNsZSBkaWZmZXJzIGZyb20gbm9ybWFsIGluaXQgY29udGFpbmVycyBhbmQgaXMgb2Z0ZW4gcmVmZXJyZWQgdG8gYXMgYSBcInNpZGVjYXJcIiBjb250YWluZXIuXG4gKlxuICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy93b3JrbG9hZHMvcG9kcy9zaWRlY2FyLWNvbnRhaW5lcnMvXG4gKi9cbmV4cG9ydCBlbnVtIENvbnRhaW5lclJlc3RhcnRQb2xpY3kge1xuXG4gIC8qKlxuICAgKiBJZiBhbiBpbml0IGNvbnRhaW5lciBpcyBjcmVhdGVkIHdpdGggaXRzIHJlc3RhcnRQb2xpY3kgc2V0IHRvIEFsd2F5cyxcbiAgICogaXQgd2lsbCBzdGFydCBhbmQgcmVtYWluIHJ1bm5pbmcgZHVyaW5nIHRoZSBlbnRpcmUgbGlmZSBvZiB0aGUgUG9kLlxuICAgKiBGb3IgcmVndWxhciBjb250YWluZXJzLCB0aGlzIGlzIGlnbm9yZWQgYnkgS3ViZXJuZXRlcy5cbiAgICovXG4gIEFMV0FZUyA9ICdBbHdheXMnXG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGEgY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lclByb3BzIGV4dGVuZHMgQ29udGFpbmVyT3B0cyB7XG5cbiAgLyoqXG4gICAqIERvY2tlciBpbWFnZSBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2U6IHN0cmluZztcbn1cblxuLyoqXG4gKiBPcHRpb25hbCBwcm9wZXJ0aWVzIG9mIGEgY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lck9wdHMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgY29udGFpbmVyIHNwZWNpZmllZCBhcyBhIEROU19MQUJFTC4gRWFjaCBjb250YWluZXIgaW4gYSBwb2QgbXVzdCBoYXZlIGEgdW5pcXVlIG5hbWUgKEROU19MQUJFTCkuIENhbm5vdCBiZSB1cGRhdGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAnbWFpbidcbiAgICovXG4gIHJlYWRvbmx5IG5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIC0gdXNlIGBwb3J0TnVtYmVyYC5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBwb3J0IHRvIGV4cG9zZSBvbiB0aGUgcG9kJ3MgSVAgYWRkcmVzcy4gVGhpcyBtdXN0IGJlIGEgdmFsaWQgcG9ydCBudW1iZXIsIDAgPCB4IDwgNjU1MzYuXG4gICAqXG4gICAqIFRoaXMgaXMgYSBjb252aW5pZW5jZSBwcm9wZXJ0eSBpZiBhbGwgeW91IG5lZWQgYSBzaW5nbGUgVENQIG51bWJlcmVkIHBvcnQuXG4gICAqIEluIGNhc2UgbW9yZSBhZHZhbmNlZCBjb25maWd1YXJ0aW9uIGlzIHJlcXVpcmVkLCB1c2UgdGhlIGBwb3J0c2AgcHJvcGVydHkuXG4gICAqXG4gICAqIFRoaXMgcG9ydCBpcyBhZGRlZCB0byB0aGUgbGlzdCBvZiBwb3J0cyBtZW50aW9uZWQgaW4gdGhlIGBwb3J0c2AgcHJvcGVydHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gT25seSB0aGUgcG9ydHMgbWVudGlvbmQgaW4gdGhlIGBwb3J0c2AgcHJvcGVydHkgYXJlIGV4cG9zZWQuXG4gICAqL1xuICByZWFkb25seSBwb3J0TnVtYmVyPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIHBvcnRzIHRvIGV4cG9zZSBmcm9tIHRoaXMgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE9ubHkgdGhlIHBvcnQgbWVudGlvbmVkIGluIHRoZSBgcG9ydE51bWJlcmAgcHJvcGVydHkgaXMgZXhwb3NlZC5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnRzPzogQ29udGFpbmVyUG9ydFtdO1xuXG4gIC8qKlxuICAgKiBFbnRyeXBvaW50IGFycmF5LiBOb3QgZXhlY3V0ZWQgd2l0aGluIGEgc2hlbGwuIFRoZSBkb2NrZXIgaW1hZ2UncyBFTlRSWVBPSU5UIGlzIHVzZWQgaWYgdGhpcyBpcyBub3QgcHJvdmlkZWQuIFZhcmlhYmxlIHJlZmVyZW5jZXMgJChWQVJfTkFNRSkgYXJlIGV4cGFuZGVkIHVzaW5nIHRoZSBjb250YWluZXIncyBlbnZpcm9ubWVudC5cbiAgICogSWYgYSB2YXJpYWJsZSBjYW5ub3QgYmUgcmVzb2x2ZWQsIHRoZSByZWZlcmVuY2UgaW4gdGhlIGlucHV0IHN0cmluZyB3aWxsIGJlIHVuY2hhbmdlZC4gVGhlICQoVkFSX05BTUUpIHN5bnRheCBjYW4gYmUgZXNjYXBlZCB3aXRoIGEgZG91YmxlICQkLCBpZTogJCQoVkFSX05BTUUpLlxuICAgKiBFc2NhcGVkIHJlZmVyZW5jZXMgd2lsbCBuZXZlciBiZSBleHBhbmRlZCwgcmVnYXJkbGVzcyBvZiB3aGV0aGVyIHRoZSB2YXJpYWJsZSBleGlzdHMgb3Igbm90LiBDYW5ub3QgYmUgdXBkYXRlZC5cbiAgICogTW9yZSBpbmZvOiBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy90YXNrcy9pbmplY3QtZGF0YS1hcHBsaWNhdGlvbi9kZWZpbmUtY29tbWFuZC1hcmd1bWVudC1jb250YWluZXIvI3J1bm5pbmctYS1jb21tYW5kLWluLWEtc2hlbGxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgZG9ja2VyIGltYWdlJ3MgRU5UUllQT0lOVC5cbiAgICovXG4gIHJlYWRvbmx5IGNvbW1hbmQ/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogQXJndW1lbnRzIHRvIHRoZSBlbnRyeXBvaW50LiBUaGUgZG9ja2VyIGltYWdlJ3MgQ01EIGlzIHVzZWQgaWYgYGNvbW1hbmRgIGlzXG4gICAqIG5vdCBwcm92aWRlZC5cbiAgICpcbiAgICogVmFyaWFibGUgcmVmZXJlbmNlcyAkKFZBUl9OQU1FKSBhcmUgZXhwYW5kZWQgdXNpbmcgdGhlIGNvbnRhaW5lcidzXG4gICAqIGVudmlyb25tZW50LiBJZiBhIHZhcmlhYmxlIGNhbm5vdCBiZSByZXNvbHZlZCwgdGhlIHJlZmVyZW5jZSBpbiB0aGUgaW5wdXRcbiAgICogc3RyaW5nIHdpbGwgYmUgdW5jaGFuZ2VkLiBUaGUgJChWQVJfTkFNRSkgc3ludGF4IGNhbiBiZSBlc2NhcGVkIHdpdGggYVxuICAgKiBkb3VibGUgJCQsIGllOiAkJChWQVJfTkFNRSkuIEVzY2FwZWQgcmVmZXJlbmNlcyB3aWxsIG5ldmVyIGJlIGV4cGFuZGVkLFxuICAgKiByZWdhcmRsZXNzIG9mIHdoZXRoZXIgdGhlIHZhcmlhYmxlIGV4aXN0cyBvciBub3QuXG4gICAqXG4gICAqIENhbm5vdCBiZSB1cGRhdGVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL3Rhc2tzL2luamVjdC1kYXRhLWFwcGxpY2F0aW9uL2RlZmluZS1jb21tYW5kLWFyZ3VtZW50LWNvbnRhaW5lci8jcnVubmluZy1hLWNvbW1hbmQtaW4tYS1zaGVsbFxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgYXJncz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBDb250YWluZXIncyB3b3JraW5nIGRpcmVjdG9yeS4gSWYgbm90IHNwZWNpZmllZCwgdGhlIGNvbnRhaW5lciBydW50aW1lJ3MgZGVmYXVsdCB3aWxsIGJlIHVzZWQsIHdoaWNoIG1pZ2h0IGJlIGNvbmZpZ3VyZWQgaW4gdGhlIGNvbnRhaW5lciBpbWFnZS4gQ2Fubm90IGJlIHVwZGF0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIGNvbnRhaW5lciBydW50aW1lJ3MgZGVmYXVsdC5cbiAgICovXG4gIHJlYWRvbmx5IHdvcmtpbmdEaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEVudmlyb25tZW50IHZhcmlhYmxlcyB0byBzZXQgaW4gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBlbnZpcm9ubWVudCB2YXJpYWJsZXMuXG4gICAqL1xuICByZWFkb25seSBlbnZWYXJpYWJsZXM/OiB7IFtuYW1lOiBzdHJpbmddOiBFbnZWYWx1ZSB9O1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIHNvdXJjZXMgdG8gcG9wdWxhdGUgZW52aXJvbm1lbnQgdmFyaWFibGVzIGluIHRoZSBjb250YWluZXIuXG4gICAqIFdoZW4gYSBrZXkgZXhpc3RzIGluIG11bHRpcGxlIHNvdXJjZXMsIHRoZSB2YWx1ZSBhc3NvY2lhdGVkIHdpdGhcbiAgICogdGhlIGxhc3Qgc291cmNlIHdpbGwgdGFrZSBwcmVjZWRlbmNlLiBWYWx1ZXMgZGVmaW5lZCBieSB0aGUgYGVudlZhcmlhYmxlc2AgcHJvcGVydHlcbiAgICogd2l0aCBhIGR1cGxpY2F0ZSBrZXkgd2lsbCB0YWtlIHByZWNlZGVuY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gc291cmNlcy5cbiAgICovXG4gIHJlYWRvbmx5IGVudkZyb20/OiBFbnZGcm9tW107XG5cbiAgLyoqXG4gICAqIFBvZCB2b2x1bWVzIHRvIG1vdW50IGludG8gdGhlIGNvbnRhaW5lcidzIGZpbGVzeXN0ZW0uIENhbm5vdCBiZSB1cGRhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lTW91bnRzPzogVm9sdW1lTW91bnRbXTtcblxuICAvKipcbiAgICogSW1hZ2UgcHVsbCBwb2xpY3kgZm9yIHRoaXMgY29udGFpbmVyXG4gICAqIEBkZWZhdWx0IEltYWdlUHVsbFBvbGljeS5BTFdBWVNcbiAgICovXG4gIHJlYWRvbmx5IGltYWdlUHVsbFBvbGljeT86IEltYWdlUHVsbFBvbGljeTtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB3aGVuIHRoZSBjb250YWluZXIgaXMgcmVhZHkgdG8gc2VydmUgdHJhZmZpYy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyByZWFkaW5lc3MgcHJvYmUgaXMgZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgcmVhZGluZXNzPzogcHJvYmUuUHJvYmU7XG5cbiAgLyoqXG4gICAqIFBlcmlvZGljIHByb2JlIG9mIGNvbnRhaW5lciBsaXZlbmVzcy4gQ29udGFpbmVyIHdpbGwgYmUgcmVzdGFydGVkIGlmIHRoZSBwcm9iZSBmYWlscy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBsaXZlbmVzcyBwcm9iZSBpcyBkZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBsaXZlbmVzcz86IHByb2JlLlByb2JlO1xuXG4gIC8qKlxuICAgKiBTdGFydHVwUHJvYmUgaW5kaWNhdGVzIHRoYXQgdGhlIFBvZCBoYXMgc3VjY2Vzc2Z1bGx5IGluaXRpYWxpemVkLlxuICAgKiBJZiBzcGVjaWZpZWQsIG5vIG90aGVyIHByb2JlcyBhcmUgZXhlY3V0ZWQgdW50aWwgdGhpcyBjb21wbGV0ZXMgc3VjY2Vzc2Z1bGx5XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgYSBwb3J0IGlzIHByb3ZpZGVkLCB0aGVuIGtub2NrcyBvbiB0aGF0IHBvcnRcbiAgICogdG8gZGV0ZXJtaW5lIHdoZW4gdGhlIGNvbnRhaW5lciBpcyByZWFkeSBmb3IgcmVhZGluZXNzIGFuZFxuICAgKiBsaXZlbmVzcyBwcm9iZSBjaGVja3MuXG4gICAqIE90aGVyd2lzZSwgbm8gc3RhcnR1cCBwcm9iZSBpcyBkZWZpbmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3RhcnR1cD86IHByb2JlLlByb2JlO1xuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYWN0aW9ucyB0aGF0IHRoZSBtYW5hZ2VtZW50IHN5c3RlbSBzaG91bGQgdGFrZSBpbiByZXNwb25zZSB0byBjb250YWluZXIgbGlmZWN5Y2xlIGV2ZW50cy5cbiAgICovXG4gIHJlYWRvbmx5IGxpZmVjeWNsZT86IENvbnRhaW5lckxpZmVjeWNsZTtcblxuICAvKipcbiAgICogQ29tcHV0ZSByZXNvdXJjZXMgKENQVSBhbmQgbWVtb3J5IHJlcXVlc3RzIGFuZCBsaW1pdHMpIHJlcXVpcmVkIGJ5IHRoZSBjb250YWluZXJcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9jb25maWd1cmF0aW9uL21hbmFnZS1yZXNvdXJjZXMtY29udGFpbmVycy9cbiAgICpcbiAgICogQGRlZmF1bHRcbiAgICogICAgY3B1OlxuICAgKiAgICAgIHJlcXVlc3Q6IDEwMDAgbWlsbGlzXG4gICAqICAgICAgbGltaXQ6IDE1MDAgbWlsbGlzXG4gICAqICAgIG1lbW9yeTpcbiAgICogICAgICByZXF1ZXN0OiA1MTIgbWViaWJ5dGVzXG4gICAqICAgICAgbGltaXQ6IDIwNDggbWViaWJ5dGVzXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZXM/OiBDb250YWluZXJSZXNvdXJjZXM7XG5cbiAgLyoqXG4gICAqIFNlY3VyaXR5Q29udGV4dCBkZWZpbmVzIHRoZSBzZWN1cml0eSBvcHRpb25zIHRoZSBjb250YWluZXIgc2hvdWxkIGJlIHJ1biB3aXRoLlxuICAgKiBJZiBzZXQsIHRoZSBmaWVsZHMgb3ZlcnJpZGUgZXF1aXZhbGVudCBmaWVsZHMgb2YgdGhlIHBvZCdzIHNlY3VyaXR5IGNvbnRleHQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvdGFza3MvY29uZmlndXJlLXBvZC1jb250YWluZXIvc2VjdXJpdHktY29udGV4dC9cbiAgICogQGRlZmF1bHRcbiAgICpcbiAgICogICBlbnN1cmVOb25Sb290OiB0cnVlXG4gICAqICAgcHJpdmlsZWdlZDogZmFsc2VcbiAgICogICByZWFkT25seVJvb3RGaWxlc3lzdGVtOiB0cnVlXG4gICAqICAgYWxsb3dQcml2aWxlZ2VFc2NhbGF0aW9uOiBmYWxzZVxuICAgKiAgIHVzZXI6IDI1MDAwXG4gICAqICAgZ3JvdXA6IDI2MDAwXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUNvbnRleHQ/OiBDb250YWluZXJTZWN1cml0eUNvbnRleHRQcm9wcztcblxuICAvKipcbiAgICogS3ViZWxldCB3aWxsIHN0YXJ0IGluaXQgY29udGFpbmVycyB3aXRoIHJlc3RhcnRQb2xpY3k9QWx3YXlzIGluIHRoZSBvcmRlciB3aXRoIG90aGVyIGluaXQgY29udGFpbmVycyxcbiAgICogYnV0IGluc3RlYWQgb2Ygd2FpdGluZyBmb3IgaXRzIGNvbXBsZXRpb24sIGl0IHdpbGwgd2FpdCBmb3IgdGhlIGNvbnRhaW5lciBzdGFydHVwIGNvbXBsZXRpb25cbiAgICogQ3VycmVudGx5LCBvbmx5IGFjY2VwdGVkIHZhbHVlIGlzIEFsd2F5c1xuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3dvcmtsb2Fkcy9wb2RzL3NpZGVjYXItY29udGFpbmVycy9cbiAgICogQGRlZmF1bHQgLSBubyByZXN0YXJ0IHBvbGljeSBpcyBkZWZpbmVkIGFuZCB0aGUgcG9kIHJlc3RhcnQgcG9saWN5IGlzIGFwcGxpZWRcbiAgICovXG4gIHJlYWRvbmx5IHJlc3RhcnRQb2xpY3k/OiBDb250YWluZXJSZXN0YXJ0UG9saWN5O1xufVxuXG4vKipcbiAqIEEgc2luZ2xlIGFwcGxpY2F0aW9uIGNvbnRhaW5lciB0aGF0IHlvdSB3YW50IHRvIHJ1biB3aXRoaW4gYSBwb2QuXG4gKi9cbmV4cG9ydCBjbGFzcyBDb250YWluZXIge1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCBudW1iZXIgdGhhdCB3YXMgY29uZmlndXJlZCBmb3IgdGhpcyBjb250YWluZXIuXG4gICAqIElmIHVuZGVmaW5lZCwgZWl0aGVyIHRoZSBjb250YWluZXIgZG9lc24ndCBleHBvc2UgYSBwb3J0LCBvciBpdHNcbiAgICogcG9ydCBjb25maWd1cmF0aW9uIGlzIHN0b3JlZCBpbiB0aGUgYHBvcnRzYCBmaWVsZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwb3J0TnVtYmVyPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBWb2x1bWUgbW91bnRzIGNvbmZpZ3VyZWQgZm9yIHRoaXMgY29udGFpbmVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG1vdW50czogVm9sdW1lTW91bnRbXTtcblxuICAvKipcbiAgICogSW1hZ2UgcHVsbCBwb2xpY3kgZm9yIHRoaXMgY29udGFpbmVyXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaW1hZ2VQdWxsUG9saWN5OiBJbWFnZVB1bGxQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFRoZSBjb250YWluZXIgaW1hZ2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaW1hZ2U6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB3b3JraW5nIGRpcmVjdG9yeSBpbnNpZGUgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB3b3JraW5nRGlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDb21wdXRlIHJlc291cmNlcyAoQ1BVIGFuZCBtZW1vcnkgcmVxdWVzdHMgYW5kIGxpbWl0cykgcmVxdWlyZWQgYnkgdGhlIGNvbnRhaW5lclxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL2NvbmZpZ3VyYXRpb24vbWFuYWdlLXJlc291cmNlcy1jb250YWluZXJzL1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlc291cmNlcz86IENvbnRhaW5lclJlc291cmNlcztcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGNvbnRleHQgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzZWN1cml0eUNvbnRleHQ6IENvbnRhaW5lclNlY3VyaXR5Q29udGV4dDtcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IG9mIHRoZSBjb250YWluZXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW52OiBFbnY7XG5cbiAgLyoqXG4gICAqIFRoZSByZXN0YXJ0IHBvbGljeSBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlc3RhcnRQb2xpY3k/OiBDb250YWluZXJSZXN0YXJ0UG9saWN5O1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2NvbW1hbmQ/OiByZWFkb25seSBzdHJpbmdbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBfYXJncz86IHJlYWRvbmx5IHN0cmluZ1tdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9wb3J0czogQ29udGFpbmVyUG9ydFtdID0gW107XG4gIHByaXZhdGUgcmVhZG9ubHkgX3JlYWRpbmVzcz86IHByb2JlLlByb2JlO1xuICBwcml2YXRlIHJlYWRvbmx5IF9saXZlbmVzcz86IHByb2JlLlByb2JlO1xuICBwcml2YXRlIHJlYWRvbmx5IF9zdGFydHVwPzogcHJvYmUuUHJvYmU7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2xpZmVjeWNsZT86IENvbnRhaW5lckxpZmVjeWNsZTtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ29udGFpbmVyUHJvcHMpIHtcbiAgICBpZiAocHJvcHMgaW5zdGFuY2VvZiBDb250YWluZXIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXR0ZW1wdGVkIHRvIGNvbnN0cnVjdCBhIGNvbnRhaW5lciBmcm9tIGEgQ29udGFpbmVyIG9iamVjdC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBkZWZhdWx0UmVzb3VyY2VTcGVjOiBDb250YWluZXJSZXNvdXJjZXMgPSB7XG4gICAgICBjcHU6IHtcbiAgICAgICAgcmVxdWVzdDogQ3B1Lm1pbGxpcygxMDAwKSxcbiAgICAgICAgbGltaXQ6IENwdS5taWxsaXMoMTUwMCksXG4gICAgICB9LFxuICAgICAgbWVtb3J5OiB7XG4gICAgICAgIHJlcXVlc3Q6IFNpemUubWViaWJ5dGVzKDUxMiksXG4gICAgICAgIGxpbWl0OiBTaXplLm1lYmlieXRlcygyMDQ4KSxcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGlmIChwcm9wcy5wb3J0ICYmIHByb3BzLnBvcnROdW1iZXIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRWl0aGVyIFxcJ3BvcnRcXCcgb3IgXFwncG9ydE51bWJlclxcJyBpcyBhbGxvd2VkLiBVc2UgXFwncG9ydE51bWJlclxcJyBzaW5jZSBcXCdwb3J0XFwnIGlzIGRlcHJlY2F0ZWQnKTtcbiAgICB9XG5cbiAgICBjb25zdCBwb3J0TnVtYmVyID0gcHJvcHMucG9ydE51bWJlciA/PyBwcm9wcy5wb3J0O1xuICAgIGNvbnN0IGRlZmF1bHRQcm9iZUNvbmZpZ3VyYXRpb246IHByb2JlLlByb2JlID0gcHJvYmUuUHJvYmUuZnJvbVRjcFNvY2tldCh7IHBvcnQ6IHBvcnROdW1iZXIgfSk7XG5cbiAgICB0aGlzLm5hbWUgPSBwcm9wcy5uYW1lID8/ICdtYWluJztcbiAgICB0aGlzLmltYWdlID0gcHJvcHMuaW1hZ2U7XG4gICAgdGhpcy5wb3J0TnVtYmVyID0gcG9ydE51bWJlcjtcbiAgICB0aGlzLl9jb21tYW5kID0gcHJvcHMuY29tbWFuZDtcbiAgICB0aGlzLl9hcmdzID0gcHJvcHMuYXJncztcbiAgICB0aGlzLl9yZWFkaW5lc3MgPSBwcm9wcy5yZWFkaW5lc3M7XG4gICAgdGhpcy5fbGl2ZW5lc3MgPSBwcm9wcy5saXZlbmVzcztcbiAgICB0aGlzLl9zdGFydHVwID0gcHJvcHMuc3RhcnR1cCA/PyAodGhpcy5wb3J0TnVtYmVyID8gZGVmYXVsdFByb2JlQ29uZmlndXJhdGlvbiA6IHVuZGVmaW5lZCk7XG4gICAgdGhpcy5fbGlmZWN5Y2xlID0gcHJvcHMubGlmZWN5Y2xlO1xuICAgIHRoaXMucmVzb3VyY2VzID0gcHJvcHMucmVzb3VyY2VzID8/IGRlZmF1bHRSZXNvdXJjZVNwZWM7XG4gICAgdGhpcy53b3JraW5nRGlyID0gcHJvcHMud29ya2luZ0RpcjtcbiAgICB0aGlzLm1vdW50cyA9IHByb3BzLnZvbHVtZU1vdW50cyA/PyBbXTtcbiAgICB0aGlzLmltYWdlUHVsbFBvbGljeSA9IHByb3BzLmltYWdlUHVsbFBvbGljeSA/PyBJbWFnZVB1bGxQb2xpY3kuQUxXQVlTO1xuICAgIHRoaXMuc2VjdXJpdHlDb250ZXh0ID0gbmV3IENvbnRhaW5lclNlY3VyaXR5Q29udGV4dChwcm9wcy5zZWN1cml0eUNvbnRleHQpO1xuICAgIHRoaXMuZW52ID0gbmV3IEVudihwcm9wcy5lbnZGcm9tID8/IFtdLCBwcm9wcy5lbnZWYXJpYWJsZXMgPz8ge30pO1xuICAgIHRoaXMucmVzdGFydFBvbGljeSA9IHByb3BzLnJlc3RhcnRQb2xpY3k7XG5cbiAgICBpZiAodGhpcy5wb3J0TnVtYmVyKSB7XG4gICAgICB0aGlzLmFkZFBvcnQoe1xuICAgICAgICBudW1iZXI6IHRoaXMucG9ydE51bWJlcixcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgcG9ydCBvZiBwcm9wcy5wb3J0cyA/PyBbXSkge1xuICAgICAgdGhpcy5hZGRQb3J0KHBvcnQpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZCAtIHVzZSBgcG9ydE51bWJlcmAuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHBvcnQoKTogbnVtYmVyIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5wb3J0TnVtYmVyO1xuICB9XG5cbiAgLyoqXG4gICAqIFBvcnRzIGV4cG9zZWQgYnkgdGhpcyBjb250YWluZXJzLlxuICAgKiBSZXR1cm5zIGEgY29weSwgdXNlIGBhZGRQb3J0YCB0byBtb2RpZnkuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHBvcnRzKCk6IENvbnRhaW5lclBvcnRbXSB7XG4gICAgcmV0dXJuIFsuLi50aGlzLl9wb3J0c107XG4gIH1cblxuICAvKipcbiAgICogRW50cnlwb2ludCBhcnJheSAodGhlIGNvbW1hbmQgdG8gZXhlY3V0ZSB3aGVuIHRoZSBjb250YWluZXIgc3RhcnRzKS5cbiAgICogQHJldHVybnMgYSBjb3B5IG9mIHRoZSBlbnRyeXBvaW50IGFycmF5LCBjYW5ub3QgYmUgbW9kaWZpZWRcbiAgICovXG4gIHB1YmxpYyBnZXQgY29tbWFuZCgpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuX2NvbW1hbmQgPyBbLi4udGhpcy5fY29tbWFuZF0gOiB1bmRlZmluZWQ7XG4gIH1cblxuICAvKipcbiAgICogQXJndW1lbnRzIHRvIHRoZSBlbnRyeXBvaW50LlxuICAgKlxuICAgKiBAcmV0dXJucyBhIGNvcHkgb2YgdGhlIGFyZ3VtZW50cyBhcnJheSwgY2Fubm90IGJlIG1vZGlmaWVkLlxuICAgKi9cbiAgcHVibGljIGdldCBhcmdzKCk6IHN0cmluZ1tdIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fYXJncyA/IFsuLi50aGlzLl9hcmdzXSA6IHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBNb3VudCBhIHZvbHVtZSB0byBhIHNwZWNpZmljIHBhdGggc28gdGhhdCBpdCBpcyBhY2Nlc3NpYmxlIGJ5IHRoZSBjb250YWluZXIuXG4gICAqIEV2ZXJ5IHBvZCB0aGF0IGlzIGNvbmZpZ3VyZWQgdG8gdXNlIHRoaXMgY29udGFpbmVyIHdpbGwgYXV0bW9hdGljYWxseSBoYXZlIGFjY2VzcyB0byB0aGUgdm9sdW1lLlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCAtIFRoZSBkZXNpcmVkIHBhdGggaW4gdGhlIGNvbnRhaW5lci5cbiAgICogQHBhcmFtIHN0b3JhZ2UgLSBUaGUgc3RvcmFnZSB0byBtb3VudC5cbiAgICovXG4gIHB1YmxpYyBtb3VudChwYXRoOiBzdHJpbmcsIHN0b3JhZ2U6IHZvbHVtZS5JU3RvcmFnZSwgb3B0aW9uczogTW91bnRPcHRpb25zID0ge30pIHtcbiAgICB0aGlzLm1vdW50cy5wdXNoKHsgcGF0aCwgdm9sdW1lOiBzdG9yYWdlLmFzVm9sdW1lKCksIC4uLm9wdGlvbnMgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgcG9ydCB0byBleHBvc2UgZnJvbSB0aGlzIGNvbnRhaW5lci5cbiAgICovXG4gIHB1YmxpYyBhZGRQb3J0KHBvcnQ6IENvbnRhaW5lclBvcnQpIHtcblxuICAgIGNvbnN0IG5hbWVzID0gdGhpcy5fcG9ydHMubWFwKHAgPT4gcC5uYW1lKS5maWx0ZXIoeCA9PiB4KTtcbiAgICBjb25zdCBudW1iZXJQcm90b2NvbHMgPSB0aGlzLl9wb3J0cy5tYXAocCA9PiBgJHtwLm51bWJlcn0vJHtwLnByb3RvY29sIHx8IFByb3RvY29sLlRDUH1gKTtcblxuICAgIGlmIChwb3J0Lm5hbWUgJiYgbmFtZXMuaW5jbHVkZXMocG9ydC5uYW1lKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBQb3J0IHdpdGggbmFtZSAke3BvcnQubmFtZX0gYWxyZWFkeSBleGlzdHNgKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcm90b2NvbCA9IGAke3BvcnQubnVtYmVyfS8ke3BvcnQucHJvdG9jb2wgfHwgUHJvdG9jb2wuVENQfWA7XG4gICAgaWYgKG51bWJlclByb3RvY29scy5pbmNsdWRlcyhwcm90b2NvbCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgUG9ydCB3aXRoIG51bWJlciAke3BvcnQubnVtYmVyfSBhbmQgcHJvdG9jb2wgJHtwb3J0LnByb3RvY29sIHx8IFByb3RvY29sLlRDUH0gYWxyZWFkeSBleGlzdHNgKTtcbiAgICB9XG5cbiAgICB0aGlzLl9wb3J0cy5wdXNoKHBvcnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLkNvbnRhaW5lciB7XG4gICAgY29uc3Qgdm9sdW1lTW91bnRzOiBrOHMuVm9sdW1lTW91bnRbXSA9IFtdO1xuXG4gICAgZm9yIChjb25zdCBtb3VudCBvZiB0aGlzLm1vdW50cykge1xuICAgICAgdm9sdW1lTW91bnRzLnB1c2goe1xuICAgICAgICBuYW1lOiBtb3VudC52b2x1bWUubmFtZSxcbiAgICAgICAgbW91bnRQYXRoOiBtb3VudC5wYXRoLFxuICAgICAgICByZWFkT25seTogbW91bnQucmVhZE9ubHksXG4gICAgICAgIG1vdW50UHJvcGFnYXRpb246IG1vdW50LnByb3BhZ2F0aW9uLFxuICAgICAgICBzdWJQYXRoOiBtb3VudC5zdWJQYXRoLFxuICAgICAgICBzdWJQYXRoRXhwcjogbW91bnQuc3ViUGF0aEV4cHIsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBwb3J0cyA9IG5ldyBBcnJheTxrOHMuQ29udGFpbmVyUG9ydD4oKTtcblxuICAgIGZvciAoY29uc3QgcG9ydCBvZiB0aGlzLnBvcnRzKSB7XG4gICAgICBwb3J0cy5wdXNoKHtcbiAgICAgICAgY29udGFpbmVyUG9ydDogcG9ydC5udW1iZXIsXG4gICAgICAgIHByb3RvY29sOiBwb3J0LnByb3RvY29sPy50b1N0cmluZygpLFxuICAgICAgICBuYW1lOiBwb3J0Lm5hbWUsXG4gICAgICAgIGhvc3RQb3J0OiBwb3J0Lmhvc3RQb3J0LFxuICAgICAgICBob3N0SXA6IHBvcnQuaG9zdElwLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gUmVzb3VyY2UgcmVxdWVzdHMgYW5kIGxpbWl0c1xuICAgIGNvbnN0IGNwdUxpbWl0ID0gdGhpcy5yZXNvdXJjZXM/LmNwdT8ubGltaXQ/LmFtb3VudDtcbiAgICBjb25zdCBjcHVSZXF1ZXN0ID0gdGhpcy5yZXNvdXJjZXM/LmNwdT8ucmVxdWVzdD8uYW1vdW50O1xuICAgIGNvbnN0IG1lbW9yeUxpbWl0ID0gdGhpcy5yZXNvdXJjZXM/Lm1lbW9yeT8ubGltaXQ7XG4gICAgY29uc3QgbWVtb3J5UmVxdWVzdCA9IHRoaXMucmVzb3VyY2VzPy5tZW1vcnk/LnJlcXVlc3Q7XG4gICAgY29uc3QgZXBoZW1lcmFsU3RvcmFnZUxpbWl0ID0gdGhpcy5yZXNvdXJjZXM/LmVwaGVtZXJhbFN0b3JhZ2U/LmxpbWl0O1xuICAgIGNvbnN0IGVwaGVtZXJhbFN0b3JhZ2VSZXF1ZXN0ID0gdGhpcy5yZXNvdXJjZXM/LmVwaGVtZXJhbFN0b3JhZ2U/LnJlcXVlc3Q7XG5cbiAgICBjb25zdCBsaW1pdHM6IHsgW2tleTogc3RyaW5nXTogazhzLlF1YW50aXR5IH0gPSB7fTtcbiAgICBjb25zdCByZXF1ZXN0czogeyBba2V5OiBzdHJpbmddOiBrOHMuUXVhbnRpdHkgfSA9IHt9O1xuXG4gICAgaWYgKGNwdUxpbWl0KSB7XG4gICAgICBsaW1pdHMuY3B1ID0gazhzLlF1YW50aXR5LmZyb21TdHJpbmcoY3B1TGltaXQpO1xuICAgIH1cbiAgICBpZiAobWVtb3J5TGltaXQpIHtcbiAgICAgIGxpbWl0cy5tZW1vcnkgPSBrOHMuUXVhbnRpdHkuZnJvbVN0cmluZyhtZW1vcnlMaW1pdC50b01lYmlieXRlcygpLnRvU3RyaW5nKCkgKyAnTWknKTtcbiAgICB9XG4gICAgaWYgKGVwaGVtZXJhbFN0b3JhZ2VMaW1pdCkge1xuICAgICAgbGltaXRzWydlcGhlbWVyYWwtc3RvcmFnZSddID0gazhzLlF1YW50aXR5LmZyb21TdHJpbmcoZXBoZW1lcmFsU3RvcmFnZUxpbWl0LnRvR2liaWJ5dGVzKCkudG9TdHJpbmcoKSArICdHaScpO1xuICAgIH1cbiAgICBpZiAoY3B1UmVxdWVzdCkge1xuICAgICAgcmVxdWVzdHMuY3B1ID0gazhzLlF1YW50aXR5LmZyb21TdHJpbmcoY3B1UmVxdWVzdCk7XG4gICAgfVxuICAgIGlmIChtZW1vcnlSZXF1ZXN0KSB7XG4gICAgICByZXF1ZXN0cy5tZW1vcnkgPSBrOHMuUXVhbnRpdHkuZnJvbVN0cmluZyhtZW1vcnlSZXF1ZXN0LnRvTWViaWJ5dGVzKCkudG9TdHJpbmcoKSArICdNaScpO1xuICAgIH1cbiAgICBpZiAoZXBoZW1lcmFsU3RvcmFnZVJlcXVlc3QpIHtcbiAgICAgIHJlcXVlc3RzWydlcGhlbWVyYWwtc3RvcmFnZSddID0gazhzLlF1YW50aXR5LmZyb21TdHJpbmcoZXBoZW1lcmFsU3RvcmFnZVJlcXVlc3QudG9HaWJpYnl0ZXMoKS50b1N0cmluZygpICsgJ0dpJyk7XG4gICAgfVxuXG4gICAgbGV0IHJlc291cmNlUmVxdWlyZW1lbnRzOiBrOHMuUmVzb3VyY2VSZXF1aXJlbWVudHMgfCB1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgaWYgKE9iamVjdC5rZXlzKGxpbWl0cykubGVuZ3RoID4gMCB8fCBPYmplY3Qua2V5cyhyZXF1ZXN0cykubGVuZ3RoID4gMCkge1xuICAgICAgcmVzb3VyY2VSZXF1aXJlbWVudHMgPSB7XG4gICAgICAgIGxpbWl0czogdW5kZWZpbmVkSWZFbXB0eShsaW1pdHMpLFxuICAgICAgICByZXF1ZXN0czogdW5kZWZpbmVkSWZFbXB0eShyZXF1ZXN0cyksXG4gICAgICB9O1xuICAgIH1cblxuICAgIGNvbnN0IGVudiA9IHRoaXMuZW52Ll90b0t1YmUoKTtcblxuICAgIHJldHVybiB7XG4gICAgICBuYW1lOiB0aGlzLm5hbWUsXG4gICAgICBpbWFnZTogdGhpcy5pbWFnZSxcbiAgICAgIGltYWdlUHVsbFBvbGljeTogdGhpcy5pbWFnZVB1bGxQb2xpY3ksXG4gICAgICBwb3J0czogdW5kZWZpbmVkSWZFbXB0eShwb3J0cyksXG4gICAgICB2b2x1bWVNb3VudHM6IHVuZGVmaW5lZElmRW1wdHkodm9sdW1lTW91bnRzKSxcbiAgICAgIGNvbW1hbmQ6IHRoaXMuY29tbWFuZCxcbiAgICAgIGFyZ3M6IHRoaXMuYXJncyxcbiAgICAgIHdvcmtpbmdEaXI6IHRoaXMud29ya2luZ0RpcixcbiAgICAgIGVudjogZW52LnZhcmlhYmxlcyxcbiAgICAgIGVudkZyb206IGVudi5mcm9tLFxuICAgICAgcmVhZGluZXNzUHJvYmU6IHRoaXMuX3JlYWRpbmVzcz8uX3RvS3ViZSh0aGlzKSxcbiAgICAgIGxpdmVuZXNzUHJvYmU6IHRoaXMuX2xpdmVuZXNzPy5fdG9LdWJlKHRoaXMpLFxuICAgICAgc3RhcnR1cFByb2JlOiB0aGlzLl9zdGFydHVwPy5fdG9LdWJlKHRoaXMpLFxuICAgICAgbGlmZWN5Y2xlOiB0aGlzLl9saWZlY3ljbGUgPyB7XG4gICAgICAgIHBvc3RTdGFydDogdGhpcy5fbGlmZWN5Y2xlLnBvc3RTdGFydD8uX3RvS3ViZSh0aGlzKSxcbiAgICAgICAgcHJlU3RvcDogdGhpcy5fbGlmZWN5Y2xlLnByZVN0b3A/Ll90b0t1YmUodGhpcyksXG4gICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgcmVzb3VyY2VzOiByZXNvdXJjZVJlcXVpcmVtZW50cyxcbiAgICAgIHJlc3RhcnRQb2xpY3k6IHRoaXMucmVzdGFydFBvbGljeSxcbiAgICAgIHNlY3VyaXR5Q29udGV4dDogdGhpcy5zZWN1cml0eUNvbnRleHQuX3RvS3ViZSgpLFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBtb3VudHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTW91bnRPcHRpb25zIHtcbiAgLyoqXG4gICAqIERldGVybWluZXMgaG93IG1vdW50cyBhcmUgcHJvcGFnYXRlZCBmcm9tIHRoZSBob3N0IHRvIGNvbnRhaW5lciBhbmQgdGhlXG4gICAqIG90aGVyIHdheSBhcm91bmQuIFdoZW4gbm90IHNldCwgTW91bnRQcm9wYWdhdGlvbk5vbmUgaXMgdXNlZC5cbiAgICpcbiAgICogTW91bnQgcHJvcGFnYXRpb24gYWxsb3dzIGZvciBzaGFyaW5nIHZvbHVtZXMgbW91bnRlZCBieSBhIENvbnRhaW5lciB0b1xuICAgKiBvdGhlciBDb250YWluZXJzIGluIHRoZSBzYW1lIFBvZCwgb3IgZXZlbiB0byBvdGhlciBQb2RzIG9uIHRoZSBzYW1lIG5vZGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IE1vdW50UHJvcGFnYXRpb24uTk9ORVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvcGFnYXRpb24/OiBNb3VudFByb3BhZ2F0aW9uO1xuXG4gIC8qKlxuICAgKiBNb3VudGVkIHJlYWQtb25seSBpZiB0cnVlLCByZWFkLXdyaXRlIG90aGVyd2lzZSAoZmFsc2Ugb3IgdW5zcGVjaWZpZWQpLlxuICAgKiBEZWZhdWx0cyB0byBmYWxzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlYWRPbmx5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUGF0aCB3aXRoaW4gdGhlIHZvbHVtZSBmcm9tIHdoaWNoIHRoZSBjb250YWluZXIncyB2b2x1bWUgc2hvdWxkIGJlIG1vdW50ZWQuKS5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJcIiB0aGUgdm9sdW1lJ3Mgcm9vdFxuICAgKi9cbiAgcmVhZG9ubHkgc3ViUGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogRXhwYW5kZWQgcGF0aCB3aXRoaW4gdGhlIHZvbHVtZSBmcm9tIHdoaWNoIHRoZSBjb250YWluZXIncyB2b2x1bWUgc2hvdWxkIGJlXG4gICAqIG1vdW50ZWQuIEJlaGF2ZXMgc2ltaWxhcmx5IHRvIFN1YlBhdGggYnV0IGVudmlyb25tZW50IHZhcmlhYmxlIHJlZmVyZW5jZXNcbiAgICogJChWQVJfTkFNRSkgYXJlIGV4cGFuZGVkIHVzaW5nIHRoZSBjb250YWluZXIncyBlbnZpcm9ubWVudC4gRGVmYXVsdHMgdG8gXCJcIlxuICAgKiAodm9sdW1lJ3Mgcm9vdCkuXG4gICAqXG4gICAqIGBzdWJQYXRoRXhwcmAgYW5kIGBzdWJQYXRoYCBhcmUgbXV0dWFsbHkgZXhjbHVzaXZlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcIlwiIHZvbHVtZSdzIHJvb3QuXG4gICAqL1xuICByZWFkb25seSBzdWJQYXRoRXhwcj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBNb3VudCBhIHZvbHVtZSBmcm9tIHRoZSBwb2QgdG8gdGhlIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWb2x1bWVNb3VudCBleHRlbmRzIE1vdW50T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgdm9sdW1lIHRvIG1vdW50LlxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lOiB2b2x1bWUuVm9sdW1lO1xuXG4gIC8qKlxuICAgKiBQYXRoIHdpdGhpbiB0aGUgY29udGFpbmVyIGF0IHdoaWNoIHRoZSB2b2x1bWUgc2hvdWxkIGJlIG1vdW50ZWQuIE11c3Qgbm90XG4gICAqIGNvbnRhaW4gJzonLlxuICAgKi9cbiAgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xufVxuXG5leHBvcnQgZW51bSBNb3VudFByb3BhZ2F0aW9uIHtcbiAgLyoqXG4gICAqIFRoaXMgdm9sdW1lIG1vdW50IHdpbGwgbm90IHJlY2VpdmUgYW55IHN1YnNlcXVlbnQgbW91bnRzIHRoYXQgYXJlIG1vdW50ZWRcbiAgICogdG8gdGhpcyB2b2x1bWUgb3IgYW55IG9mIGl0cyBzdWJkaXJlY3RvcmllcyBieSB0aGUgaG9zdC4gSW4gc2ltaWxhclxuICAgKiBmYXNoaW9uLCBubyBtb3VudHMgY3JlYXRlZCBieSB0aGUgQ29udGFpbmVyIHdpbGwgYmUgdmlzaWJsZSBvbiB0aGUgaG9zdC5cbiAgICpcbiAgICogVGhpcyBpcyB0aGUgZGVmYXVsdCBtb2RlLlxuICAgKlxuICAgKiBUaGlzIG1vZGUgaXMgZXF1YWwgdG8gYHByaXZhdGVgIG1vdW50IHByb3BhZ2F0aW9uIGFzIGRlc2NyaWJlZCBpbiB0aGUgTGludXhcbiAgICoga2VybmVsIGRvY3VtZW50YXRpb25cbiAgICovXG4gIE5PTkUgPSAnTm9uZScsXG5cbiAgLyoqXG4gICAqIFRoaXMgdm9sdW1lIG1vdW50IHdpbGwgcmVjZWl2ZSBhbGwgc3Vic2VxdWVudCBtb3VudHMgdGhhdCBhcmUgbW91bnRlZCB0b1xuICAgKiB0aGlzIHZvbHVtZSBvciBhbnkgb2YgaXRzIHN1YmRpcmVjdG9yaWVzLlxuICAgKlxuICAgKiBJbiBvdGhlciB3b3JkcywgaWYgdGhlIGhvc3QgbW91bnRzIGFueXRoaW5nIGluc2lkZSB0aGUgdm9sdW1lIG1vdW50LCB0aGVcbiAgICogQ29udGFpbmVyIHdpbGwgc2VlIGl0IG1vdW50ZWQgdGhlcmUuXG4gICAqXG4gICAqIFNpbWlsYXJseSwgaWYgYW55IFBvZCB3aXRoIEJpZGlyZWN0aW9uYWwgbW91bnQgcHJvcGFnYXRpb24gdG8gdGhlIHNhbWVcbiAgICogdm9sdW1lIG1vdW50cyBhbnl0aGluZyB0aGVyZSwgdGhlIENvbnRhaW5lciB3aXRoIEhvc3RUb0NvbnRhaW5lciBtb3VudFxuICAgKiBwcm9wYWdhdGlvbiB3aWxsIHNlZSBpdC5cbiAgICpcbiAgICogVGhpcyBtb2RlIGlzIGVxdWFsIHRvIGByc2xhdmVgIG1vdW50IHByb3BhZ2F0aW9uIGFzIGRlc2NyaWJlZCBpbiB0aGUgTGludXhcbiAgICoga2VybmVsIGRvY3VtZW50YXRpb25cbiAgICovXG4gIEhPU1RfVE9fQ09OVEFJTkVSID0gJ0hvc3RUb0NvbnRhaW5lcicsXG5cbiAgLyoqXG4gICAqIFRoaXMgdm9sdW1lIG1vdW50IGJlaGF2ZXMgdGhlIHNhbWUgdGhlIEhvc3RUb0NvbnRhaW5lciBtb3VudC4gSW4gYWRkaXRpb24sXG4gICAqIGFsbCB2b2x1bWUgbW91bnRzIGNyZWF0ZWQgYnkgdGhlIENvbnRhaW5lciB3aWxsIGJlIHByb3BhZ2F0ZWQgYmFjayB0byB0aGVcbiAgICogaG9zdCBhbmQgdG8gYWxsIENvbnRhaW5lcnMgb2YgYWxsIFBvZHMgdGhhdCB1c2UgdGhlIHNhbWUgdm9sdW1lXG4gICAqXG4gICAqIEEgdHlwaWNhbCB1c2UgY2FzZSBmb3IgdGhpcyBtb2RlIGlzIGEgUG9kIHdpdGggYSBGbGV4Vm9sdW1lIG9yIENTSSBkcml2ZXJcbiAgICogb3IgYSBQb2QgdGhhdCBuZWVkcyB0byBtb3VudCBzb21ldGhpbmcgb24gdGhlIGhvc3QgdXNpbmcgYSBob3N0UGF0aCB2b2x1bWUuXG4gICAqXG4gICAqIFRoaXMgbW9kZSBpcyBlcXVhbCB0byBgcnNoYXJlZGAgbW91bnQgcHJvcGFnYXRpb24gYXMgZGVzY3JpYmVkIGluIHRoZSBMaW51eFxuICAgKiBrZXJuZWwgZG9jdW1lbnRhdGlvblxuICAgKlxuICAgKiBDYXV0aW9uOiBCaWRpcmVjdGlvbmFsIG1vdW50IHByb3BhZ2F0aW9uIGNhbiBiZSBkYW5nZXJvdXMuIEl0IGNhbiBkYW1hZ2VcbiAgICogdGhlIGhvc3Qgb3BlcmF0aW5nIHN5c3RlbSBhbmQgdGhlcmVmb3JlIGl0IGlzIGFsbG93ZWQgb25seSBpbiBwcml2aWxlZ2VkXG4gICAqIENvbnRhaW5lcnMuIEZhbWlsaWFyaXR5IHdpdGggTGludXgga2VybmVsIGJlaGF2aW9yIGlzIHN0cm9uZ2x5IHJlY29tbWVuZGVkLlxuICAgKiBJbiBhZGRpdGlvbiwgYW55IHZvbHVtZSBtb3VudHMgY3JlYXRlZCBieSBDb250YWluZXJzIGluIFBvZHMgbXVzdCBiZVxuICAgKiBkZXN0cm95ZWQgKHVubW91bnRlZCkgYnkgdGhlIENvbnRhaW5lcnMgb24gdGVybWluYXRpb24uXG4gICAqXG4gICAqL1xuICBCSURJUkVDVElPTkFMID0gJ0JpZGlyZWN0aW9uYWwnLFxufVxuXG4vKipcbiAqIENQVSBhbmQgbWVtb3J5IGNvbXB1dGUgcmVzb3VyY2VzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29udGFpbmVyUmVzb3VyY2VzIHtcbiAgcmVhZG9ubHkgY3B1PzogQ3B1UmVzb3VyY2VzO1xuICByZWFkb25seSBtZW1vcnk/OiBNZW1vcnlSZXNvdXJjZXM7XG4gIHJlYWRvbmx5IGVwaGVtZXJhbFN0b3JhZ2U/OiBFcGhlbWVyYWxTdG9yYWdlUmVzb3VyY2VzO1xufVxuXG4vKipcbiAqIENQVSByZXF1ZXN0IGFuZCBsaW1pdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENwdVJlc291cmNlcyB7XG4gIHJlYWRvbmx5IHJlcXVlc3Q/OiBDcHU7XG4gIHJlYWRvbmx5IGxpbWl0PzogQ3B1O1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgdGhlIGFtb3VudCBvZiBDUFUuXG4gKiBUaGUgYW1vdW50IGNhbiBiZSBwYXNzZWQgYXMgbWlsbGlzIG9yIHVuaXRzLlxuICovXG5leHBvcnQgY2xhc3MgQ3B1IHtcbiAgc3RhdGljIG1pbGxpcyhhbW91bnQ6IG51bWJlcik6IENwdSB7XG4gICAgcmV0dXJuIG5ldyBDcHUoYW1vdW50ICsgJ20nKTtcbiAgfVxuICBzdGF0aWMgdW5pdHMoYW1vdW50OiBudW1iZXIpOiBDcHUge1xuICAgIHJldHVybiBuZXcgQ3B1KGFtb3VudC50b1N0cmluZygpKTtcbiAgfVxuICBhbW91bnQ6IHN0cmluZztcbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihhbW91bnQ6IHN0cmluZykge1xuICAgIHRoaXMuYW1vdW50ID0gYW1vdW50O1xuICB9XG59XG5cbi8qKlxuICogTWVtb3J5IHJlcXVlc3QgYW5kIGxpbWl0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTWVtb3J5UmVzb3VyY2VzIHtcbiAgcmVhZG9ubHkgcmVxdWVzdD86IFNpemU7XG4gIHJlYWRvbmx5IGxpbWl0PzogU2l6ZTtcbn1cblxuLyoqXG4gKiBFbXBoZW1lcmFsIHN0b3JhZ2UgcmVxdWVzdCBhbmQgbGltaXRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFcGhlbWVyYWxTdG9yYWdlUmVzb3VyY2VzIHtcbiAgcmVhZG9ubHkgcmVxdWVzdD86IFNpemU7XG4gIHJlYWRvbmx5IGxpbWl0PzogU2l6ZTtcbn1cblxuLyoqXG4gKiBBIGNvbGxlY3Rpb24gb2YgZW52IHZhcmlhYmxlcyBkZWZpbmVkIGluIG90aGVyIHJlc291cmNlcy5cbiAqL1xuZXhwb3J0IGNsYXNzIEVudkZyb20ge1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHByaXZhdGUgcmVhZG9ubHkgY29uZmlnTWFwPzogY29uZmlnbWFwLklDb25maWdNYXAsXG4gICAgcHJpdmF0ZSByZWFkb25seSBwcmVmaXg/OiBzdHJpbmcsXG4gICAgcHJpdmF0ZSByZWFkb25seSBzZWM/OiBzZWNyZXQuSVNlY3JldCkgeyB9O1xuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5FbnZGcm9tU291cmNlIHtcbiAgICByZXR1cm4ge1xuICAgICAgY29uZmlnTWFwUmVmOiB0aGlzLmNvbmZpZ01hcCA/IHtcbiAgICAgICAgbmFtZTogdGhpcy5jb25maWdNYXAubmFtZSxcbiAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICBzZWNyZXRSZWY6IHRoaXMuc2VjID8ge1xuICAgICAgICBuYW1lOiB0aGlzLnNlYy5uYW1lLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHByZWZpeDogdGhpcy5wcmVmaXgsXG4gICAgfTtcbiAgfVxuXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBleHRyYWN0Q29udGFpbmVyUG9ydHMoc2VsZWN0b3I/OiBhbnkpOiBDb250YWluZXJQb3J0W10ge1xuXG4gIGlmICghc2VsZWN0b3IpIHsgcmV0dXJuIFtdOyB9XG5cbiAgLy8gd2UgZG9uJ3QgdXNlIGluc3RhbmNlb2YgaW50ZW50aW9uYWxseSBzaW5jZSBpdCBjYW4gY3JlYXRlXG4gIC8vIGN5Y2xpYyBpbXBvcnQgcHJvYmxlbXMuXG4gIGNvbnN0IGNvbnRhaW5lcnM6IENvbnRhaW5lcltdID0gKHNlbGVjdG9yIGFzIGFueSkuY29udGFpbmVycyA/PyBbXTtcblxuICByZXR1cm4gY29udGFpbmVycy5mbGF0TWFwKGMgPT4gYy5wb3J0cyk7XG59XG5cbi8qKlxuICogQ29udGFpbmVyIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAqL1xuZXhwb3J0IGNsYXNzIEVudiB7XG5cbiAgLyoqXG4gICAqIFNlbGVjdHMgYSBDb25maWdNYXAgdG8gcG9wdWxhdGUgdGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyB3aXRoLlxuICAgKiBUaGUgY29udGVudHMgb2YgdGhlIHRhcmdldCBDb25maWdNYXAncyBEYXRhIGZpZWxkIHdpbGwgcmVwcmVzZW50XG4gICAqIHRoZSBrZXktdmFsdWUgcGFpcnMgYXMgZW52aXJvbm1lbnQgdmFyaWFibGVzLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQ29uZmlnTWFwKGNvbmZpZ01hcDogY29uZmlnbWFwLklDb25maWdNYXAsIHByZWZpeD86IHN0cmluZyk6IEVudkZyb20ge1xuICAgIHJldHVybiBuZXcgRW52RnJvbShjb25maWdNYXAsIHByZWZpeCwgdW5kZWZpbmVkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZWxlY3RzIGEgU2VjcmV0IHRvIHBvcHVsYXRlIHRoZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgd2l0aC5cbiAgICogVGhlIGNvbnRlbnRzIG9mIHRoZSB0YXJnZXQgU2VjcmV0J3MgRGF0YSBmaWVsZCB3aWxsIHJlcHJlc2VudFxuICAgKiB0aGUga2V5LXZhbHVlIHBhaXJzIGFzIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVNlY3JldChzZWNyOiBzZWNyZXQuSVNlY3JldCk6IEVudkZyb20ge1xuICAgIHJldHVybiBuZXcgRW52RnJvbSh1bmRlZmluZWQsIHVuZGVmaW5lZCwgc2Vjcik7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IF9zb3VyY2VzOiBFbnZGcm9tW107XG4gIHByaXZhdGUgcmVhZG9ubHkgX3ZhcmlhYmxlczogeyBba2V5OiBzdHJpbmddOiBFbnZWYWx1ZSB9O1xuXG4gIHB1YmxpYyBjb25zdHJ1Y3Rvcihzb3VyY2VzOiBFbnZGcm9tW10sIHZhcmlhYmxlczogeyBbbmFtZTogc3RyaW5nXTogRW52VmFsdWUgfSkge1xuICAgIHRoaXMuX3NvdXJjZXMgPSBzb3VyY2VzO1xuICAgIHRoaXMuX3ZhcmlhYmxlcyA9IHZhcmlhYmxlcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBzaW5nbGUgdmFyaWFibGUgYnkgbmFtZSBhbmQgdmFsdWUuXG4gICAqIFRoZSB2YXJpYWJsZSB2YWx1ZSBjYW4gY29tZSBmcm9tIHZhcmlvdXMgZHluYW1pYyBzb3VyY2VzIHN1Y2ggYSBzZWNyZXRzIG9mIGNvbmZpZyBtYXBzLlxuICAgKiBVc2UgYEVudlZhbHVlLmZyb21YWFhgIHRvIHNlbGVjdCBzb3VyY2VzLlxuICAgKi9cbiAgcHVibGljIGFkZFZhcmlhYmxlKG5hbWU6IHN0cmluZywgdmFsdWU6IEVudlZhbHVlKSB7XG4gICAgdGhpcy5fdmFyaWFibGVzW25hbWVdID0gdmFsdWU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyBmb3IgdGhpcyBjb250YWluZXIuXG4gICAqIFJldHVybnMgYSBjb3B5LiBUbyBhZGQgZW52aXJvbm1lbnQgdmFyaWFibGVzIHVzZSBgY29udGFpbmVyLmVudi5hZGRWYXJpYWJsZSgpYC5cbiAgICovXG4gIHB1YmxpYyBnZXQgdmFyaWFibGVzKCk6IHsgW25hbWU6IHN0cmluZ106IEVudlZhbHVlIH0ge1xuICAgIHJldHVybiB7IC4uLnRoaXMuX3ZhcmlhYmxlcyB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGNvbGxlY3Rpb24gb2YgdmFyaWFibGVzIGJ5IGNvcHlpbmcgZnJvbSBhbm90aGVyIHNvdXJjZS5cbiAgICogVXNlIGBFbnYuZnJvbVhYWGAgZnVuY3Rpb25zIHRvIHNlbGVjdCBzb3VyY2VzLlxuICAgKi9cbiAgcHVibGljIGNvcHlGcm9tKGZyb206IEVudkZyb20pIHtcbiAgICB0aGlzLl9zb3VyY2VzLnB1c2goZnJvbSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2Ygc291cmNlcyB1c2VkIHRvIHBvcHVsYXRlIHRoZSBjb250YWluZXIgZW52aXJvbm1lbnQsXG4gICAqIGluIGFkZGl0aW9uIHRvIHRoZSBgdmFyaWFibGVzYC5cbiAgICpcbiAgICogUmV0dXJucyBhIGNvcHkuIFRvIGFkZCBhIHNvdXJjZSB1c2UgYGNvbnRhaW5lci5lbnYuY29weUZyb20oKWAuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHNvdXJjZXMoKTogRW52RnJvbVtdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX3NvdXJjZXNdO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJFbnYoZW52OiB7IFtuYW1lOiBzdHJpbmddOiBFbnZWYWx1ZSB9KTogazhzLkVudlZhcltdIHtcbiAgICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXk8azhzLkVudlZhcj4oKTtcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCB2XSBvZiBPYmplY3QuZW50cmllcyhlbnYpKSB7XG4gICAgICByZXN1bHQucHVzaCh7XG4gICAgICAgIG5hbWUsXG4gICAgICAgIHZhbHVlOiB2LnZhbHVlLFxuICAgICAgICB2YWx1ZUZyb206IHYudmFsdWVGcm9tLFxuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiB7IHZhcmlhYmxlcz86IGs4cy5FbnZWYXJbXTsgZnJvbT86IGs4cy5FbnZGcm9tU291cmNlW10gfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGZyb206IHVuZGVmaW5lZElmRW1wdHkodGhpcy5fc291cmNlcy5tYXAocyA9PiBzLl90b0t1YmUoKSkpLFxuICAgICAgdmFyaWFibGVzOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMucmVuZGVyRW52KHRoaXMuX3ZhcmlhYmxlcykpLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==