# Myst Python Library

This is the official Python client library for the Myst Platform.

## Requirements

- Python 3.7+

## Installation

To install the package from PyPI:

    $ pip install --upgrade myst-alpha

## Authentication

The Myst API uses JSON Web Tokens (JWTs) to authenticate requests.

The Myst Python library handles the sending of JWTs to the API automatically and currently supports two ways to
authenticate to obtain a JWT: through your Google user account or a Myst service account.

### Authenticating using your user account

If you don't yet have a Google account, you can create one on the
[Google Account Signup](https://accounts.google.com/signup) page.

Once you have access to a Google account, send an email to `support@myst.ai` with your email so we can authorize you to
use the Myst Platform.

Use the following code snippet to authenticate using your user account:

```python
import myst

myst.authenticate()
```

The first time you run this, you'll be presented with a web browser and asked to authorize the Myst Python library to
make requests on behalf of your Google user account. If you'd like to re-authorize (for example with a different 
account), pass `use_cache=False` to be presented with the web browser authorization once again.

### Authenticating using a service account

You can also authenticate using a Myst service account. To request a service account, email `support@myst.ai`.

To authenticate using a service account, set the `MYST_APPLICATION_CREDENTIALS` environment variable to the path to your
service account key file:

```sh
$ export MYST_APPLICATION_CREDENTIALS=</path/to/key/file.json>
```

Then, go through the service account authentication flow:

```python
import myst

myst.authenticate_with_service_account()
```

Alternatively, you can explicitly pass the path to your service account key:

```python
from pathlib import Path

import myst

myst.authenticate_with_service_account(key_file_path=Path("/path/to/key/file.json"))
```

## Connecting to a different environment

Contributors may want to connect to a non-production environment that they are authorized to develop in. In that case,
you can set the client with the API host you'd like to connect to.

```python
import myst

myst.set_client(myst.Client(api_host="https://petstore.api"))

myst.authenticate()
```

## Working with projects and graphs

A project is a workspace for setting up a graph of sources, models, operations, and time series to achieve a particular
objective. The sources, model, operations, and time series therein are _nodes_ of the graph, and they are connected by
different types of _edges_.

For more of a conceptual overview, see [the platform documentation](https://docs.myst.ai/docs). The following assumes
some familiarity with those concepts and focuses instead on demonstrating how to use the Myst client library to
interact with the platform.

### Projects

#### Create a project

```python
import myst

project = myst.Project.create(title="SF Electricity Load")
```

#### List projects

```python
import myst

projects = myst.Project.list()
```

#### Retrieve a project

```python
import myst

project = myst.Project.get(uuid="f89d7fbe-a051-4d0c-aa60-d6838b7e64a0")
```

### Nodes (Sources, Models, Operations, Time Series)

A node (source, model, operation, or time series) is always associated with a project, and there are multiple patterns
in the client library API by which you can list or create them.

#### Create a node

For example, suppose you want to create a temperature time series to be used as a feature in your model. Assuming that
you have the variable `project: Project` in scope, you can write the following to create a new time series:

```python
ksfo_temperature_time_series = project.create_time_series(
    title="Temperature (KSFO)",
    sample_period=myst.TimeDelta("PT1H"),  # Sample period of one hour. "PT1H" is an ISO 8601 duration string.
)
```

Or, to exactly the same effect:

```python
import myst

ksfo_temperature_time_series = myst.TimeSeries.create(
    project=project,  # Notice that project must be specified in this formulation.
    title="Temperature (KSFO)",
    sample_period=myst.TimeDelta("PT1H"),
)
```

This is true for the other types of nodes, too. In all, the client library offers the following equivalent ways to
create the different types of nodes:

- `project.create_source(...)` <=> `Source.create(project=project, ...)`
- `project.create_operation(...)` <=> `Operation.create(project=project, ...)`
- `project.create_model(...)` <=> `Model.create(project=project, ...)`
- `project.create_time_series(...)` <=> `TimeSeries.create(project=project, ...)`

#### Create a node with connector

For nodes that are powered by connectors, you must specify the parameters of that connector during construction. For
example, suppose you wanted to create a source node based on The Weather Company's Cleaned Observations API, to be used
as the source of temperature data in the time series we created above. To do so, you would write:

```python
from myst.connectors.source_connectors import cleaned_observations

ksfo_cleaned_observations = project.create_source(
    title="Cleaned Weather (KSFO)",
    connector=cleaned_observations.CleanedObservations(
        latitude=37.619,
        longitude=-122.374,
        fields=[
            cleaned_observations.Field.SURFACE_TEMPERATURE_CELSIUS,
        ],
    ),
)
```

`Model` and `Operation` nodes are constructed similarly. As another example, if we wanted to take the 3-hour rolling
mean of the temperature, we could create an operation as follows:

```python
import myst

from myst.connectors.operation_connectors import time_transformations

rolling_mean_ksfo_temperature = project.create_operation(
    title="Temperature (KSFO) - 3H Rolling Mean",
    connector=time_transformations.TimeTransformations(
        rolling_window_parameters=time_transformations.RollingWindowParameters(
            window_period=myst.TimeDelta("PT3H"),
            min_periods=1,
            centered=False,
            aggregation_function=time_transformations.AggregationFunction.MEAN,
        )
    ),
)
```

We will see how to connect an input to this operation [in a following step](#create-an-input).

#### List nodes in a project

```python
nodes = project.list_nodes()
```

#### Retrieve a node

```python
import myst

model = myst.Model.get("a5ba722c-6750-4796-8b43-230b5e0f4c4a")
```

Similar for `myst.Source.get`, `myst.Operation.get`, and `myst.TimeSeries.get`.

### Edges (Inputs, Layers)

#### Create a layer

A layer is an edge that feeds into a time series. You can create a layer into a time series with either:

```python
import myst
from myst.connectors.source_connectors import cleaned_observations

layer = ksfo_temperature_time_series.create_layer(
    upstream_node=ksfo_cleaned_observations,
    order=0,
    end_timing=myst.TimeDelta("-PT23H"),
    label_indexer=cleaned_observations.Field.SURFACE_TEMPERATURE_CELSIUS.value,
)
```

or:

```python
import myst
from myst.connectors.source_connectors import cleaned_observations

layer = myst.Layer.create(
    downstream_node=ksfo_temperature_time_series,
    upstream_node=ksfo_cleaned_observations,
    order=0,
    end_timing=myst.TimeDelta("-PT23H"),
    label_indexer=cleaned_observations.Field.SURFACE_TEMPERATURE_CELSIUS.value,
)
```

#### Create an input

An input is an edge that feeds into a model or an operation. To connect the temperature time series into the operation
we constructed before, we would write:

```python
from myst.connectors.operation_connectors import time_transformations

operation_input = rolling_mean_ksfo_temperature.create_input(
    upstream_node=ksfo_temperature_time_series,
    group_name=time_transformations.GroupName.OPERANDS,
)
```

As always, this creation method is also available as:

```python
import myst

from myst.connectors.operation_connectors import time_transformations


operation_input = myst.Input.create(
    downstream_node=rolling_mean_ksfo_temperature,
    upstream_node=ksfo_temperature_time_series,
    group_name=time_transformations.GroupName.OPERANDS,
)
```

#### List time series layers

```python
layers = ksfo_temperature_time_series.list_layers()
```

#### List model/operation inputs

```python
inputs = rolling_mean_ksfo_temperature.list_inputs()
```

## Working with time series

Time series are at the core of Myst's API. In addition to the functionality offered by a generic node, time series also
support querying and inserting data.

First, retrieve a time series:

```python
import myst

time_series = myst.TimeSeries.get("ca2a63d1-3515-47b4-afc7-13c6656dd744")
```

To insert a `TimeArray` of random scalar data into the time series:

```python
import myst
import numpy as np

time_array = myst.TimeArray(
    sample_period="PT1H",
    start_time="2021-07-01T00:00:00Z",
    end_time="2021-07-08T00:00:00Z",
    as_of_time="2021-07-01T00:00:00Z",
    values=np.random.randn(168),
)
time_series.insert_time_array(time_array=time_array)
```

You can also query a time series for a given as of time and natural time range. In this example, the query will return
the data we just inserted:

```python
import myst

returned_time_array = time_series.query_time_array(
    start_time=myst.Time("2021-07-01T00:00:00Z"),
    end_time=myst.Time("2021-07-08T00:00:00Z"),
    as_of_time=myst.Time("2021-07-01T00:00:00Z"),
)
assert returned_time_array == time_array
```

## Working with policies

A policy is the way to specify the schedule on which a particular target will be fit or run, as well as the parameters
around the target time range.

At the time of this writing, the Myst Platform supports two types of policies: _time series run policies_ and _model fit
policies_.

### Time series run policies

#### Create a time series run policy

```python
import myst

ksfo_temp_run_policy = ksfo_temperature_time_series.create_run_policy(
    schedule_timing=myst.TimeDelta("PT1H"),  # Run every hour.
    start_timing=myst.TimeDelta("PT1H"),  # Run on data starting from an hour from now (inclusive)...
    end_timing=myst.TimeDelta("P7D"),  # ...up to 7 days from now (exclusive).
)
```

### Model fit policies

Suppose we have a variable `xgboost_model` that contains a value of type `Model`.

#### Create a model fit policy

```python
import myst

xgboost_model_fit_policy = xgboost_model.create_fit_policy(
    schedule_timing=myst.TimeDelta("PT24H"),  # Run every 24 hours.
    start_timing=myst.Time("2021-01-01T00:00:00Z"),  # Fit on data from the beginning of 2021 (UTC)...
    end_timing=myst.TimeDelta("-PT1H"),  # ...up to an hour ago (exclusive).
)
```

## Deploying

In order for the graph to be executed, it must first be _deployed_. The Python client library does not currently support
this functionality; we recommend using the Myst Platform UI to deploy a project.

## Further Examples

For more full-featured usage examples of the Myst Platform Python client library, see the [`/examples`](/examples) folder.

## Support

For questions or just to say hi, reach out to `support@myst.ai`.
