# Copyright (C) 2019 Majormode.  All rights reserved.
#
# This software is the confidential and proprietary information of
# Majormode or one of its subsidiaries.  You shall not disclose this
# confidential information and shall use it only in accordance with the
# terms of the license agreement or other applicable agreement you
# entered into with Majormode.
#
# MAJORMODE MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
# OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
# TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE, OR NON-INFRINGEMENT.  MAJORMODE SHALL NOT BE LIABLE FOR ANY
# LOSSES OR DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
# OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

from typing import Any
import base64
import datetime
import hashlib
import hmac
import uuid

from majormode.perseus.constant.obj import ObjectStatus
from majormode.perseus.constant.sort_order import SortOrder
from majormode.perseus.constant.stage import EnvironmentStage
from majormode.perseus.model.app import ClientApplication
from majormode.perseus.model.enum import Enum
from majormode.perseus.model.geolocation import GeoPoint
from majormode.perseus.model.obj import Object
from majormode.perseus.model.version import Version
from majormode.perseus.service.account.account_service import AccountService
from majormode.perseus.service.application.application_service import ApplicationService
from majormode.perseus.service.base_rdbms_service import BaseRdbmsService
from majormode.perseus.service.base_rdbms_service import RdbmsConnection
from majormode.perseus.service.team.team_service import TeamService
from majormode.perseus.utils import cast
from majormode.perseus.utils.date_util import ISO8601DateTime
from majormode.xeberus.constant.tracker import BatteryPlugMode
from majormode.xeberus.constant.tracker import BatteryStateEventType
from majormode.xeberus.constant.tracker import SecurityLevel

import settings


class DeviceService(BaseRdbmsService):
    # Cryptographic function to hash sensitive information used to generate
    # device activation code.
    DEFAULT_CRYPTOGRAPHIC_HASH_FUNCTION = hashlib.sha1

    # Default time-to-live in seconds of a device activation code.
    #
    # A single activation code can be used to activate several ID-R devices
    # of a same organization in a row.  The longest the time-to-live, the
    # less secure the activation code (spoofing attack), but the more
    # user-friendly.
    DEFAULT_DEVICE_ACTIVATION_CODE_TTL = 30

    # Default number of characters that compose the nonce of the seed
    # generated by the device application.
    DEFAULT_SEED_NONCE_LENGTH = 8


    def __check_device_access(
            self,
            device,
            account_id=None,
            connection=None,
            team_id=None):
        """
        Check whether the specified user account is granted access to a
        device.

        The function verifies that either the user is the owner of the device,
        either a member of the organization managing this device.


        @param device: An instance representing a device.

        @param account_id: The identification of the account of a user who
            requests the access to the device.

        @param connection: An existing connection to the device database for
            read-only access.

        @param team_id: The identification of the organization that the user
            belongs to.  This organization must be responsible for the device.


        @raise IllegalAccessException: If the specified account doesn't belong
            to the individual user who owns this device, or if the specified
            organization is not managing for this device.
        """
        # If the device is registered on behalf of an organization, verify that
        # the specified user is a member of this organization.
        if device.team_id:
            if team_id != device.team_id:
                raise self.IllegalAccessException(f"The organization {team_id} is not managing for the device {device.device_id}")
            TeamService().assert_member(account_id, team_id, check_status=True, connection=connection)

        # If the device is registered on behalf of an individual user, verify
        # that the specified user is the owner of this device.
        elif account_id != device.account_id:
            AccountService.get_account(account_id, check_status=True, connection=connection)

    def __check_device_status(self, device: Any, strict_status=False):
        """
        Check that a device is enabled.


        @param device: An instance representing a device.

        @param strict_status: Indicate whether the device MUST be enabled (i.e.,
            activated), or whether its activation can be still pending.


        @raise DeleteObjectException: If the device has been banned by an
            administrator of the server platform.

        @raise DisabledObjectException: If the device has been suspended by
            the individual user who owns this device or an administrator of
            the organization that manages this device.

        @raise InvalidOperationException: If the activation of this device is
            pending.
        """
        if device.object_status == ObjectStatus.disabled:
            raise self.DisabledObjectException(f"The device {device.device_id} has been deactivated")
        elif device.object_status == ObjectStatus.deleted:
            raise self.DeletedObjectException(f"The device {device.device_id} has been suspended")
        elif device.object_status == ObjectStatus.pending and strict_status:
            raise self.InvalidOperationException(f"The device {device.device_id} has not been activated")

    def __get_device_apps(self, device_id, connection=None) -> list[Any]:
        """
        Return the list of client applications installed in a mobile device.


        @param device_id: The identification of a mobile device.

        @param connection: An existing connection to the device database.


        @return: A list of objects containing the following attributes:

            * `app_id: uuid.UUID`: The identification of the application.

            * `app_version: Version`: The version of the application installed in
              the mobile device.

            * `creation_time: ISO8601DateTime`: The time when the application has
              been registered as installed on the mobile device.

            * `object_status: ObjectStatus`: The current status of the application
              installed in this mobile device.

            * `picture_id: uuid.UUID`: The identification of the application's logo.

            * `picture_url: str`: The Uniform Resource Locator (URL) of the
              application's logo.

            * `update_time: ISO8601DateTime`: The time of the most recent
              modification of an attribute of the application installed in the
              mobile device (version, status, and/or logo).
        """
        with self.acquire_rdbms_connection(auto_commit=False, connection=connection) as connection:

            cursor = connection.execute(
                """
                SELECT
                    application.app_id,
                    application.app_name,
                    device_application.app_version,
                    device_application.creation_time,
                    device_application.object_status,
                    device_application.picture_id,
                    device_application.update_time
                  FROM
                    device_application
                  INNER JOIN application
                    USING (app_id)
                  WHERE
                    device_application.device_id = %(device_id)s
                """,
                {
                    'device_id': device_id,
                }
            )

            rows = cursor.fetch_all()
            apps = [
                row.get_object({
                    'app_id': cast.string_to_uuid,
                    'app_version': Version,
                    'creation_time': cast.string_to_timestamp,
                    'object_status': ObjectStatus,
                    'picture_id': cast.string_to_uuid,
                    'update_time': cast.string_to_timestamp,
                })
                for row in rows
            ]

            for app in apps:
                app.picture_url = ApplicationService().build_picture_url(app.app_id)

            return apps

    def __get_device_by_id(
            self,
            device_id: uuid.UUID,
            account_id: uuid.UUID = None,
            check_access: bool = False,
            check_status: bool = False,
            connection: RdbmsConnection = None,
            include_app_list: bool = False,
            include_system_info: bool = False,
            strict_status: bool = True,
            team_id: uuid.UUID = None) -> Any:
        """
        Return the information about a mobile device.


        @param device_id: The identification of a mobile device.

        @param account_id: The identification of the user on behalf of whom
            the function is called.  This user must either be the individual
            user who owns this mobile device, either a member of the
            organization that manages this device.

        @param check_access: Indicate whether to check the status of the
            mobile device.  The function raises an exception if the user and/
            or the organization on behalf of whom/which the function is called
            is not granted access to the mobile device.

        @param check_status: Indicate whether to check the status of the
            mobile device.  The function raises an exception if the device is
            not enabled.

        @param connection: An existing connection to the device database.

        @param include_app_list: Indicate whether to return information about
            the client applications installed in the mobile device.

        @param include_system_info: Indicate whether to return system
            information about the device, such as its serial number, its MAC
            address, its operating system.

        @param strict_status: Indicate whether the device MUST be enabled (i.e.,
            activated), or whether its activation can be still pending.

        @param team_id: The identification of the organization that the user
            (on behalf of whom the function is called) belongs to.


        @return: An object containing the following attributes:

            * `account_id: uuid.UUID` (optional): The identification of the
              individual user who owns this device, or the identification of the
              member of the organization on behalf of which the device has been
              activated.

            * `activation_time: ISO8601DateTime (required)': The time when the
              device has been activated.

            * `apps: list` (optional): The list of the client applications installed
              on this device.

            * `creation_time: ISO8601DateTime` (required): The time when the device
              has been registered to the server platform.

            * `mac_address: str` (optional): The Media Access Control (MAC) address
              of the mobile device.

            * `name: str` (optional): The human-readable name or a reference given
              to device.

            * `object_status: ObjectStatus` (required): The current status of the
              device:

              * `ObjectStatus.deleted`: The device has been banned by an
                administrator of the server platform.

              * `ObjectStatus.disabled`: The device has been suspended by the
                individual user who owns this device or an administrator of the
                organization that manages this device.

              * `ObjectStatus.enabled`: The device is activated and is expected to
                operate properly.

              * `ObjectStatus.pending`: The device has not been activated yet.

            * `os_name: str` (optional): The Name of the operating system of the
              mobile device.  This information is immutable.

            * `os_version: Version` (optional): The version of the operating system
              of the mobile device.  This information may be updated from time to
              time when the operating system of the device is upgraded.

            * `picture_id: uuid.UUID` (optional): The identification of the mobile
              device's picture.

            * `picture_url: str` (optional): The Uniform Resource Locator (URL) of
              the mobile device's picture.

            * `serial_number: str`: The hardware serial number of the mobile device.
              It corresponds to a unique number assigned by the manufacturer to help
              identify an individual device.

            * `team_id: uuid.UUID`: The identification of the organization that
             manages this device.

            * `update_time: ISO8601DateTime`: The time of the most recent
              modification of one or more attributes of this device.
        """
        with self.acquire_rdbms_connection(auto_commit=False, connection=connection) as connection:
            cursor = connection.execute(
                """
                SELECT
                    account_id,
                    activation_time,
                    creation_time,
                    mac_address,
                    name,
                    object_status,
                    os_name,
                    os_version,
                    picture_id,
                    serial_number,
                    team_id,
                    update_time
                  FROM
                    device
                  WHERE
                    device_id = %(device_id)s
                """,
                {
                    'device_id': device_id
                }
            )

            row = cursor.fetch_one()
            if row is None:
                raise self.UndefinedObjectException(f"No device registered with the identification {device_id}")

            device = row.get_object({
                'account_id': cast.string_to_uuid,
                'activation_time': cast.string_to_timestamp,
                'creation_time': cast.string_to_timestamp,
                'object_status': ObjectStatus,
                'os_version': Version,
                'picture_id': cast.string_to_uuid,
                'team_id': cast.string_to_uuid,
                'update_time': cast.string_to_timestamp,
            })

            # Check whether the user is granted access to the device.
            if check_access:
                self.__check_device_access(device, account_id=account_id, connection=connection, team_id=team_id)

            # Check whether the device is enabled.
            if check_status:
                self.__check_device_status(device, strict_status=strict_status)

            if include_app_list:
                device.apps = self.__get_device_apps(device_id)

            # Remove extended information if not requested.
            if not include_system_info:
                del device.activation_time
                del device.device_model
                del device.mac_address
                del device.os_name
                del device.os_version
                del device.serial_number

            return device

    def __get_device_by_mac_address(
            self,
            mac_address: str,
            connection: RdbmsConnection = None) -> Any:
        """
        Return the information of a mobile device specified with its MAC
        address.


        @param mac_address: The Media Access Control (MAC) address of the
            mobile device.

        @param connection: An existing connection to the device database for
            read-only access.


        @return: An object containing the following attributes:

            * `account_id: uuid.UUID` (optional): The identification of the account
              of the user who has activated the device.  This is either an
              individual user (the owner of the device), either an administrator of
              the organization that manages this device.

            * `device_id: uuid.UUID` (required): The identification of the device.

            * `object_status: ObjectStatus` (required): The current status of the
              device:

              * {@link ObjectStatus.deleted}: The device has been banned by an
                administrator of the server platform.

              * {@link ObjectStatus.disabled}: The device has been suspended by the
                individual user who has activated this device or an administrator of
                the organization that manages this device.

              * {@link ObjectStatus.enabled}: The device is activated and is expected
                to operate properly.

              * {@link ObjectStatus.pending}: The device has not been activated yet.

            * `team_id: uuid.UUID` (optional): The organization on behalf of which
              an administrator has activated the device.


        @raise UndefinedObjectException: If no device is registered with the
            specified MAC address.
        """
        with self.acquire_rdbms_connection(auto_commit=False, connection=connection) as connection:
            cursor = connection.execute(
                """
                SELECT
                    account_id,
                    device_id,
                    object_status,
                    team_id
                  FROM
                    device
                  WHERE
                    mac_address = %(mac_address)s
                """,
                {
                    'mac_address': mac_address,
                }
            )

            row = cursor.fetch_one()
            if not row:
                raise self.UndefinedObjectException(f"No device registered with the MAC address {mac_address}")

            device = row.get_object({
                'account_id': cast.string_to_uuid,
                'device_id': cast.string_to_uuid,
                'object_status': ObjectStatus,
                'team_id': cast.string_to_uuid,
            })

            return device

    def __insert_device(
            self,
            agent_application: ClientApplication,
            mac_address: str,
            serial_number,
            connection=None,
            location=None) -> Any:
        """
        Insert the information of a mobile device into the database.


        @param agent_application: The information about the client application
            and the device that initiates this call.

        @param mac_address: The Media Access Control (MAC) address of the
            mobile device.  A MAC address is a unique identifier assigned to a
            network interface for communications at the data link layer of a
            network segment.

        @param serial_number:  The hardware serial number of the mobile device.
            It corresponds to a unique number assigned by the manufacturer to
            help identify an individual device.

            Because the operating system of the mobile device may restrict
            access to persistent device identifiers, the serial number of the
            mobile device may not be provided.

            Serial number is case-sensitive.

        @param connection: An existing connection to the device database, with
            auto-commit option enabled.

        @param location: The last known location of the mobile device.


        @return: An object containing the following attributes:

            * `device_id: uuid.UUID`: The identification of the mobile device.

            * `creation_time: ISO8601DateTime`: The time when the device has been
              registered to the device database.

            * `object_status: ObjectStatus`: The current status of the mobile
              device ({@link ObjectStatus.pending}).
        """
        with self.acquire_rdbms_connection(auto_commit=True, connection=connection) as connection:
            cursor = connection.execute(
                """
                INSERT INTO device (
                    accuracy,
                    bearing,
                    device_model,
                    fix_time,
                    location,
                    mac_address,
                    object_status,
                    os_name,
                    os_version,
                    provider,
                    serial_number,
                    speed
                  )
                  VALUES (
                    %(accuracy)s,
                    %(bearing)s,
                    %(device_model)s,
                    %(fix_time)s,
                    ST_SetSRID(ST_MakePoint(%(longitude)s, %(latitude)s, %(altitude)s), 4326),
                    %(mac_address)s,
                    %(OBJECT_STATUS_PENDING)s,
                    %(os_name)s,
                    %(os_version)s,
                    %(product_name)s,
                    %(product_version)s,
                    %(provider)s,
                    %(serial_number)s,
                    %(speed)s
                  )
                  RETURNING
                    device_id
                """,
                {
                    'OBJECT_STATUS_PENDING': ObjectStatus.pending,
                    'accuracy': location and location.accuracy,
                    'altitude': location and location.altitude,
                    'bearing': location and location.bearing,
                    'device_model': agent_application.device_model,
                    'fix_time': location and location.fix_time,
                    'latitude': location and location.latitude,
                    'longitude': location and location.longitude,
                    'mac_address': mac_address,
                    'os_name': agent_application.os_name,
                    'os_version': str(agent_application.os_version),
                    'product_name': agent_application.product_name,
                    'product_version': str(agent_application.product_version),
                    'provider': location and location.provider,
                    'serial_number': serial_number,
                    'speed': location and location.speed,
                }
            )

            row = cursor.fetch_one()
            device_id = row.get_value('device_id', cast.string_to_uuid)

            return device_id

    def __insert_device_application(
            self,
            device_id: uuid.UUID,
            app_id: uuid.UUID,
            agent_application: ClientApplication,
            connection: RdbmsConnection = None):
        """
        Insert into the device database the information about a client
        application installed on a mobile device.


        @param device_id: The identification of the mobile device which the
            client application is installed on.

        @param app_id: The identification of the client application.

        @param agent_application: The information about the client application
            and the device that initiates this call.

        @param connection: An existing connection to the device database, with
            auto-commit option enabled.
        """
        with self.acquire_rdbms_connection(auto_commit=True, connection=connection) as connection:
            connection.execute(
                """
                INSERT INTO device_application (
                    app_id,
                    app_version,
                    device_id,
                    object_status
                  )
                  VALUES (
                    %(app_id)s,
                    %(app_version)s,
                    %(device_id)s,
                    %(OBJECT_STATUS_PENDING)s
                  )
                """,
                {
                    'OBJECT_STATUS_PENDING': ObjectStatus.pending,
                    'app_id': app_id,
                    'app_version': agent_application.product_version,
                    'device_id': device_id,
                }
            )

    def __register_device(
            self,
            app_id: uuid.UUID,
            agent_application: ClientApplication,
            ip_address: str,
            mac_address: str,
            serial_number: str,
            connection: RdbmsConnection = None,
            location: GeoPoint = None):
        """
        Register a device that shakes hands for the first time.

        An individual user or an administrator of the organization officially
        responsible for managing this device will need to activate this device.


        @param app_id: The identification of the client application that
            initiates this call.

        @param agent_application: The information about the client application
            and the device that initiates this call.

        @param ip_address: A dotted-decimal notation of an IPv4 address,
            consisting of four decimal numbers, each ranging from ``0`` to
            ``255``, separated by dots.

        @param mac_address: The Media Access Control (MAC) address of the
            mobile device.  A MAC address is a unique identifier assigned to a
            network interface for communications at the data link layer of a
            network segment.

        @param serial_number:  The hardware serial number of the mobile device.
            It corresponds to a unique number assigned by the manufacturer to
            help identify an individual device.

            Because the operating system of the mobile device may restrict
            access to persistent device identifiers, the serial number of the
            mobile device may not be provided.

            Serial number is case-sensitive.

        @param connection: An existing connection to the device database, with
            auto-commit option enabled.

        @param location: The last known location of the mobile device.


        @return: An object containing the following attributes:

            * `device_id: uuid.UUID`: The identification of the mobile device.

            * `creation_time: ISO8601DateTime`: The time when the device has been
              registered to the server platform.
        """
        with self.acquire_rdbms_connection(auto_commit=True, connection=connection) as connection:
            device = self.__insert_device(
                agent_application,
                mac_address,
                serial_number,
                connection=connection, location=location)

            self.__insert_device_application(
                device.device_id,
                app_id,
                agent_application,
                connection=connection)

            return device

    def __update_device_application(
            self,
            device_id: uuid.UUID,
            app_id: uuid.UUID,
            app_version: Version,
            connection: RdbmsConnection = None):
        pass

    def __update_device(
            self,
            mac_address,
            is_battery_present: bool,
            is_battery_plugged: bool,
            battery_level: float,
            agent_application: ClientApplication,
            connection: RdbmsConnection = None,
            device_id: uuid.UUID = None):
        pass

    def shake_hands(
            self,
            app_id: uuid.UUID,
            agent_application: ClientApplication,
            ip_address: str,
            mac_address: str,
            serial_number: str,
            connection: RdbmsConnection = None,
            location: GeoPoint = None):
        """
        Establish the connection between a device and the server platform
        before normal communication begins.

        If the device is not registered, the function registers the device
        with its system information and the version of the information about
        the client application that initiates this call.  The device's status
        remains pending until an individual user or an administrator of an
        organization activates the device.

If the device was already registered, but the client application that
initiates this call is not registered yet, the function registers the
information about the client application.  If the application is the
sole application registered with the device, the function
automatically activates this application.  If other applications are
registered with the device, the application's status remains pending
until the individual user or an administrator of the organization
owning this device activates the application.


        @param app_id: The identification of the client application that
            initiates this call.

        @param agent_application: The information about the client application
            and the device that initiates this call.

        @param ip_address: A dotted-decimal notation of an IPv4 address,
            consisting of four decimal numbers, each ranging from ``0`` to
            ``255``, separated by dots.

        @param mac_address: The Media Access Control (MAC) address of the
            mobile device.  A MAC address is a unique identifier assigned to a
            network interface for communications at the data link layer of a
            network segment.

        @param serial_number:  The hardware serial number of the mobile device.
            It corresponds to a unique number assigned by the manufacturer to
            help identify an individual device.

            Because the operating system of the mobile device may restrict
            access to persistent device identifiers, the serial number of the
            mobile device may not be provided.

            Serial number is case-sensitive.

        @param connection: An existing connection to the device database, with
            auto-commit option enabled.

        @param location: The last known location of the mobile device.


        @return: An object containing the following attributes:

* `account_id: uuid.UUID` (optional): The identification of the account
  of the user who has activated the device.  This is either an
  individual user (the owner of the device), either an administrator of
  the organization that manages this device.

* `device_id: uuid.UUID` (required): The identification of the device.

* `object_status: ObjectStatus` (required): The current status of the
  device:

  * {@link ObjectStatus.deleted}: The device has been banned by an
    administrator of the server platform.

  * {@link ObjectStatus.disabled}: The device has been suspended by the
    individual user who has activated this device or an administrator of
    the organization that manages this device.

  * {@link ObjectStatus.enabled}: The device is activated and is expected
    to operate properly.

  * {@link ObjectStatus.pending}: The device has not been activated yet.

* `team_id: uuid.UUID` (optional): The organization on behalf of which
  an administrator has activated the device.
        """
        try:
            device = self.__get_device_by_mac_address(mac_address, connection=connection)



        except self.UndefinedObjectException:
            device = self.__register_device(
                app_id,
                agent_application,
                ip_address,
                mac_address,
                serial_number,
                connection=connection,
                location=location)


        return device

# Check that the registered mac address corresponds the mac address
# that is provided, and that the device identification corresponds to
# the one that has been registered.
#
# The provided mac address may not have been provided

        # Upsert the device
        #    Upsert the application installed in the device


    def verify_seed(self, device, app, seed):
        """
        Verify that a seed has been generated by the specified device.

        A seed is composed of two parts concatenated together:

        - A randomly number, also known as a nonce.

        - The nonce encrypted with the security key shared between the server
          platform and the application, and converted to a BASE64 string.


        @param device A device.

        @param app The application installed in the device that is sending the
            seed.

        @param seed A seed, which is a "number used once" (nonce) -- also
            known as pseudo-random number -- encrypted with a security key
            generated by the server platform and shared with the device.  A
            seed is used to verify the authentication of the device in order
            to prevent spoofing attack.


        @raise IllegalAccessException If the given seed has not been
            successfully verified, meaning that someone or a program tries to
            masquerade as the specified device (spoofing attack).
        """
        encryption_key = f'{device.mac_address}{app.security_key}'

        # Retrieve the nonce and the encrypted version of the nonce.
        nonce = seed[0:self.DEFAULT_SEED_NONCE_LENGTH]
        encrypted_nonce = seed[self.DEFAULT_SEED_NONCE_LENGTH:]

        # Check that the nonce has been correctly encrypted with the security
        # key of the client application installed on the device.
        hashed_message = hmac.new(
            encryption_key.encode(),
            msg=nonce.encode(),
            digestmod=self.DEFAULT_CRYPTOGRAPHIC_HASH_FUNCTION)

        base64encoded_digest = base64 \
           .b64encode(hashed_message.digest()) \
           .decode()[:-1]  # Remove final "=" character

        if encrypted_nonce != base64encoded_digest:
            self.log_warning(f"Wrong encrypted seed!")
            if settings.ENVIRONMENT_STAGE != EnvironmentStage.prod:
                self.log_warning(f"\tExpected: {base64encoded_digest}")
                self.log_warning(f"\tReceived: {encrypted_nonce}")
            raise self.IllegalAccessException("Wrong seed")