"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const connections_1 = require("./connections");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway() {
        return new NatGatewayProvider();
    }
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props) {
        return new NatInstanceProvider(props);
    }
}
exports.NatProvider = NatProvider;
/**
 * Provider for NAT Gateways
 */
class NatGatewayProvider extends NatProvider {
    constructor() {
        super(...arguments);
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        // Create the NAT gateways
        for (const sub of options.natSubnets) {
            const gateway = sub.addNatGateway();
            this.gateways.add(sub.availabilityZone, gateway.ref);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
/**
 * NAT provider which uses NAT Instances
 */
class NatInstanceProvider extends NatProvider {
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        var _a, _b;
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        this._securityGroup = (_a = this.props.securityGroup) !== null && _a !== void 0 ? _a : new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
        });
        this._connections = new connections_1.Connections({ securityGroups: [this._securityGroup] });
        if ((_b = this.props.allowAllTraffic) !== null && _b !== void 0 ? _b : true) {
            this.connections.allowFromAnyIpv4(port_1.Port.allTraffic());
        }
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: this.props.keyName,
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    /**
     * The Security Group associated with the NAT instances
     */
    get securityGroup() {
        if (!this._securityGroup) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'securityGroup\'');
        }
        return this._securityGroup;
    }
    /**
     * Manage the Security Groups associated with the NAT instances
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'connections\'');
        }
        return this._connections;
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
}
exports.NatInstanceProvider = NatInstanceProvider;
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * Machine image representing the latest NAT instance image
 *
 * @experimental
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
//# sourceMappingURL=data:application/json;base64,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