"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const tmp = require("tmp");
const fs = require("fs");
class Seeder extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.setup || !Array.isArray(props.setup))
            throw new Error('setup value must be an array of JSON objects');
        this.props = props;
        const destinationBucket = new aws_s3_1.Bucket(this, 'acds-bucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            this.writeTempFile(dir, 'setup.json', props.setup);
            if (props.teardown) {
                this.writeTempFile(dir, 'teardown.json', props.teardown);
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false,
            });
        });
        const fn = new aws_lambda_1.Function(this, 'handler', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.fromInline(`
console.log('function loaded');

const AWS = require('aws-sdk');
const s3 = new AWS.S3();

const writeTypeFromAction = (action) => {
  if (action === "put")
    return "Item";
  if (action === "delete")
    return "Key";
}

const run = async (filename, action) => {
  console.log('reading from s3');
  const data = await s3.getObject({
    Bucket: "${destinationBucket.bucketName}", 
    Key: filename
  }).promise();
  console.log('finished reading from s3');
  
  console.log('transforming seed data');
  const seed = JSON.parse(data.Body.toString());
  console.log('finished transforming seed data');
  
  const documentClient = new AWS.DynamoDB.DocumentClient();
  console.log('sending data to dynamodb');
  for(let i = 0; i < seed.length;i++) {
    await documentClient[action]({
      TableName: '${props.table.tableName}',
      [writeTypeFromAction(action)]: seed[i]
    }).promise();
  };
  console.log('finished sending data to dynamodb');
}

exports.handler = async (event) => {
  if (event.mode === "delete" || event.mode === "update")
    await run("teardown.json", "delete");
  if (event.mode === "create" || event.mode === "update")
    await run("setup.json", "put");
}`),
        });
        destinationBucket.grantRead(fn);
        props.table.grantWriteData(fn);
        const onEvent = new custom_resources_1.AwsCustomResource(this, 'on-event', {
            onCreate: {
                ...this.callLambdaOptions(),
                parameters: {
                    FunctionName: fn.functionArn,
                    InvokeArgs: JSON.stringify({
                        mode: 'create',
                    }),
                },
            },
            onDelete: props.teardown
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'delete',
                        }),
                    },
                }
                : undefined,
            onUpdate: props.refreshOnUpdate
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'update',
                        }),
                    },
                }
                : undefined,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        fn.grantInvoke(onEvent);
    }
    callLambdaOptions() {
        return {
            service: 'Lambda',
            action: 'invokeAsync',
            apiVersion: '2015-03-31',
            physicalResourceId: {
                id: `${this.props.table.tableArn}-seeder`,
            },
        };
    }
    writeTempFile(dir, filename, data) {
        const buffer = Buffer.from(JSON.stringify(data));
        const filepath = dir + '/' + filename;
        fs.writeFileSync(filepath, buffer);
    }
}
exports.Seeder = Seeder;
//# sourceMappingURL=data:application/json;base64,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