# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['hera']

package_data = \
{'': ['*']}

install_requires = \
['argo-workflows==6.3.5',
 'certifi>=2021.10.8',
 'python-dateutil>=2.8.2',
 'pytz>=2021.3',
 'urllib3>=1.26.8']

extras_require = \
{'yaml': ['pyyaml>=6.0']}

setup_kwargs = {
    'name': 'hera-workflows',
    'version': '4.1.0',
    'description': 'Hera is a Python framework for constructing and submitting Argo Workflows. The main goal of Hera is to make Argo Workflows more accessible by abstracting away some setup that is typically necessary for constructing Argo workflows.',
    'long_description': '# Hera (hera-workflows)\n\n```text\nThe Argo was constructed by the shipwright Argus,\nand its crew were specially protected by the goddess Hera.\n```\n(https://en.wikipedia.org/wiki/Argo)\n\n\n[![Open in Gitpod](https://gitpod.io/button/open-in-gitpod.svg)](https://gitpod.io/#https://github.com/argoproj-labs/hera-workflows)\n\n[![Build](https://github.com/argoproj-labs/hera-workflows/actions/workflows/cicd.yaml/badge.svg)](https://github.com/argoproj-labs/hera-workflows/blob/main/.github/workflows/cicd.yaml)\n[![Docs](https://readthedocs.org/projects/hera-workflows/badge/?version=latest)](https://hera-workflows.readthedocs.io/en/latest/?badge=latest)\n[![codecov](https://codecov.io/gh/argoproj-labs/hera-workflows/branch/main/graph/badge.svg?token=x4tvsQRKXP)](https://codecov.io/gh/argoproj-labs/hera-workflows)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n\n[![Pypi](https://img.shields.io/pypi/v/hera-workflows.svg)](https://pypi.python.org/pypi/hera-workflows)\n[![CondaForge](https://anaconda.org/conda-forge/hera-workflows/badges/version.svg)](https://anaconda.org/conda-forge/hera-workflows)\n[![Versions](https://img.shields.io/pypi/pyversions/hera-workflows.svg)](https://github.com/argoproj-labs/hera-workflows)\n\n[![Downloads](https://pepy.tech/badge/hera-workflows)](https://pepy.tech/project/hera-workflows)\n[![Downloads/month](https://pepy.tech/badge/hera-workflows/month)](https://pepy.tech/project/hera-workflows)\n[![Downloads/week](https://pepy.tech/badge/hera-workflows/week)](https://pepy.tech/project/hera-workflows)\n\n\nHera is a Python framework for constructing and submitting Argo Workflows. The main goal of Hera is to make the Argo\necosystem accessible by simplifying workflow construction and submission.\n\nYou can watch the introductory Hera presentation at the "Argo Workflows and Events Community Meeting 20 Oct\n2021" [here](https://www.youtube.com/watch?v=QETfzfVV-GY&t=181s)!\n\n# Table of content\n\n- [Installation](#installation)\n- [Examples](#examples)\n- [Requirements](#requirements)\n- [Contributing](#contributing)\n- [Comparison](#comparison)\n\n# Requirements\n\nHera requires an Argo server to be deployed to a Kubernetes cluster. Currently, Hera assumes that the Argo server sits\nbehind an authentication layer that can authenticate workflow submission requests by using the Bearer token on the\nrequest. To learn how to deploy Argo to your own Kubernetes cluster you can follow the\n[Argo Workflows](https://argoproj.github.io/argo-workflows/quick-start/) guide!\n\nAnother option for workflow submission without the authentication layer is using port forwarding to your Argo server\ndeployment and submitting workflows to `localhost:2746` (2746 is the default, but you are free to use yours). Please\nrefer to the documentation of [Argo Workflows](https://argoproj.github.io/argo-workflows/quick-start/) to see the\ncommand for port forward!\n\n# Installation\n\n| Source                                                         | Command                                                                                                        |\n|----------------------------------------------------------------|----------------------------------------------------------------------------------------------------------------|\n| [PyPi](https://pypi.org/project/hera-workflows/)               | `pip install hera-workflows`                                                                                   |\n| [Conda](https://anaconda.org/conda-forge/hera-workflows)       | `conda install -c conda-forge hera-workflows`                                                                  |\n| [GitHub repo](https://github.com/argoproj-labs/hera-workflows) | `python -m pip install git+https://github.com/argoproj-labs/hera-workflows --ignore-installed`/`pip install .` |\n\n# Examples\n\n```python\nfrom hera import Task, Workflow\n\n\ndef say(message: str):\n    print(message)\n\n\nwith Workflow("diamond") as w:\n    a = Task(\'a\', say, [\'This is task A!\'])\n    b = Task(\'b\', say, [\'This is task B!\'])\n    c = Task(\'c\', say, [\'This is task C!\'])\n    d = Task(\'d\', say, [\'This is task D!\'])\n\n    a >> [b, c] >> d\n\nw.create()\n```\n\nSee the [examples](https://github.com/argoproj-labs/hera-workflows/tree/main/examples) directory for a collection of\nArgo workflow construction and submission via Hera!\n\n# Contributing\n\nIf you plan to submit contributions to Hera you can install Hera in a virtual environment managed by `poetry`:\n\n```shell\npoetry install\n```\n\nIn your activated `poetry shell`, you can utilize the tasks found in `tox.ini`, e.g.:\n\nTo run tests on all supported python versions with coverage run [tox](https://tox.wiki/en/latest/):\n\n```shell\ntox\n```\n\nTo list all available `tox` envs run:\n\n```shell\ntox -a\n```\n\nTo run selected tox envs, e.g. for a specific python version with coverage run:\n\n```shell\ntox -e py37,coverage\n```\n\nAs `coverage` *depends* on `py37`, it will run *after* `py37`\n\nSee project `tox.ini` for more details\n\nAlso, see the [contributing guide](https://github.com/argoproj-labs/hera-workflows/blob/main/CONTRIBUTING.md)!\n\n# Comparison\n\nThere are other libraries currently available for structuring and submitting Argo Workflows:\n\n- [Couler](https://github.com/couler-proj/couler), which aims to provide a unified interface for constructing and\n  managing workflows on different workflow engines;\n- [Argo Python DSL](https://github.com/argoproj-labs/argo-python-dsl), which allows you to programmaticaly define Argo\n  worfklows using Python.\n\nWhile the aforementioned libraries provide amazing functionality for Argo workflow construction and submission, they\nrequire an advanced understanding of Argo concepts. When [Dyno Therapeutics](https://dynotx.com) started using Argo\nWorkflows, it was challenging to construct and submit experimental machine learning workflows. Scientists and engineers\nat [Dyno Therapeutics](https://dynotx.com) used a lot of time for workflow definition rather than the implementation of\nthe atomic unit of execution - the Python function - that performed, for instance, model training.\n\nHera presents a much simpler interface for task and workflow construction, empowering users to focus on their own\nexecutable payloads rather than workflow setup. Here\'s a side by side comparison of Hera, Argo Python DSL, and Couler:\n\n<table>\n<tr><th>Hera</th><th>Couler</th><th>Argo Python DSL</th></tr>\n<tr>\n\n<td valign="top"><p>\n\n```python\nfrom hera import Task, Workflow\n\n\ndef say(message: str):\n    print(message)\n\n\nwith Workflow("diamond") as w:\n    a = Task(\'a\', say, [\'This is task A!\'])\n    b = Task(\'b\', say, [\'This is task B!\'])\n    c = Task(\'c\', say, [\'This is task C!\'])\n    d = Task(\'d\', say, [\'This is task D!\'])\n\n    a >> [b, c] >> d\n\nw.create()\n```\n\n</p></td>\n\n<td valign="top"><p>\n\n```python\nimport couler.argo as couler\nfrom couler.argo_submitter import ArgoSubmitter\n\n\ndef job(name):\n    couler.run_container(\n        image="docker/whalesay:latest",\n        command=["cowsay"],\n        args=[name],\n        step_name=name,\n    )\n\n\ndef diamond():\n    couler.dag(\n        [\n            [lambda: job(name="A")],\n            [lambda: job(name="A"), lambda: job(name="B")],  # A -> B\n            [lambda: job(name="A"), lambda: job(name="C")],  # A -> C\n            [lambda: job(name="B"), lambda: job(name="D")],  # B -> D\n            [lambda: job(name="C"), lambda: job(name="D")],  # C -> D\n        ]\n    )\n\n\ndiamond()\nsubmitter = ArgoSubmitter()\ncouler.run(submitter=submitter)\n```\n\n</p></td>\n\n<td valign="top"><p>\n\n```python\nfrom argo.workflows.dsl import Workflow\n\nfrom argo.workflows.dsl.tasks import *\nfrom argo.workflows.dsl.templates import *\n\n\nclass DagDiamond(Workflow):\n\n    @task\n    @parameter(name="message", value="A")\n    def A(self, message: V1alpha1Parameter) -> V1alpha1Template:\n        return self.echo(message=message)\n\n    @task\n    @parameter(name="message", value="B")\n    @dependencies(["A"])\n    def B(self, message: V1alpha1Parameter) -> V1alpha1Template:\n        return self.echo(message=message)\n\n    @task\n    @parameter(name="message", value="C")\n    @dependencies(["A"])\n    def C(self, message: V1alpha1Parameter) -> V1alpha1Template:\n        return self.echo(message=message)\n\n    @task\n    @parameter(name="message", value="D")\n    @dependencies(["B", "C"])\n    def D(self, message: V1alpha1Parameter) -> V1alpha1Template:\n        return self.echo(message=message)\n\n    @template\n    @inputs.parameter(name="message")\n    def echo(self, message: V1alpha1Parameter) -> V1Container:\n        container = V1Container(\n            image="alpine:3.7",\n            name="echo",\n            command=["echo", "{{inputs.parameters.message}}"],\n        )\n\n        return container\n```\n\n</p></td>\n</tr>\n</table>\n',
    'author': 'Flaviu Vadan',
    'author_email': 'flaviu.vadan@dynotx.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/argoproj-labs/hera-workflows',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
