"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/cdk");
const lambda = require("../lib");
/**
 * Use a class as test fixture
 *
 * setUp() will be mapped to the (synchronous) constructor. tearDown(cb) will be called if available.
 */
function classFixture(klass) {
    let fixture;
    const ret = {
        setUp(cb) {
            fixture = new klass();
            cb();
        },
        tearDown(cb) {
            if (fixture.tearDown) {
                fixture.tearDown(cb);
            }
            else {
                cb();
            }
        }
    };
    const testNames = Reflect.ownKeys(klass.prototype).filter(m => m !== 'tearDown' && m !== 'constructor');
    for (const testName of testNames) {
        ret[testName] = (test) => fixture[testName](test);
    }
    return ret;
}
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'lambda in a VPC': classFixture(class Henk {
        constructor() {
            // GIVEN
            this.app = new cdk.App();
            this.stack = new cdk.Stack(this.app, 'stack');
            this.vpc = new ec2.Vpc(this.stack, 'VPC');
            // WHEN
            this.lambda = new lambda.Function(this.stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NodeJS810,
                vpc: this.vpc,
                allowAllOutbound: false
            });
        }
        'has subnet and securitygroup'(test) {
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] }
                    ],
                    SubnetIds: [
                        { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                        { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
                        { Ref: "VPCPrivateSubnet3Subnet3EDCD457" }
                    ]
                }
            }));
            test.done();
        }
        'participates in Connections objects'(test) {
            // GIVEN
            const securityGroup = new ec2.SecurityGroup(this.stack, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            this.lambda.connections.allowTo(somethingConnectable, new ec2.TcpAllPorts(), 'Lambda can call connectable');
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(this.stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
                GroupId: { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] },
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                DestinationSecurityGroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                FromPort: 0,
                ToPort: 65535
            }));
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(this.stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                FromPort: 0,
                GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                SourceSecurityGroupId: { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] },
                ToPort: 65535
            }));
            test.done();
        }
        'can still make Connections after export/import'(test) {
            // GIVEN
            const stack2 = new cdk.Stack(this.app, 'stack2');
            const securityGroup = new ec2.SecurityGroup(stack2, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            somethingConnectable.connections.allowFrom(this.lambda.connections, new ec2.TcpAllPorts(), 'Lambda can call connectable');
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(stack2).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
                GroupId: {
                    "Fn::ImportValue": "stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1"
                },
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                DestinationSecurityGroupId: {
                    "Fn::GetAtt": [
                        "SomeSecurityGroupEF219AD6",
                        "GroupId"
                    ]
                },
                FromPort: 0,
                ToPort: 65535
            }));
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(stack2).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                FromPort: 0,
                GroupId: {
                    "Fn::GetAtt": [
                        "SomeSecurityGroupEF219AD6",
                        "GroupId"
                    ]
                },
                SourceSecurityGroupId: {
                    "Fn::ImportValue": "stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1"
                },
                ToPort: 65535
            }));
            test.done();
        }
    }),
    'lambda without VPC throws Error upon accessing connections'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const lambdaFn = new lambda.Function(stack, 'Lambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        // WHEN
        test.throws(() => {
            lambdaFn.connections.allowToAnyIPv4(new ec2.TcpAllPorts(), 'Reach for the world Lambda!');
        });
        test.done();
    },
    'picking public subnets is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NodeJS810,
                vpc,
                vpcSubnets: { subnetType: ec2.SubnetType.Public }
            });
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC52cGMtbGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC52cGMtbGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBdUQ7QUFDdkQsd0NBQXlDO0FBQ3pDLG9DQUFxQztBQUVyQyxpQ0FBa0M7QUE4SmxDOzs7O0dBSUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxLQUFVO0lBQzlCLElBQUksT0FBWSxDQUFDO0lBRWpCLE1BQU0sR0FBRyxHQUFRO1FBQ2YsS0FBSyxDQUFDLEVBQXFCO1lBQ3pCLE9BQU8sR0FBRyxJQUFJLEtBQUssRUFBRSxDQUFDO1lBQ3RCLEVBQUUsRUFBRSxDQUFDO1FBQ1AsQ0FBQztRQUVELFFBQVEsQ0FBQyxFQUFxQjtZQUM1QixJQUFJLE9BQU8sQ0FBQyxRQUFRLEVBQUU7Z0JBQ3BCLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDdEI7aUJBQU07Z0JBQ0wsRUFBRSxFQUFFLENBQUM7YUFDTjtRQUNILENBQUM7S0FDRixDQUFDO0lBRUYsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLFVBQVUsSUFBSSxDQUFDLEtBQUssYUFBYSxDQUFDLENBQUM7SUFDeEcsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLEVBQUU7UUFDaEMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBVSxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDekQ7SUFFRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFFRCxNQUFNLG9CQUFvQjtJQUN4QixZQUE0QixXQUE0QjtRQUE1QixnQkFBVyxHQUFYLFdBQVcsQ0FBaUI7SUFDeEQsQ0FBQztDQUNGO0FBOUxELGlCQUFTO0lBQ1AsaUJBQWlCLEVBQUUsWUFBWSxDQUFDLE1BQU0sSUFBSTtRQU14QztZQUNFLFFBQVE7WUFDUixJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ3pCLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDOUMsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztZQUUxQyxPQUFPO1lBQ1AsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3RELElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO2dCQUNsQyxPQUFPLEVBQUUsZUFBZTtnQkFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztnQkFDakMsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLGdCQUFnQixFQUFFLEtBQUs7YUFDeEIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVNLDhCQUE4QixDQUFDLElBQVU7WUFDOUMsT0FBTztZQUNQLGVBQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7Z0JBQzFELFNBQVMsRUFBRTtvQkFDVCxnQkFBZ0IsRUFBRTt3QkFDaEIsRUFBQyxZQUFZLEVBQUUsQ0FBRSw2QkFBNkIsRUFBRSxTQUFTLENBQUUsRUFBQztxQkFDN0Q7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULEVBQUMsR0FBRyxFQUFFLGlDQUFpQyxFQUFDO3dCQUN4QyxFQUFDLEdBQUcsRUFBRSxpQ0FBaUMsRUFBQzt3QkFDeEMsRUFBQyxHQUFHLEVBQUUsaUNBQWlDLEVBQUM7cUJBQ3pDO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRU0scUNBQXFDLENBQUMsSUFBVTtZQUNyRCxRQUFRO1lBQ1IsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUUsRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDaEcsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLG9CQUFvQixDQUFDLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBRWhILE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsb0JBQW9CLEVBQUUsSUFBSSxHQUFHLENBQUMsV0FBVyxFQUFFLEVBQUUsNkJBQTZCLENBQUMsQ0FBQztZQUU1RyxnREFBZ0Q7WUFDaEQsZUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQywrQkFBK0IsRUFBRTtnQkFDbEUsT0FBTyxFQUFFLEVBQUMsWUFBWSxFQUFFLENBQUMsNkJBQTZCLEVBQUUsU0FBUyxDQUFDLEVBQUM7Z0JBQ25FLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixXQUFXLEVBQUUsNkJBQTZCO2dCQUMxQywwQkFBMEIsRUFBRSxFQUFDLFlBQVksRUFBRSxDQUFFLDJCQUEyQixFQUFFLFNBQVMsQ0FBRSxFQUFDO2dCQUN0RixRQUFRLEVBQUUsQ0FBQztnQkFDWCxNQUFNLEVBQUUsS0FBSzthQUNkLENBQUMsQ0FBQyxDQUFDO1lBRUosMERBQTBEO1lBQzFELGVBQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0NBQWdDLEVBQUU7Z0JBQ25FLFVBQVUsRUFBRSxLQUFLO2dCQUNqQixXQUFXLEVBQUUsNkJBQTZCO2dCQUMxQyxRQUFRLEVBQUUsQ0FBQztnQkFDWCxPQUFPLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQywyQkFBMkIsRUFBRSxTQUFTLENBQUMsRUFBRTtnQkFDbkUscUJBQXFCLEVBQUUsRUFBQyxZQUFZLEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxTQUFTLENBQUUsRUFBQztnQkFDbEYsTUFBTSxFQUFFLEtBQUs7YUFDZCxDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFTSxnREFBZ0QsQ0FBQyxJQUFVO1lBQ2hFLFFBQVE7WUFDUixNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUNqRCxNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLG1CQUFtQixFQUFFLEVBQUUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBQzVGLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjLEVBQUUsQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUVoSCxPQUFPO1lBQ1Asb0JBQW9CLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRSxJQUFJLEdBQUcsQ0FBQyxXQUFXLEVBQUUsRUFBRSw2QkFBNkIsQ0FBQyxDQUFDO1lBRTFILDBEQUEwRDtZQUMxRCxlQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsK0JBQStCLEVBQUU7Z0JBQzlELE9BQU8sRUFBRTtvQkFDUCxpQkFBaUIsRUFBRSx1RUFBdUU7aUJBQzNGO2dCQUNELFVBQVUsRUFBRSxLQUFLO2dCQUNqQixXQUFXLEVBQUUsNkJBQTZCO2dCQUMxQywwQkFBMEIsRUFBRTtvQkFDMUIsWUFBWSxFQUFFO3dCQUNaLDJCQUEyQjt3QkFDM0IsU0FBUztxQkFDVjtpQkFDRjtnQkFDRCxRQUFRLEVBQUUsQ0FBQztnQkFDWCxNQUFNLEVBQUUsS0FBSzthQUNkLENBQUMsQ0FBQyxDQUFDO1lBRUosZ0RBQWdEO1lBQ2hELGVBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDL0QsVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLFdBQVcsRUFBRSw2QkFBNkI7Z0JBQzFDLFFBQVEsRUFBRSxDQUFDO2dCQUNYLE9BQU8sRUFBRTtvQkFDUCxZQUFZLEVBQUU7d0JBQ1osMkJBQTJCO3dCQUMzQixTQUFTO3FCQUNWO2lCQUNGO2dCQUNELHFCQUFxQixFQUFFO29CQUNyQixpQkFBaUIsRUFBRSx1RUFBdUU7aUJBQzNGO2dCQUNELE1BQU0sRUFBRSxLQUFLO2FBQ2QsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0YsQ0FBQztJQUVGLDREQUE0RCxDQUFDLElBQVU7UUFDckUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sUUFBUSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ3BELElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsUUFBUSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsSUFBSSxHQUFHLENBQUMsV0FBVyxFQUFFLEVBQUUsNkJBQTZCLENBQUMsQ0FBQztRQUM1RixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1Q0FBdUMsQ0FBQyxJQUFVO1FBQ2hELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXRDLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNuQyxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztnQkFDbEMsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7Z0JBQ2pDLEdBQUc7Z0JBQ0gsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFO2FBQ2xELENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgZWMyID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWVjMicpO1xuaW1wb3J0IGNkayA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2NkaycpO1xuaW1wb3J0IHsgSUNhbGxiYWNrRnVuY3Rpb24sIFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgbGFtYmRhID0gcmVxdWlyZSgnLi4vbGliJyk7XG5cbmV4cG9ydCA9IHtcbiAgJ2xhbWJkYSBpbiBhIFZQQyc6IGNsYXNzRml4dHVyZShjbGFzcyBIZW5rIHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGFwcDogY2RrLkFwcDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHN0YWNrOiBjZGsuU3RhY2s7XG4gICAgcHJpdmF0ZSByZWFkb25seSB2cGM6IGVjMi5WcGM7XG4gICAgcHJpdmF0ZSByZWFkb25seSBsYW1iZGE6IGxhbWJkYS5GdW5jdGlvbjtcblxuICAgIGNvbnN0cnVjdG9yKCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIHRoaXMuYXBwID0gbmV3IGNkay5BcHAoKTtcbiAgICAgIHRoaXMuc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHRoaXMuYXBwLCAnc3RhY2snKTtcbiAgICAgIHRoaXMudnBjID0gbmV3IGVjMi5WcGModGhpcy5zdGFjaywgJ1ZQQycpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB0aGlzLmxhbWJkYSA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcy5zdGFjaywgJ0xhbWJkYScsIHtcbiAgICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlM4MTAsXG4gICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBwdWJsaWMgJ2hhcyBzdWJuZXQgYW5kIHNlY3VyaXR5Z3JvdXAnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdCh0aGlzLnN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICAgICAgVnBjQ29uZmlnOiB7XG4gICAgICAgICAgU2VjdXJpdHlHcm91cElkczogW1xuICAgICAgICAgICAge1wiRm46OkdldEF0dFwiOiBbIFwiTGFtYmRhU2VjdXJpdHlHcm91cEU3NDY1OUExXCIsIFwiR3JvdXBJZFwiIF19XG4gICAgICAgICAgXSxcbiAgICAgICAgICBTdWJuZXRJZHM6IFtcbiAgICAgICAgICAgIHtSZWY6IFwiVlBDUHJpdmF0ZVN1Ym5ldDFTdWJuZXQ4QkNBMTBFMFwifSxcbiAgICAgICAgICAgIHtSZWY6IFwiVlBDUHJpdmF0ZVN1Ym5ldDJTdWJuZXRDRkNEQUE3QVwifSxcbiAgICAgICAgICAgIHtSZWY6IFwiVlBDUHJpdmF0ZVN1Ym5ldDNTdWJuZXQzRURDRDQ1N1wifVxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG5cbiAgICBwdWJsaWMgJ3BhcnRpY2lwYXRlcyBpbiBDb25uZWN0aW9ucyBvYmplY3RzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc2VjdXJpdHlHcm91cCA9IG5ldyBlYzIuU2VjdXJpdHlHcm91cCh0aGlzLnN0YWNrLCAnU29tZVNlY3VyaXR5R3JvdXAnLCB7IHZwYzogdGhpcy52cGMgfSk7XG4gICAgICBjb25zdCBzb21ldGhpbmdDb25uZWN0YWJsZSA9IG5ldyBTb21ldGhpbmdDb25uZWN0YWJsZShuZXcgZWMyLkNvbm5lY3Rpb25zKHsgc2VjdXJpdHlHcm91cHM6IFtzZWN1cml0eUdyb3VwXSB9KSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHRoaXMubGFtYmRhLmNvbm5lY3Rpb25zLmFsbG93VG8oc29tZXRoaW5nQ29ubmVjdGFibGUsIG5ldyBlYzIuVGNwQWxsUG9ydHMoKSwgJ0xhbWJkYSBjYW4gY2FsbCBjb25uZWN0YWJsZScpO1xuXG4gICAgICAvLyBUSEVOOiBMYW1iZGEgY2FuIGNvbm5lY3QgdG8gU29tZVNlY3VyaXR5R3JvdXBcbiAgICAgIGV4cGVjdCh0aGlzLnN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEVncmVzc1wiLCB7XG4gICAgICAgIEdyb3VwSWQ6IHtcIkZuOjpHZXRBdHRcIjogW1wiTGFtYmRhU2VjdXJpdHlHcm91cEU3NDY1OUExXCIsIFwiR3JvdXBJZFwiXX0sXG4gICAgICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgICAgIERlc2NyaXB0aW9uOiBcIkxhbWJkYSBjYW4gY2FsbCBjb25uZWN0YWJsZVwiLFxuICAgICAgICBEZXN0aW5hdGlvblNlY3VyaXR5R3JvdXBJZDoge1wiRm46OkdldEF0dFwiOiBbIFwiU29tZVNlY3VyaXR5R3JvdXBFRjIxOUFENlwiLCBcIkdyb3VwSWRcIiBdfSxcbiAgICAgICAgRnJvbVBvcnQ6IDAsXG4gICAgICAgIFRvUG9ydDogNjU1MzVcbiAgICAgIH0pKTtcblxuICAgICAgLy8gVEhFTjogU29tZVNlY3VyaXR5R3JvdXAgYWNjZXB0cyBjb25uZWN0aW9ucyBmcm9tIExhbWJkYVxuICAgICAgZXhwZWN0KHRoaXMuc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzc1wiLCB7XG4gICAgICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgICAgIERlc2NyaXB0aW9uOiBcIkxhbWJkYSBjYW4gY2FsbCBjb25uZWN0YWJsZVwiLFxuICAgICAgICBGcm9tUG9ydDogMCxcbiAgICAgICAgR3JvdXBJZDogeyBcIkZuOjpHZXRBdHRcIjogW1wiU29tZVNlY3VyaXR5R3JvdXBFRjIxOUFENlwiLCBcIkdyb3VwSWRcIl0gfSxcbiAgICAgICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XCJGbjo6R2V0QXR0XCI6IFtcIkxhbWJkYVNlY3VyaXR5R3JvdXBFNzQ2NTlBMVwiLCBcIkdyb3VwSWRcIiBdfSxcbiAgICAgICAgVG9Qb3J0OiA2NTUzNVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG5cbiAgICBwdWJsaWMgJ2NhbiBzdGlsbCBtYWtlIENvbm5lY3Rpb25zIGFmdGVyIGV4cG9ydC9pbXBvcnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjazIgPSBuZXcgY2RrLlN0YWNrKHRoaXMuYXBwLCAnc3RhY2syJyk7XG4gICAgICBjb25zdCBzZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHN0YWNrMiwgJ1NvbWVTZWN1cml0eUdyb3VwJywgeyB2cGM6IHRoaXMudnBjIH0pO1xuICAgICAgY29uc3Qgc29tZXRoaW5nQ29ubmVjdGFibGUgPSBuZXcgU29tZXRoaW5nQ29ubmVjdGFibGUobmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzOiBbc2VjdXJpdHlHcm91cF0gfSkpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBzb21ldGhpbmdDb25uZWN0YWJsZS5jb25uZWN0aW9ucy5hbGxvd0Zyb20odGhpcy5sYW1iZGEuY29ubmVjdGlvbnMsIG5ldyBlYzIuVGNwQWxsUG9ydHMoKSwgJ0xhbWJkYSBjYW4gY2FsbCBjb25uZWN0YWJsZScpO1xuXG4gICAgICAvLyBUSEVOOiBTb21lU2VjdXJpdHlHcm91cCBhY2NlcHRzIGNvbm5lY3Rpb25zIGZyb20gTGFtYmRhXG4gICAgICBleHBlY3Qoc3RhY2syKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEVncmVzc1wiLCB7XG4gICAgICAgIEdyb3VwSWQ6IHtcbiAgICAgICAgICBcIkZuOjpJbXBvcnRWYWx1ZVwiOiBcInN0YWNrOkV4cG9ydHNPdXRwdXRGbkdldEF0dExhbWJkYVNlY3VyaXR5R3JvdXBFNzQ2NTlBMUdyb3VwSWQ4RjNFQzZGMVwiXG4gICAgICAgIH0sXG4gICAgICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgICAgIERlc2NyaXB0aW9uOiBcIkxhbWJkYSBjYW4gY2FsbCBjb25uZWN0YWJsZVwiLFxuICAgICAgICBEZXN0aW5hdGlvblNlY3VyaXR5R3JvdXBJZDoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlNvbWVTZWN1cml0eUdyb3VwRUYyMTlBRDZcIixcbiAgICAgICAgICAgIFwiR3JvdXBJZFwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBGcm9tUG9ydDogMCxcbiAgICAgICAgVG9Qb3J0OiA2NTUzNVxuICAgICAgfSkpO1xuXG4gICAgICAvLyBUSEVOOiBMYW1iZGEgY2FuIGNvbm5lY3QgdG8gU29tZVNlY3VyaXR5R3JvdXBcbiAgICAgIGV4cGVjdChzdGFjazIpLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6RUMyOjpTZWN1cml0eUdyb3VwSW5ncmVzc1wiLCB7XG4gICAgICAgIElwUHJvdG9jb2w6IFwidGNwXCIsXG4gICAgICAgIERlc2NyaXB0aW9uOiBcIkxhbWJkYSBjYW4gY2FsbCBjb25uZWN0YWJsZVwiLFxuICAgICAgICBGcm9tUG9ydDogMCxcbiAgICAgICAgR3JvdXBJZDoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlNvbWVTZWN1cml0eUdyb3VwRUYyMTlBRDZcIixcbiAgICAgICAgICAgIFwiR3JvdXBJZFwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICAgICBcIkZuOjpJbXBvcnRWYWx1ZVwiOiBcInN0YWNrOkV4cG9ydHNPdXRwdXRGbkdldEF0dExhbWJkYVNlY3VyaXR5R3JvdXBFNzQ2NTlBMUdyb3VwSWQ4RjNFQzZGMVwiXG4gICAgICAgIH0sXG4gICAgICAgIFRvUG9ydDogNjU1MzVcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfVxuICB9KSxcblxuICAnbGFtYmRhIHdpdGhvdXQgVlBDIHRocm93cyBFcnJvciB1cG9uIGFjY2Vzc2luZyBjb25uZWN0aW9ucycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgbGFtYmRhRm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTGFtYmRhJywge1xuICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5vZGVKUzgxMCxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBsYW1iZGFGbi5jb25uZWN0aW9ucy5hbGxvd1RvQW55SVB2NChuZXcgZWMyLlRjcEFsbFBvcnRzKCksICdSZWFjaCBmb3IgdGhlIHdvcmxkIExhbWJkYSEnKTtcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdwaWNraW5nIHB1YmxpYyBzdWJuZXRzIGlzIG5vdCBhbGxvd2VkJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB2cGMgPSBuZXcgZWMyLlZwYyhzdGFjaywgJ1ZQQycpO1xuXG4gICAgLy8gV0hFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdMYW1iZGEnLCB7XG4gICAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgnZm9vJyksXG4gICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTODEwLFxuICAgICAgICB2cGMsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUHVibGljIH1cbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH1cbn07XG5cbi8qKlxuICogVXNlIGEgY2xhc3MgYXMgdGVzdCBmaXh0dXJlXG4gKlxuICogc2V0VXAoKSB3aWxsIGJlIG1hcHBlZCB0byB0aGUgKHN5bmNocm9ub3VzKSBjb25zdHJ1Y3Rvci4gdGVhckRvd24oY2IpIHdpbGwgYmUgY2FsbGVkIGlmIGF2YWlsYWJsZS5cbiAqL1xuZnVuY3Rpb24gY2xhc3NGaXh0dXJlKGtsYXNzOiBhbnkpIHtcbiAgbGV0IGZpeHR1cmU6IGFueTtcblxuICBjb25zdCByZXQ6IGFueSA9IHtcbiAgICBzZXRVcChjYjogSUNhbGxiYWNrRnVuY3Rpb24pIHtcbiAgICAgIGZpeHR1cmUgPSBuZXcga2xhc3MoKTtcbiAgICAgIGNiKCk7XG4gICAgfSxcblxuICAgIHRlYXJEb3duKGNiOiBJQ2FsbGJhY2tGdW5jdGlvbikge1xuICAgICAgaWYgKGZpeHR1cmUudGVhckRvd24pIHtcbiAgICAgICAgZml4dHVyZS50ZWFyRG93bihjYik7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBjYigpO1xuICAgICAgfVxuICAgIH1cbiAgfTtcblxuICBjb25zdCB0ZXN0TmFtZXMgPSBSZWZsZWN0Lm93bktleXMoa2xhc3MucHJvdG90eXBlKS5maWx0ZXIobSA9PiBtICE9PSAndGVhckRvd24nICYmIG0gIT09ICdjb25zdHJ1Y3RvcicpO1xuICBmb3IgKGNvbnN0IHRlc3ROYW1lIG9mIHRlc3ROYW1lcykge1xuICAgIHJldFt0ZXN0TmFtZV0gPSAodGVzdDogVGVzdCkgPT4gZml4dHVyZVt0ZXN0TmFtZV0odGVzdCk7XG4gIH1cblxuICByZXR1cm4gcmV0O1xufVxuXG5jbGFzcyBTb21ldGhpbmdDb25uZWN0YWJsZSBpbXBsZW1lbnRzIGVjMi5JQ29ubmVjdGFibGUge1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucykge1xuICB9XG59XG4iXX0=