"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const nodeunit_1 = require("nodeunit");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'creating a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // WHEN
        new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NodeJS810]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersion', {
            Content: {
                S3Bucket: stack.resolve(bucket.bucketName),
                S3Key: 'ObjectKey',
            },
            CompatibleRuntimes: ['nodejs8.10']
        }));
        test.done();
    },
    'granting access to a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const layer = new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NodeJS810]
        });
        // WHEN
        layer.addPermission('GrantUsage-123456789012', { accountId: '123456789012' });
        layer.addPermission('GrantUsage-o-123456', { accountId: '*', organizationId: 'o-123456' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '123456789012',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '*',
            OrganizationId: 'o-123456'
        }));
        test.done();
    },
    'creating a layer with no runtimes compatible'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // THEN
        test.throws(() => new lambda.LayerVersion(stack, 'LayerVersion', { code, compatibleRuntimes: [] }), /supports no runtime/);
        test.done();
    },
    'singleton layers are created exactly once'(test) {
        // Given
        const stack = new cdk.Stack(undefined, 'TestStack');
        const uuid = '75F9D74A-67AF-493E-888A-20976130F0B1';
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // When
        for (let i = 0; i < 5; i++) {
            new lambda.SingletonLayerVersion(stack, `Layer-${i}`, { uuid, code });
        }
        // Then
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::LayerVersion', 1));
        test.done();
    }
});
//# sourceMappingURL=data:application/json;base64,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