"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/cdk");
const path = require("path");
const lambda = require("../lib");
function newTestLambda(scope) {
    return new lambda.Function(scope, 'MyLambda', {
        code: new lambda.InlineCode('foo'),
        handler: 'bar',
        runtime: lambda.Runtime.Python27
    });
}
module.exports = {
    'default function'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs8.10' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'adds policy permissions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            initialPolicy: [new iam.PolicyStatement().addAction("*").addAllResources()]
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaServiceRoleDefaultPolicy5BBC6F68: {
                    Type: "AWS::IAM::Policy",
                    Properties: {
                        PolicyDocument: {
                            Statement: [
                                {
                                    Action: "*",
                                    Effect: "Allow",
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        PolicyName: "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        Roles: [
                            {
                                Ref: "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs8.10' },
                    DependsOn: ['MyLambdaServiceRoleDefaultPolicy5BBC6F68', 'MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'fails if inline code is used for an invalid runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'bar',
            runtime: lambda.Runtime.DotNetCore2
        }));
        test.done();
    },
    'addToResourcePolicy': {
        'can be used to add permissions to the Lambda function'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            fn.addPermission('S3Permission', {
                action: 'lambda:*',
                principal: new iam.ServicePrincipal('s3.amazonaws.com'),
                sourceAccount: stack.accountId,
                sourceArn: 'arn:aws:s3:::my_bucket'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyLambdaServiceRole4539ECB6": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "ManagedPolicyArns": 
                            // tslint:disable-next-line:max-line-length
                            [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                        }
                    },
                    "MyLambdaCCE802FB": {
                        "Type": "AWS::Lambda::Function",
                        "Properties": {
                            "Code": {
                                "ZipFile": "foo"
                            },
                            "Handler": "bar",
                            "Role": {
                                "Fn::GetAtt": [
                                    "MyLambdaServiceRole4539ECB6",
                                    "Arn"
                                ]
                            },
                            "Runtime": "python2.7"
                        },
                        "DependsOn": [
                            "MyLambdaServiceRole4539ECB6"
                        ]
                    },
                    "MyLambdaS3Permission99D0EA08": {
                        "Type": "AWS::Lambda::Permission",
                        "Properties": {
                            "Action": "lambda:*",
                            "FunctionName": {
                                "Fn::GetAtt": [
                                    "MyLambdaCCE802FB",
                                    "Arn"
                                ]
                            },
                            "Principal": "s3.amazonaws.com",
                            "SourceAccount": {
                                "Ref": "AWS::AccountId"
                            },
                            "SourceArn": "arn:aws:s3:::my_bucket"
                        }
                    }
                }
            });
            test.done();
        },
        'fails if the principal is not a service or account principals'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            test.throws(() => fn.addPermission('F1', { principal: new iam.ArnPrincipal('just:arn') }), /Invalid principal type for Lambda permission statement/);
            fn.addPermission('S1', { principal: new iam.ServicePrincipal('my-service') });
            fn.addPermission('S2', { principal: new iam.AccountPrincipal('account') });
            test.done();
        },
        'BYORole'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'SomeRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            role.addToPolicy(new iam.PolicyStatement().addAction('confirm:itsthesame'));
            // WHEN
            const fn = new lambda.Function(stack, 'Function', {
                code: new lambda.InlineCode('test'),
                runtime: lambda.Runtime.Python36,
                handler: 'index.test',
                role,
                initialPolicy: [
                    new iam.PolicyStatement().addAction('inline:inline')
                ]
            });
            fn.addToRolePolicy(new iam.PolicyStatement().addAction('explicit:explicit'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Version": "2012-10-17",
                    "Statement": [
                        { "Action": "confirm:itsthesame", "Effect": "Allow" },
                        { "Action": "inline:inline", "Effect": "Allow" },
                        { "Action": "explicit:explicit", "Effect": "Allow" }
                    ],
                },
            }));
            test.done();
        }
    },
    'fromFunctionArn'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const imported = lambda.Function.fromFunctionArn(stack2, 'Imported', 'arn:aws:lambda:us-east-1:123456789012:function:ProcessKinesisRecords');
        // Can call addPermission() but it won't do anything
        imported.addPermission('Hello', {
            principal: new iam.ServicePrincipal('harry')
        });
        // THEN
        test.deepEqual(imported.functionArn, 'arn:aws:lambda:us-east-1:123456789012:function:ProcessKinesisRecords');
        test.deepEqual(imported.functionName, 'ProcessKinesisRecords');
        test.done();
    },
    'Lambda code can be read from a local directory via an asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: lambda.Code.directory(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            runtime: lambda.Runtime.Python36
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Code": {
                "S3Bucket": {
                    "Ref": "MyLambdaCodeS3BucketC82A5870"
                },
                "S3Key": { "Fn::Join": ["", [
                            { "Fn::Select": [0, { "Fn::Split": ["||", { "Ref": "MyLambdaCodeS3VersionKey47762537" }] }] },
                            { "Fn::Select": [1, { "Fn::Split": ["||", { "Ref": "MyLambdaCodeS3VersionKey47762537" }] }] },
                        ]] }
            },
            "Handler": "index.handler",
            "Role": {
                "Fn::GetAtt": [
                    "MyLambdaServiceRole4539ECB6",
                    "Arn"
                ]
            },
            "Runtime": "python3.6"
        }));
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            functionName: 'OneFunctionToRuleThemAll',
            deadLetterQueueEnabled: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        },
                        "FunctionName": "OneFunctionToRuleThemAll"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName not defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            deadLetterQueueEnabled: true,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to false'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            deadLetterQueueEnabled: false,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriodSec: 1209600
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to true'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriodSec: 1209600
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            deadLetterQueueEnabled: true,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs8.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6",
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'error when default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to false'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriodSec: 1209600
        });
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            deadLetterQueueEnabled: false,
            deadLetterQueue: dlQueue,
        }), /deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false/);
        test.done();
    },
    'default function with Active tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            tracing: lambda.Tracing.Active
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10",
                "TracingConfig": {
                    "Mode": "Active"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with PassThrough tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            tracing: lambda.Tracing.PassThrough
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10",
                "TracingConfig": {
                    "Mode": "PassThrough"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with Disabled tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
            tracing: lambda.Tracing.Disabled
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs8.10"
            },
            "DependsOn": [
                "MyLambdaServiceRole4539ECB6"
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grantInvoke adds iam:InvokeFunction'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AccountPrincipal('1234'),
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        // WHEN
        fn.grantInvoke(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                        Resource: { "Fn::GetAtt": ["Function76856677", "Arn"] }
                    }
                ]
            }
        }));
        test.done();
    },
    'grantInvoke with a service principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        const service = new iam.ServicePrincipal('apigateway.amazonaws.com');
        // WHEN
        fn.grantInvoke(service);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'apigateway.amazonaws.com'
        }));
        test.done();
    },
    'grantInvoke with an account principal'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        const account = new iam.AccountPrincipal('123456789012');
        // WHEN
        fn.grantInvoke(account);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: '123456789012'
        }));
        test.done();
    },
    'Can use metricErrors on a lambda Function'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        // THEN
        test.deepEqual(stack.resolve(fn.metricErrors()), {
            dimensions: { FunctionName: { Ref: 'Function76856677' } },
            namespace: 'AWS/Lambda',
            metricName: 'Errors',
            periodSec: 300,
            statistic: 'Sum',
        });
        test.done();
    },
    'addEventSource calls bind'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        let bindTarget;
        class EventSourceMock {
            bind(target) {
                bindTarget = target;
            }
        }
        // WHEN
        fn.addEventSource(new EventSourceMock());
        // THEN
        test.same(bindTarget, fn);
        test.done();
    },
    'support inline code for Ruby runtime'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.Ruby25,
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'ruby2.5' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'using an incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layer = lambda.LayerVersion.fromLayerVersionAttributes(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NodeJS810],
        });
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers: [layer],
            runtime: lambda.Runtime.NodeJS610,
            code: lambda.Code.inline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /nodejs6.10 is not in \[nodejs8.10\]/);
        test.done();
    },
    'using more than 5 layers'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layers = new Array(6).fill(lambda.LayerVersion.fromLayerVersionAttributes(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NodeJS810],
        }));
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers,
            runtime: lambda.Runtime.NodeJS810,
            code: lambda.Code.inline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /Unable to add layer:/);
        test.done();
    },
    'environment variables are prohibited in China'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'MyLambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NodeJS,
                environment: {
                    SOME: 'Variable'
                }
            });
        }, /Environment variables are not supported/);
        test.done();
    },
    'environment variables work in an unspecified region'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS,
            environment: {
                SOME: 'Variable'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    SOME: "Variable"
                }
            }
        }));
        test.done();
    },
    'support reserved concurrent executions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS,
            reservedConcurrentExecutions: 10
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        ReservedConcurrentExecutions: 10,
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'its possible to specify event sources upon creation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let bindCount = 0;
        class EventSource {
            bind(_) {
                bindCount++;
            }
        }
        // WHEN
        new lambda.Function(stack, 'fn', {
            code: lambda.Code.inline('boom'),
            runtime: lambda.Runtime.NodeJS810,
            handler: 'index.bam',
            events: [
                new EventSource(),
                new EventSource(),
            ]
        });
        // THEN
        test.deepEqual(bindCount, 2);
        test.done();
    },
    'Provided Runtime returns the right values'(test) {
        const rt = lambda.Runtime.Provided;
        test.equal(rt.name, 'provided');
        test.equal(rt.supportsInlineCode, false);
        test.done();
    },
    'specify log retention'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS,
            logRetentionDays: logs.RetentionDays.OneMonth
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            'LogGroupName': {
                'Fn::Join': [
                    '',
                    [
                        '/aws/lambda/',
                        {
                            Ref: 'MyLambdaCCE802FB'
                        }
                    ]
                ]
            },
            'RetentionInDays': 30
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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