"use strict";
const assert_1 = require("@aws-cdk/assert");
const assets = require("@aws-cdk/assets");
const cdk = require("@aws-cdk/cdk");
const cxapi = require("@aws-cdk/cx-api");
const path = require("path");
const lambda = require("../lib");
function defineFunction(code, runtime = lambda.Runtime.NodeJS810) {
    const stack = new cdk.Stack();
    return new lambda.Function(stack, 'Func', {
        handler: 'foom',
        code, runtime
    });
}
function generateRandomString(bytes) {
    let s = '';
    for (let i = 0; i < bytes; ++i) {
        s += String.fromCharCode(Math.round(Math.random() * 256));
    }
    return s;
}
module.exports = {
    'lambda.Code.inline': {
        'fails if used with unsupported runtimes'(test) {
            test.throws(() => defineFunction(lambda.Code.inline('boom'), lambda.Runtime.Go1x), /Inline source not allowed for go1\.x/);
            test.throws(() => defineFunction(lambda.Code.inline('boom'), lambda.Runtime.Java8), /Inline source not allowed for java8/);
            test.done();
        },
        'fails if larger than 4096 bytes'(test) {
            test.throws(() => defineFunction(lambda.Code.inline(generateRandomString(4097)), lambda.Runtime.NodeJS810), /Lambda source is too large, must be <= 4096 but is 4097/);
            test.done();
        }
    },
    'lambda.Code.asset': {
        'determines packaging type from file type'(test) {
            // WHEN
            const fileAsset = lambda.Code.asset(path.join(__dirname, 'handler.zip'));
            const directoryAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler'));
            // THEN
            test.deepEqual(fileAsset.packaging, assets.AssetPackaging.File);
            test.deepEqual(directoryAsset.packaging, assets.AssetPackaging.ZipDirectory);
            test.done();
        },
        'fails if a non-zip asset is used'(test) {
            // GIVEN
            const fileAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler', 'index.py'));
            // THEN
            test.throws(() => defineFunction(fileAsset), /Asset must be a \.zip file or a directory/);
            test.done();
        },
        'only one Asset object gets created even if multiple functions use the same AssetCode'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'MyStack');
            const directoryAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler'));
            // WHEN
            new lambda.Function(stack, 'Func1', {
                handler: 'foom',
                runtime: lambda.Runtime.NodeJS810,
                code: directoryAsset
            });
            new lambda.Function(stack, 'Func2', {
                handler: 'foom',
                runtime: lambda.Runtime.NodeJS810,
                code: directoryAsset
            });
            // THEN
            const assembly = app.synth();
            const synthesized = assembly.stacks[0];
            // Func1 has an asset, Func2 does not
            const metadata = synthesized.manifest.metadata || {};
            test.ok(metadata['/MyStack/Func1/Code']);
            test.deepEqual(metadata['/MyStack/Func1/Code'].length, 1);
            test.deepEqual(metadata['/MyStack/Func1/Code'][0].type, 'aws:cdk:asset');
            test.deepEqual(metadata['/MyStack/Func2/Code'], undefined);
            test.done();
        },
        'adds code asset metadata'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
            const location = path.join(__dirname, 'my-lambda-handler');
            // WHEN
            new lambda.Function(stack, 'Func1', {
                code: lambda.Code.asset(location),
                runtime: lambda.Runtime.NodeJS810,
                handler: 'foom',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Metadata: {
                    [cxapi.ASSET_RESOURCE_METADATA_PATH_KEY]: 'asset.9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232',
                    [cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY]: 'Code'
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }
    },
    'lambda.Code.cfnParameters': {
        "automatically creates the Bucket and Key parameters when it's used in a Function"(test) {
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode();
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NodeJS810,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: "FunctionLambdaSourceBucketNameParameter9E9E108F",
                    },
                    S3Key: {
                        Ref: "FunctionLambdaSourceObjectKeyParameter1C7AED11",
                    },
                },
            }));
            test.equal(stack.resolve(code.bucketNameParam), 'FunctionLambdaSourceBucketNameParameter9E9E108F');
            test.equal(stack.resolve(code.objectKeyParam), 'FunctionLambdaSourceObjectKeyParameter1C7AED11');
            test.done();
        },
        'does not allow accessing the Parameter properties before being used in a Function'(test) {
            const code = new lambda.CfnParametersCode();
            test.throws(() => {
                test.notEqual(code.bucketNameParam, undefined);
            }, /bucketNameParam/);
            test.throws(() => {
                test.notEqual(code.objectKeyParam, undefined);
            }, /objectKeyParam/);
            test.done();
        },
        'allows passing custom Parameters when creating it'(test) {
            const stack = new cdk.Stack();
            const bucketNameParam = new cdk.CfnParameter(stack, 'BucketNameParam', {
                type: 'String',
            });
            const bucketKeyParam = new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                type: 'String',
            });
            const code = lambda.Code.cfnParameters({
                bucketNameParam,
                objectKeyParam: bucketKeyParam,
            });
            test.equal(stack.resolve(code.bucketNameParam), 'BucketNameParam');
            test.equal(stack.resolve(code.objectKeyParam), 'ObjectKeyParam');
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NodeJS810,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: "BucketNameParam",
                    },
                    S3Key: {
                        Ref: "ObjectKeyParam",
                    },
                },
            }));
            test.done();
        },
        'can assign parameters'(test) {
            // given
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode({
                bucketNameParam: new cdk.CfnParameter(stack, 'BucketNameParam', {
                    type: 'String',
                }),
                objectKeyParam: new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                    type: 'String',
                }),
            });
            // when
            const overrides = stack.resolve(code.assign({
                bucketName: 'SomeBucketName',
                objectKey: 'SomeObjectKey',
            }));
            // then
            test.equal(overrides['BucketNameParam'], 'SomeBucketName');
            test.equal(overrides['ObjectKeyParam'], 'SomeObjectKey');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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