"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const path = require("path");
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.asset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NodeJS810,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
        });
        provider.addToRolePolicy(// Duplicate statements will be deduplicated by `PolicyDocument`
        new iam.PolicyStatement()
            .addActions('logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy')
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing it's name
            // creates a CF circular dependency.
            .addAllResources());
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                RetentionInDays: props.retentionDays === Infinity ? undefined : props.retentionDays
            }
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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