"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends cdk_1.Resource {
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    /**
     * Imports a layer version by ARN. Assumes it is compatible with all Lambda runtimes.
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.All
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        props.code.bind(this);
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: new cdk_1.Token(() => props.code._toJSON(resource)),
            description: props.description,
            layerName: props.name,
            licenseInfo: props.license,
        });
        this.layerVersionArn = resource.layerVersionArn;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
}
exports.LayerVersion = LayerVersion;
/**
 * A Singleton Lambda Layer Version. The construct gurantees exactly one LayerVersion will be created in a given Stack
 * for the provided ``uuid``. It is recommended to use ``uuidgen`` to create a new ``uuid`` each time a new singleton
 * layer is created.
 */
class SingletonLayerVersion extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.layerVersion = this.ensureLayerVersion(props);
    }
    get layerVersionArn() {
        return this.layerVersion.layerVersionArn;
    }
    get compatibleRuntimes() {
        return this.layerVersion.compatibleRuntimes;
    }
    addPermission(id, grantee) {
        this.layerVersion.addPermission(id, grantee);
    }
    ensureLayerVersion(props) {
        const singletonId = `SingletonLayer-${props.uuid}`;
        const existing = cdk_1.Stack.of(this).node.tryFindChild(singletonId);
        if (existing) {
            return existing;
        }
        return new LayerVersion(cdk_1.Stack.of(this), singletonId, props);
    }
}
exports.SingletonLayerVersion = SingletonLayerVersion;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGF5ZXJzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibGF5ZXJzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsc0NBQTRFO0FBRTVFLHlEQUFnRjtBQUNoRix1Q0FBb0M7QUFpRXBDOztHQUVHO0FBQ0gsTUFBZSxnQkFBaUIsU0FBUSxjQUFRO0lBSXZDLGFBQWEsQ0FBQyxFQUFVLEVBQUUsVUFBa0M7UUFDakUsSUFBSSxVQUFVLENBQUMsY0FBYyxJQUFJLElBQUksSUFBSSxVQUFVLENBQUMsU0FBUyxLQUFLLEdBQUcsRUFBRTtZQUNyRSxNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxVQUFVLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztTQUNuSDtRQUVELElBQUksNENBQXlCLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUN0QyxNQUFNLEVBQUUsd0JBQXdCO1lBQ2hDLGVBQWUsRUFBRSxJQUFJLENBQUMsZUFBZTtZQUNyQyxTQUFTLEVBQUUsVUFBVSxDQUFDLFNBQVM7WUFDL0IsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjO1NBQzFDLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQW1DRDs7R0FFRztBQUNILE1BQWEsWUFBYSxTQUFRLGdCQUFnQjtJQUVoRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxlQUF1QjtRQUNyRixPQUFPLFlBQVksQ0FBQywwQkFBMEIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ3hELGVBQWU7WUFDZixrQkFBa0IsRUFBRSxpQkFBTyxDQUFDLEdBQUc7U0FDaEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNsRyxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNyRSxNQUFNLElBQUksS0FBSyxDQUFDLDhEQUE4RCxDQUFDLENBQUM7U0FDakY7UUFFRCxNQUFNLE1BQU8sU0FBUSxnQkFBZ0I7WUFBckM7O2dCQUNrQixvQkFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUM7Z0JBQ3hDLHVCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztZQUNoRSxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBS0QsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3QjtRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksS0FBSyxDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3JFLE1BQU0sSUFBSSxLQUFLLENBQUMsOERBQThELENBQUMsQ0FBQztTQUNqRjtRQUNELElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxrREFBa0QsQ0FBQyxDQUFDO1NBQ3JFO1FBQ0QsNkNBQTZDO1FBQzdDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXRCLE1BQU0sUUFBUSxHQUFHLElBQUksa0NBQWUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3JELGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztZQUN6RixPQUFPLEVBQUUsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDdEQsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLFNBQVMsRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNyQixXQUFXLEVBQUUsS0FBSyxDQUFDLE9BQU87U0FDM0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsR0FBRyxRQUFRLENBQUMsZUFBZSxDQUFDO1FBQ2hELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDckQsQ0FBQztDQUNGO0FBekRELG9DQXlEQztBQWVEOzs7O0dBSUc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLGVBQVM7SUFHbEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQztRQUN6RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRCxJQUFXLGVBQWU7UUFDeEIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGVBQWUsQ0FBQztJQUMzQyxDQUFDO0lBRUQsSUFBVyxrQkFBa0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGtCQUFrQixDQUFDO0lBQzlDLENBQUM7SUFFTSxhQUFhLENBQUMsRUFBVSxFQUFFLE9BQStCO1FBQzlELElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRU8sa0JBQWtCLENBQUMsS0FBaUM7UUFDMUQsTUFBTSxXQUFXLEdBQUcsa0JBQWtCLEtBQUssQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNuRCxNQUFNLFFBQVEsR0FBRyxXQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDL0QsSUFBSSxRQUFRLEVBQUU7WUFDWixPQUFPLFFBQW9DLENBQUM7U0FDN0M7UUFDRCxPQUFPLElBQUksWUFBWSxDQUFDLFdBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzlELENBQUM7Q0FDRjtBQTdCRCxzREE2QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIElSZXNvdXJjZSwgUmVzb3VyY2UsIFN0YWNrLCBUb2tlbiB9IGZyb20gJ0Bhd3MtY2RrL2Nkayc7XG5pbXBvcnQgeyBDb2RlIH0gZnJvbSAnLi9jb2RlJztcbmltcG9ydCB7IENmbkxheWVyVmVyc2lvbiwgQ2ZuTGF5ZXJWZXJzaW9uUGVybWlzc2lvbiB9IGZyb20gJy4vbGFtYmRhLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBSdW50aW1lIH0gZnJvbSAnLi9ydW50aW1lJztcblxuZXhwb3J0IGludGVyZmFjZSBMYXllclZlcnNpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgcnVudGltZXMgY29tcGF0aWJsZSB3aXRoIHRoaXMgTGF5ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQWxsIHJ1bnRpbWVzIGFyZSBzdXBwb3J0ZWQuXG4gICAqL1xuICByZWFkb25seSBjb21wYXRpYmxlUnVudGltZXM/OiBSdW50aW1lW107XG5cbiAgLyoqXG4gICAqIFRoZSBjb250ZW50IG9mIHRoaXMgTGF5ZXIuIFVzaW5nICppbmxpbmUqIChwZXIgYGBjb2RlLmlzSW5saW5lYGApIGNvZGUgaXMgbm90IHBlcm1pdHRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGNvZGU6IENvZGU7XG5cbiAgLyoqXG4gICAqIFRoZSBkZXNjcmlwdGlvbiB0aGUgdGhpcyBMYW1iZGEgTGF5ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZGVzY3JpcHRpb24uXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFNQRFggbGljZW5jZSBpZGVudGlmaWVyIG9yIFVSTCB0byB0aGUgbGljZW5zZSBmaWxlIGZvciB0aGlzIGxheWVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGxpY2Vuc2UgaW5mb3JtYXRpb24gd2lsbCBiZSByZWNvcmRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGxpY2Vuc2U/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBsYXllci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIG5hbWUgd2lsbCBiZSBnZW5lcmF0ZWQuXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIElMYXllclZlcnNpb24gZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgTGFtYmRhIExheWVyIHZlcnNpb24gdGhhdCB0aGlzIExheWVyIGRlZmluZXMuXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGxheWVyVmVyc2lvbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcnVudGltZXMgY29tcGF0aWJsZSB3aXRoIHRoaXMgTGF5ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFJ1bnRpbWUuQWxsXG4gICAqL1xuICByZWFkb25seSBjb21wYXRpYmxlUnVudGltZXM/OiBSdW50aW1lW107XG5cbiAgLyoqXG4gICAqIEFkZCBwZXJtaXNzaW9uIGZvciB0aGlzIGxheWVyIHZlcnNpb24gdG8gc3BlY2lmaWMgZW50aXRpZXMuIFVzYWdlIHdpdGhpblxuICAgKiB0aGUgc2FtZSBhY2NvdW50IHdoZXJlIHRoZSBsYXllciBpcyBkZWZpbmVkIGlzIGFsd2F5cyBhbGxvd2VkIGFuZCBkb2VzIG5vdFxuICAgKiByZXF1aXJlIGNhbGxpbmcgdGhpcyBtZXRob2QuIE5vdGUgdGhhdCB0aGUgcHJpbmNpcGFsIHRoYXQgY3JlYXRlcyB0aGVcbiAgICogTGFtYmRhIGZ1bmN0aW9uIHVzaW5nIHRoZSBsYXllciAoZm9yIGV4YW1wbGUsIGEgQ2xvdWRGb3JtYXRpb24gY2hhbmdlc2V0XG4gICAqIGV4ZWN1dGlvbiByb2xlKSBhbHNvIG5lZWRzIHRvIGhhdmUgdGhlIGBgbGFtYmRhOkdldExheWVyVmVyc2lvbmBgXG4gICAqIHBlcm1pc3Npb24gb24gdGhlIGxheWVyIHZlcnNpb24uXG4gICAqXG4gICAqIEBwYXJhbSBpZCB0aGUgSUQgb2YgdGhlIGdyYW50IGluIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAgICogQHBhcmFtIHBlcm1pc3Npb24gdGhlIGlkZW50aWZpY2F0aW9uIG9mIHRoZSBncmFudGVlLlxuICAgKi9cbiAgYWRkUGVybWlzc2lvbihpZDogc3RyaW5nLCBwZXJtaXNzaW9uOiBMYXllclZlcnNpb25QZXJtaXNzaW9uKTogdm9pZDtcbn1cblxuLyoqXG4gKiBBIHJlZmVyZW5jZSB0byBhIExhbWJkYSBMYXllciB2ZXJzaW9uLlxuICovXG5hYnN0cmFjdCBjbGFzcyBMYXllclZlcnNpb25CYXNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJTGF5ZXJWZXJzaW9uIHtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGxheWVyVmVyc2lvbkFybjogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgY29tcGF0aWJsZVJ1bnRpbWVzPzogUnVudGltZVtdO1xuXG4gIHB1YmxpYyBhZGRQZXJtaXNzaW9uKGlkOiBzdHJpbmcsIHBlcm1pc3Npb246IExheWVyVmVyc2lvblBlcm1pc3Npb24pIHtcbiAgICBpZiAocGVybWlzc2lvbi5vcmdhbml6YXRpb25JZCAhPSBudWxsICYmIHBlcm1pc3Npb24uYWNjb3VudElkICE9PSAnKicpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgT3JnYW5pemF0aW9uSWQgY2FuIG9ubHkgYmUgc3BlY2lmaWVkIGlmIEF3c0FjY291bnRJZCBpcyAnKicsIGJ1dCBpdCBpcyAke3Blcm1pc3Npb24uYWNjb3VudElkfWApO1xuICAgIH1cblxuICAgIG5ldyBDZm5MYXllclZlcnNpb25QZXJtaXNzaW9uKHRoaXMsIGlkLCB7XG4gICAgICBhY3Rpb246ICdsYW1iZGE6R2V0TGF5ZXJWZXJzaW9uJyxcbiAgICAgIGxheWVyVmVyc2lvbkFybjogdGhpcy5sYXllclZlcnNpb25Bcm4sXG4gICAgICBwcmluY2lwYWw6IHBlcm1pc3Npb24uYWNjb3VudElkLFxuICAgICAgb3JnYW5pemF0aW9uSWQ6IHBlcm1pc3Npb24ub3JnYW5pemF0aW9uSWQsXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBJZGVudGlmaWNhdGlvbiBvZiBhbiBhY2NvdW50IChvciBvcmdhbml6YXRpb24pIHRoYXQgaXMgYWxsb3dlZCB0byBhY2Nlc3MgYSBMYW1iZGEgTGF5ZXIgVmVyc2lvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXllclZlcnNpb25QZXJtaXNzaW9uIHtcbiAgLyoqXG4gICAqIFRoZSBBV1MgQWNjb3VudCBpZCBvZiB0aGUgYWNjb3VudCB0aGF0IGlzIGF1dGhvcml6ZWQgdG8gdXNlIGEgTGFtYmRhIExheWVyIFZlcnNpb24uIFRoZSB3aWxkLWNhcmQgYGAnKidgYCBjYW4gYmVcbiAgICogdXNlZCB0byBncmFudCBhY2Nlc3MgdG8gXCJhbnlcIiBhY2NvdW50IChvciBhbnkgYWNjb3VudCBpbiBhbiBvcmdhbml6YXRpb24gd2hlbiBgYG9yZ2FuaXphdGlvbklkYGAgaXMgc3BlY2lmaWVkKS5cbiAgICovXG4gIHJlYWRvbmx5IGFjY291bnRJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIEFXUyBPcmdhbml6YXRpb24gdG8gaHdpY2ggdGhlIGdyYW50IGlzIHJlc3RyaWN0ZWQuXG4gICAqXG4gICAqIENhbiBvbmx5IGJlIHNwZWNpZmllZCBpZiBgYGFjY291bnRJZGBgIGlzIGBgJyonYGBcbiAgICovXG4gIHJlYWRvbmx5IG9yZ2FuaXphdGlvbklkPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgbmVjZXNzYXJ5IHRvIGltcG9ydCBhIExheWVyVmVyc2lvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXllclZlcnNpb25BdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIExheWVyVmVyc2lvbi5cbiAgICovXG4gIHJlYWRvbmx5IGxheWVyVmVyc2lvbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdCBvZiBjb21wYXRpYmxlIHJ1bnRpbWVzIHdpdGggdGhpcyBMYXllci5cbiAgICovXG4gIHJlYWRvbmx5IGNvbXBhdGlibGVSdW50aW1lcz86IFJ1bnRpbWVbXTtcbn1cblxuLyoqXG4gKiBEZWZpbmVzIGEgbmV3IExhbWJkYSBMYXllciB2ZXJzaW9uLlxuICovXG5leHBvcnQgY2xhc3MgTGF5ZXJWZXJzaW9uIGV4dGVuZHMgTGF5ZXJWZXJzaW9uQmFzZSB7XG5cbiAgLyoqXG4gICAqIEltcG9ydHMgYSBsYXllciB2ZXJzaW9uIGJ5IEFSTi4gQXNzdW1lcyBpdCBpcyBjb21wYXRpYmxlIHdpdGggYWxsIExhbWJkYSBydW50aW1lcy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxheWVyVmVyc2lvbkFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBsYXllclZlcnNpb25Bcm46IHN0cmluZyk6IElMYXllclZlcnNpb24ge1xuICAgIHJldHVybiBMYXllclZlcnNpb24uZnJvbUxheWVyVmVyc2lvbkF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7XG4gICAgICBsYXllclZlcnNpb25Bcm4sXG4gICAgICBjb21wYXRpYmxlUnVudGltZXM6IFJ1bnRpbWUuQWxsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogSW1wb3J0cyBhIExheWVyIHRoYXQgaGFzIGJlZW4gZGVmaW5lZCBleHRlcm5hbGx5LlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgdGhlIHBhcmVudCBDb25zdHJ1Y3QgdGhhdCB3aWxsIHVzZSB0aGUgaW1wb3J0ZWQgbGF5ZXIuXG4gICAqIEBwYXJhbSBpZCAgICB0aGUgaWQgb2YgdGhlIGltcG9ydGVkIGxheWVyIGluIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAgICogQHBhcmFtIGF0dHJzIHRoZSBwcm9wZXJ0aWVzIG9mIHRoZSBpbXBvcnRlZCBsYXllci5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxheWVyVmVyc2lvbkF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IExheWVyVmVyc2lvbkF0dHJpYnV0ZXMpOiBJTGF5ZXJWZXJzaW9uIHtcbiAgICBpZiAoYXR0cnMuY29tcGF0aWJsZVJ1bnRpbWVzICYmIGF0dHJzLmNvbXBhdGlibGVSdW50aW1lcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXR0ZW1wdGVkIHRvIGltcG9ydCBhIExhbWJkYSBsYXllciB0aGF0IHN1cHBvcnRzIG5vIHJ1bnRpbWUhJyk7XG4gICAgfVxuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgTGF5ZXJWZXJzaW9uQmFzZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgbGF5ZXJWZXJzaW9uQXJuID0gYXR0cnMubGF5ZXJWZXJzaW9uQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGNvbXBhdGlibGVSdW50aW1lcyA9IGF0dHJzLmNvbXBhdGlibGVSdW50aW1lcztcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGxheWVyVmVyc2lvbkFybjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY29tcGF0aWJsZVJ1bnRpbWVzPzogUnVudGltZVtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBMYXllclZlcnNpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgaWYgKHByb3BzLmNvbXBhdGlibGVSdW50aW1lcyAmJiBwcm9wcy5jb21wYXRpYmxlUnVudGltZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0dGVtcHRlZCB0byBkZWZpbmUgYSBMYW1iZGEgbGF5ZXIgdGhhdCBzdXBwb3J0cyBubyBydW50aW1lIScpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuY29kZS5pc0lubGluZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdMYW1iZGEgbGF5ZXJzIGNhbm5vdCBiZSBjcmVhdGVkIGZyb20gaW5saW5lIGNvZGUnKTtcbiAgICB9XG4gICAgLy8gQWxsb3cgdXNhZ2Ugb2YgdGhlIGNvZGUgaW4gdGhpcyBjb250ZXh0Li4uXG4gICAgcHJvcHMuY29kZS5iaW5kKHRoaXMpO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuTGF5ZXJWZXJzaW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNvbXBhdGlibGVSdW50aW1lczogcHJvcHMuY29tcGF0aWJsZVJ1bnRpbWVzICYmIHByb3BzLmNvbXBhdGlibGVSdW50aW1lcy5tYXAociA9PiByLm5hbWUpLFxuICAgICAgY29udGVudDogbmV3IFRva2VuKCgpID0+IHByb3BzLmNvZGUuX3RvSlNPTihyZXNvdXJjZSkpLFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgbGF5ZXJOYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgbGljZW5zZUluZm86IHByb3BzLmxpY2Vuc2UsXG4gICAgfSk7XG5cbiAgICB0aGlzLmxheWVyVmVyc2lvbkFybiA9IHJlc291cmNlLmxheWVyVmVyc2lvbkFybjtcbiAgICB0aGlzLmNvbXBhdGlibGVSdW50aW1lcyA9IHByb3BzLmNvbXBhdGlibGVSdW50aW1lcztcbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgb2YgYSBTaW5nbGV0b24gTGFtYmRhIExheWVyIFZlcnNpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2luZ2xldG9uTGF5ZXJWZXJzaW9uUHJvcHMgZXh0ZW5kcyBMYXllclZlcnNpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBBIHVuaXF1ZSBpZGVudGlmaWVyIHRvIGlkZW50aWZ5IHRoaXMgbGFtYmRhIGxheWVyIHZlcnNpb24uXG4gICAqXG4gICAqIFRoZSBpZGVudGlmaWVyIHNob3VsZCBiZSB1bmlxdWUgYWNyb3NzIGFsbCBsYXllciBwcm92aWRlcnMuXG4gICAqIFdlIHJlY29tbWVuZCBnZW5lcmF0aW5nIGEgVVVJRCBwZXIgcHJvdmlkZXIuXG4gICAqL1xuICByZWFkb25seSB1dWlkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQSBTaW5nbGV0b24gTGFtYmRhIExheWVyIFZlcnNpb24uIFRoZSBjb25zdHJ1Y3QgZ3VyYW50ZWVzIGV4YWN0bHkgb25lIExheWVyVmVyc2lvbiB3aWxsIGJlIGNyZWF0ZWQgaW4gYSBnaXZlbiBTdGFja1xuICogZm9yIHRoZSBwcm92aWRlZCBgYHV1aWRgYC4gSXQgaXMgcmVjb21tZW5kZWQgdG8gdXNlIGBgdXVpZGdlbmBgIHRvIGNyZWF0ZSBhIG5ldyBgYHV1aWRgYCBlYWNoIHRpbWUgYSBuZXcgc2luZ2xldG9uXG4gKiBsYXllciBpcyBjcmVhdGVkLlxuICovXG5leHBvcnQgY2xhc3MgU2luZ2xldG9uTGF5ZXJWZXJzaW9uIGV4dGVuZHMgQ29uc3RydWN0IGltcGxlbWVudHMgSUxheWVyVmVyc2lvbiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgbGF5ZXJWZXJzaW9uOiBJTGF5ZXJWZXJzaW9uO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTaW5nbGV0b25MYXllclZlcnNpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmxheWVyVmVyc2lvbiA9IHRoaXMuZW5zdXJlTGF5ZXJWZXJzaW9uKHByb3BzKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgbGF5ZXJWZXJzaW9uQXJuKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMubGF5ZXJWZXJzaW9uLmxheWVyVmVyc2lvbkFybjtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgY29tcGF0aWJsZVJ1bnRpbWVzKCk6IFJ1bnRpbWVbXSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMubGF5ZXJWZXJzaW9uLmNvbXBhdGlibGVSdW50aW1lcztcbiAgfVxuXG4gIHB1YmxpYyBhZGRQZXJtaXNzaW9uKGlkOiBzdHJpbmcsIGdyYW50ZWU6IExheWVyVmVyc2lvblBlcm1pc3Npb24pIHtcbiAgICB0aGlzLmxheWVyVmVyc2lvbi5hZGRQZXJtaXNzaW9uKGlkLCBncmFudGVlKTtcbiAgfVxuXG4gIHByaXZhdGUgZW5zdXJlTGF5ZXJWZXJzaW9uKHByb3BzOiBTaW5nbGV0b25MYXllclZlcnNpb25Qcm9wcyk6IElMYXllclZlcnNpb24ge1xuICAgIGNvbnN0IHNpbmdsZXRvbklkID0gYFNpbmdsZXRvbkxheWVyLSR7cHJvcHMudXVpZH1gO1xuICAgIGNvbnN0IGV4aXN0aW5nID0gU3RhY2sub2YodGhpcykubm9kZS50cnlGaW5kQ2hpbGQoc2luZ2xldG9uSWQpO1xuICAgIGlmIChleGlzdGluZykge1xuICAgICAgcmV0dXJuIGV4aXN0aW5nIGFzIHVua25vd24gYXMgSUxheWVyVmVyc2lvbjtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBMYXllclZlcnNpb24oU3RhY2sub2YodGhpcyksIHNpbmdsZXRvbklkLCBwcm9wcyk7XG4gIH1cbn1cbiJdfQ==