"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
class FunctionBase extends cdk_1.Resource {
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        new lambda_generated_1.CfnPermission(this, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    get id() {
        return this.node.id;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['lambda:InvokeFunction'],
            resourceArns: [this.functionArn],
            // Fake resource-like object on which to call addToResourcePolicy(), which actually
            // calls addPermission()
            resource: {
                addToResourcePolicy: (_statement) => {
                    // Couldn't add permissions to the principal, so add them locally.
                    const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the princpal
                    this.addPermission(identifier, {
                        principal: grantee.grantPrincipal,
                        action: 'lambda:InvokeFunction',
                    });
                },
                node: this.node,
            },
        });
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // use duck-typing, not instance of
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if (`service` in principal) {
            return principal.service;
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
//# sourceMappingURL=data:application/json;base64,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