"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("@aws-cdk/assets");
const cdk = require("@aws-cdk/cdk");
const fs = require("fs");
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static bucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static inline(code) {
        return new InlineCode(code);
    }
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static asset(path) {
        return new AssetCode(path);
    }
    /**
     * @returns Zip archives the contents of a directory on disk and uses this
     * as the lambda handler's code.
     * @param directoryToZip The directory to zip
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static directory(directoryToZip) {
        return new AssetCode(directoryToZip, assets.AssetPackaging.ZipDirectory);
    }
    /**
     * @returns Uses a file on disk as a lambda handler's code.
     * @param filePath The file path
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static file(filePath) {
        return new AssetCode(filePath, assets.AssetPackaging.File);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static cfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to
     * bind to the stack, add resources and have fun.
     */
    bind(_construct) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * @internal
     */
    _toJSON(_) {
        return {
            s3Bucket: this.bucketName,
            s3Key: this.key,
            s3ObjectVersion: this.objectVersion
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length > 4096) {
            throw new Error("Lambda source is too large, must be <= 4096 but is " + code.length);
        }
    }
    bind(construct) {
        const runtime = construct.runtime;
        if (!runtime.supportsInlineCode) {
            throw new Error(`Inline source not allowed for ${runtime && runtime.name}`);
        }
    }
    /**
     * @internal
     */
    _toJSON(_) {
        return {
            zipFile: this.code
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     * @param packaging The asset packaging format (optional, determined automatically)
     */
    constructor(path, packaging) {
        super();
        this.path = path;
        this.isInline = false;
        if (packaging !== undefined) {
            this.packaging = packaging;
        }
        else {
            this.packaging = fs.lstatSync(path).isDirectory()
                ? assets.AssetPackaging.ZipDirectory
                : assets.AssetPackaging.File;
        }
    }
    bind(construct) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new assets.Asset(construct, 'Code', {
                path: this.path,
                packaging: this.packaging
            });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
    }
    /**
     * @internal
     */
    _toJSON(resource) {
        if (resource) {
            // https://github.com/awslabs/aws-cdk/issues/1432
            this.asset.addResourceMetadata(resource, 'Code');
        }
        return {
            s3Bucket: this.asset.s3BucketName,
            s3Key: this.asset.s3ObjectKey
        };
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(construct) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(construct, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(construct, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 coordinates
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Coordinates` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param coordinates the coordinates of the object in S3 that represents the Lambda code
     */
    assign(coordinates) {
        const ret = {};
        ret[this.bucketNameParam] = coordinates.bucketName;
        ret[this.objectKeyParam] = coordinates.objectKey;
        return ret;
    }
    /** @internal */
    _toJSON(_) {
        return {
            s3Bucket: this._bucketNameParam.stringValue,
            s3Key: this._objectKeyParam.stringValue,
        };
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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