import assets = require('@aws-cdk/assets');
import s3 = require('@aws-cdk/aws-s3');
import cdk = require('@aws-cdk/cdk');
export declare abstract class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static bucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Code;
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static inline(code: string): InlineCode;
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static asset(path: string): AssetCode;
    /**
     * @returns Zip archives the contents of a directory on disk and uses this
     * as the lambda handler's code.
     * @param directoryToZip The directory to zip
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static directory(directoryToZip: string): AssetCode;
    /**
     * @returns Uses a file on disk as a lambda handler's code.
     * @param filePath The file path
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static file(filePath: string): AssetCode;
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static cfnParameters(props?: CfnParametersCodeProps): CfnParametersCode;
    /**
     * Determines whether this Code is inline code or not.
     */
    abstract readonly isInline: boolean;
    /**
     * Called when the lambda or layer is initialized to allow this object to
     * bind to the stack, add resources and have fun.
     */
    bind(_construct: cdk.Construct): void;
}
/**
 * Lambda code from an S3 archive.
 */
export declare class S3Code extends Code {
    private key;
    private objectVersion?;
    readonly isInline = false;
    private bucketName;
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
}
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
export declare class InlineCode extends Code {
    private code;
    readonly isInline = true;
    constructor(code: string);
    bind(construct: cdk.Construct): void;
}
/**
 * Lambda code from a local directory.
 */
export declare class AssetCode extends Code {
    readonly path: string;
    readonly isInline = false;
    /**
     * The asset packaging.
     */
    readonly packaging: assets.AssetPackaging;
    private asset?;
    /**
     * @param path The path to the asset file or directory.
     * @param packaging The asset packaging format (optional, determined automatically)
     */
    constructor(path: string, packaging?: assets.AssetPackaging);
    bind(construct: cdk.Construct): void;
}
/**
 * Construction properties for {@link CfnParametersCode}.
 */
export interface CfnParametersCodeProps {
    /**
     * The CloudFormation parameter that represents the name of the S3 Bucket
     * where the Lambda code will be located in.
     * Must be of type 'String'.
     *
     * @default a new parameter will be created
     */
    readonly bucketNameParam?: cdk.CfnParameter;
    /**
     * The CloudFormation parameter that represents the path inside the S3 Bucket
     * where the Lambda code will be located at.
     * Must be of type 'String'.
     *
     * @default a new parameter will be created
     */
    readonly objectKeyParam?: cdk.CfnParameter;
}
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
export declare class CfnParametersCode extends Code {
    readonly isInline = false;
    private _bucketNameParam?;
    private _objectKeyParam?;
    constructor(props?: CfnParametersCodeProps);
    bind(construct: cdk.Construct): void;
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 coordinates
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Coordinates` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param coordinates the coordinates of the object in S3 that represents the Lambda code
     */
    assign(coordinates: s3.Coordinates): {
        [name: string]: any;
    };
    readonly bucketNameParam: string;
    readonly objectKeyParam: string;
}
