"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const function_base_1 = require("./function-base");
const lambda_generated_1 = require("./lambda.generated");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.canCreatePermissions = true;
        this.aliasName = props.aliasName;
        this.underlyingLambda = props.version.lambda;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: props.aliasName,
            description: props.description,
            functionName: this.underlyingLambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props)
        });
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${cdk_1.Stack.of(this).parseArn(alias.aliasArn, ":").resourceName}:${props.aliasName}`;
        this.functionArn = alias.aliasArn;
    }
    /**
     * Role associated with this alias
     */
    get role() {
        return this.underlyingLambda.role;
    }
    get grantPrincipal() {
        return this.underlyingLambda.grantPrincipal;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.underlyingLambda.functionName,
                // construct the ARN from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/awslabs/aws-cdk/issues/2231
                Resource: `${this.underlyingLambda.functionArn}:${this.aliasName}`
            },
            ...props
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight
                };
            })
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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