import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import { Construct } from '@aws-cdk/cdk';
import { FunctionBase } from './function-base';
import { IVersion } from './lambda-version';
/**
 * Properties for a new Lambda alias
 */
export interface AliasProps {
    /**
     * Description for the alias
     *
     * @default No description
     */
    readonly description?: string;
    /**
     * Function version this alias refers to
     *
     * Use lambda.addVersion() to obtain a new lambda version to refer to.
     */
    readonly version: IVersion;
    /**
     * Name of this alias
     */
    readonly aliasName: string;
    /**
     * Additional versions with individual weights this alias points to
     *
     * Individual additional version weights specified here should add up to
     * (less than) one. All remaining weight is routed to the default
     * version.
     *
     * For example, the config is
     *
     *    version: "1"
     *    additionalVersions: [{ version: "2", weight: 0.05 }]
     *
     * Then 5% of traffic will be routed to function version 2, while
     * the remaining 95% of traffic will be routed to function version 1.
     *
     * @default No additional versions
     */
    readonly additionalVersions?: VersionWeight[];
}
/**
 * A new alias to a particular version of a Lambda function.
 */
export declare class Alias extends FunctionBase {
    /**
     * Name of this alias.
     *
     * @attribute
     */
    readonly aliasName: string;
    /**
     * ARN of this alias
     *
     * Used to be able to use Alias in place of a regular Lambda. Lambda accepts
     * ARNs everywhere it accepts function names.
     */
    readonly functionName: string;
    /**
     * ARN of this alias
     *
     * Used to be able to use Alias in place of a regular Lambda. Lambda accepts
     * ARNs everywhere it accepts function names.
     */
    readonly functionArn: string;
    protected readonly canCreatePermissions: boolean;
    /**
     * The actual Lambda function object that this Alias is pointing to
     */
    private readonly underlyingLambda;
    constructor(scope: Construct, id: string, props: AliasProps);
    /**
     * Role associated with this alias
     */
    readonly role: import("@aws-cdk/aws-iam").IRole | undefined;
    readonly grantPrincipal: import("@aws-cdk/aws-iam").IPrincipal;
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Calculate the routingConfig parameter from the input props
     */
    private determineRoutingConfig;
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    private validateAdditionalWeights;
}
/**
 * A version/weight pair for routing traffic to Lambda functions
 */
export interface VersionWeight {
    /**
     * The version to route traffic to
     */
    readonly version: IVersion;
    /**
     * How much weight to assign to this version (0..1)
     */
    readonly weight: number;
}
