# -*- coding: utf-8 -*-
# File: identifier.py

# Copyright 2021 Dr. Janis Meyer. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Methods for generating and checking uuids
"""
import hashlib
import uuid

from .detection_types import Pathlike

__all__ = ["is_uuid_like", "get_uuid_from_str", "get_uuid"]


def is_uuid_like(input_id: str) -> bool:
    """
    Check, if input string has uuid3 string representation format.

    **Example:**

            "886313e1-3b8a-5372-9b90-0c9aee199e5d"

    :param input_id: An input string
    :return: A boolean output
    """
    try:
        uuid.UUID(str(input_id))
        return True
    except ValueError:
        return False


def get_uuid_from_str(input_id: str) -> str:
    """
    Returns an uuid3 string representation generated from an input string.

    :param input_id:
    :return: uuid3 string representation
    """
    return str(uuid.uuid3(uuid.NAMESPACE_DNS, input_id))


def get_uuid(*inputs: str) -> str:
    """
    Set an uuid generated by the concatenation of string inputs.

    :param inputs: string inputs
    :return: uuid3 string representation
    """
    str_input = "".join(inputs)
    return get_uuid_from_str(str_input)


def get_md5_hash(path: Pathlike, buffer_size: int = 65536) -> str:
    """
    Calculate a md5 hash for a given file

    :param path: path to a file
    :param buffer_size: Will calculate the hash in chunks
    :return: md5 string
    """

    hash_md5 = hashlib.md5()

    with open(path, "rb") as file:
        for chunk in iter(lambda: file.read(buffer_size), b""):
            hash_md5.update(chunk)

    return hash_md5.hexdigest()
