"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_1 = require("graphql");
const resources_1 = require("./resources");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_transformer_common_2 = require("graphql-transformer-common");
const definitions_1 = require("./definitions");
const HTTP_STACK_NAME = 'HttpStack';
class HttpTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('HttpTransformer', graphql_transformer_core_1.gql `
        directive @http(method: HttpMethod = GET, url: String!, headers: [HttpHeader] = []) on FIELD_DEFINITION
        enum HttpMethod {
          GET
          POST
          PUT
          DELETE
          PATCH
        }
        input HttpHeader {
          key: String
          value: String
        }
      `);
        this.before = (ctx) => {
            let directiveList = [];
            for (const def of ctx.inputDocument.definitions) {
                if (def.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION) {
                    for (const field of def.fields) {
                        const httpDirective = field.directives.find(dir => dir.name.value === 'http');
                        if (httpDirective) {
                            directiveList.push(httpDirective);
                        }
                    }
                }
            }
            directiveList.forEach((value) => {
                const url = graphql_transformer_common_1.getDirectiveArgument(value, 'url');
                const protocolMatcher = /^http(s)?:\/\//;
                if (!protocolMatcher.test(url)) {
                    throw new graphql_transformer_core_1.TransformerContractError(`@http directive at location ${value.loc.start} ` + `requires a url parameter that begins with http:// or https://.`);
                }
                const baseURL = url.replace(HttpTransformer.urlRegex, '$1');
                const dataSourceID = graphql_transformer_common_2.HttpResourceIDs.HttpDataSourceID(baseURL);
                if (!ctx.getResource(dataSourceID)) {
                    ctx.mapResourceToStack(HTTP_STACK_NAME, dataSourceID);
                    ctx.setResource(dataSourceID, this.resources.makeHttpDataSource(baseURL));
                }
            });
        };
        this.field = (parent, field, directive, ctx) => {
            ctx.mapResourceToStack(HTTP_STACK_NAME, graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value));
            const url = graphql_transformer_common_1.getDirectiveArgument(directive, 'url');
            const baseURL = url.replace(HttpTransformer.urlRegex, '$1');
            let path = url.split(/(http(s)?:\/\/|www\.)|(\/.*)/g).slice(-2, -1)[0];
            let urlParams = path.match(/:\w+/g);
            let queryBodyArgsArray = field.arguments;
            let newFieldArgsArray = [];
            if (urlParams) {
                urlParams = urlParams.map(p => p.replace(':', ''));
                queryBodyArgsArray = field.arguments.filter(e => graphql_transformer_common_1.isScalar(e.type) && !urlParams.includes(e.name.value));
                path = path.replace(/:\w+/g, (str) => {
                    return `\$\{ctx.args.params.${str.replace(':', '')}\}`;
                });
                const urlParamInputObject = definitions_1.makeUrlParamInputObject(parent, field, urlParams);
                ctx.addInput(urlParamInputObject);
                newFieldArgsArray.push(definitions_1.makeHttpArgument('params', urlParamInputObject, true));
            }
            let method = graphql_transformer_common_1.getDirectiveArgument(directive, 'method');
            if (!method) {
                method = 'GET';
            }
            let headers = graphql_transformer_common_1.getDirectiveArgument(directive, 'headers');
            if (!headers || !Array.isArray(headers)) {
                headers = [];
            }
            if (queryBodyArgsArray.length > 0) {
                const queryInputObject = definitions_1.makeHttpQueryInputObject(parent, field, queryBodyArgsArray, method === 'GET' ? false : true);
                const bodyInputObject = definitions_1.makeHttpBodyInputObject(parent, field, queryBodyArgsArray, true);
                const makeNonNull = queryInputObject.fields.filter(a => a.type.kind === graphql_1.Kind.NON_NULL_TYPE).length > 0 ? true : false;
                ctx.addInput(queryInputObject);
                newFieldArgsArray.push(definitions_1.makeHttpArgument('query', queryInputObject, makeNonNull));
                if (method !== 'GET' && method !== 'DELETE') {
                    ctx.addInput(bodyInputObject);
                    newFieldArgsArray.push(definitions_1.makeHttpArgument('body', bodyInputObject, makeNonNull));
                }
            }
            switch (method) {
                case 'GET':
                    const getResourceID = graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value);
                    if (!ctx.getResource(getResourceID)) {
                        const getResolver = this.resources.makeGetResolver(baseURL, path, parent.name.value, field.name.value, headers);
                        ctx.setResource(getResourceID, getResolver);
                    }
                    break;
                case 'POST':
                    const postResourceID = graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value);
                    if (!ctx.getResource(postResourceID)) {
                        const postResolver = this.resources.makePostResolver(baseURL, path, parent.name.value, field.name.value, queryBodyArgsArray.filter(a => a.type.kind === graphql_1.Kind.NON_NULL_TYPE).map(a => a.name.value), headers);
                        ctx.setResource(postResourceID, postResolver);
                    }
                    break;
                case 'PUT':
                    const putResourceID = graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value);
                    if (!ctx.getResource(putResourceID)) {
                        const putResolver = this.resources.makePutResolver(baseURL, path, parent.name.value, field.name.value, queryBodyArgsArray.filter(a => a.type.kind === graphql_1.Kind.NON_NULL_TYPE).map(a => a.name.value), headers);
                        ctx.setResource(putResourceID, putResolver);
                    }
                    break;
                case 'DELETE':
                    const deleteResourceID = graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value);
                    if (!ctx.getResource(deleteResourceID)) {
                        const deleteResolver = this.resources.makeDeleteResolver(baseURL, path, parent.name.value, field.name.value, headers);
                        ctx.setResource(deleteResourceID, deleteResolver);
                    }
                    break;
                case 'PATCH':
                    const patchResourceID = graphql_transformer_common_2.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value);
                    if (!ctx.getResource(patchResourceID)) {
                        const patchResolver = this.resources.makePatchResolver(baseURL, path, parent.name.value, field.name.value, queryBodyArgsArray.filter(a => a.type.kind === graphql_1.Kind.NON_NULL_TYPE).map(a => a.name.value), headers);
                        ctx.setResource(patchResourceID, patchResolver);
                    }
                    break;
                default:
            }
            if (newFieldArgsArray.length > 0) {
                const updatedField = {
                    ...field,
                    arguments: newFieldArgsArray,
                };
                const mostRecentParent = ctx.getType(parent.name.value);
                let updatedFieldsInParent = mostRecentParent.fields.filter(f => f.name.value !== field.name.value);
                updatedFieldsInParent.push(updatedField);
                const updatedParentType = {
                    ...mostRecentParent,
                    fields: updatedFieldsInParent,
                };
                ctx.putType(updatedParentType);
            }
        };
        this.resources = new resources_1.ResourceFactory();
    }
}
exports.HttpTransformer = HttpTransformer;
HttpTransformer.urlRegex = /(http(s)?:\/\/)|(\/.*)/g;
//# sourceMappingURL=HttpTransformer.js.map