# Copyright (c) 2020 Edwin Wise
# MIT License
# See LICENSE for details
"""
    Buttons are widgets that are pressed. They typically update their function
    on release, and the contents can be as simple as a label and as complex
    as a multivalued graph.
"""
import logging
import uuid
from typing import Tuple, Optional

import pygame
from custom_inherit import doc_inherit

from pi_touch_gui._utilities import snapped_value_string
from pi_touch_gui._widget_bases import (FONT_NAME, IButtonWidget,
                                        IWidgetInterface)

LOG = logging.getLogger(__name__)


class Button(IButtonWidget):
    """ A simple rectangular button that calls its function on release.

    It is an empty instantiation of IButtonWidget.
    """
    # Button style
    RECTANGLE = 0
    LEFT_ROUND = 1
    RIGHT_ROUND = 2
    OBROUND = 3

    # Button behavior
    CLICK = 0
    HOLD = 1
    TOGGLE = 2

    @doc_inherit(IButtonWidget.__init__, style='numpy_with_merge')
    def __init__(self, position, size,
                 style=None,
                 behavior=None,
                 **kwargs):
        """ Initialize the Button.

        Parameters
        ----------
        style : Int
            The style in which the button is rendered (Button.RECTANGLE is the
            default, but you can also specify LEFT_ROUND, RIGHT_ROUND, or both
            with OBROUND)
        behavior : Int
            The behavior of the button (Button.CLICK is the default, activating
            on release, but you can specify HOLD which activates on press
            and stops on release, or TOGGLE which turns on and off on each
            release).
        """
        super(Button, self).__init__(position, size, **kwargs)
        self.style = Button.RECTANGLE if style is None else style
        self.behavior = Button.CLICK if behavior is None else behavior
        self.active = False

    @doc_inherit(IButtonWidget.render, style='numpy_with_merge')
    def render(self, surface):
        """ Render the button.

        Complicated by the need for one or two rounded ends.  If you play
        your parameters right, you get a circle!
        """
        color, label_color = self.state_colors(self.active)

        radius = int(self.h / 2)
        inset_x = radius if self.style & self.LEFT_ROUND else 0
        inset_w = radius if self.style & self.RIGHT_ROUND else 0
        inset_w = inset_w + inset_x
        subrect = ((self.x + inset_x, self.y),
                   (self.w - inset_w, self.h))
        pygame.draw.rect(surface, color, subrect, 0)
        if self.style & self.LEFT_ROUND:
            pygame.draw.circle(surface, color,
                               (self.x + radius, self.y + radius),
                               radius)
        if self.style & self.RIGHT_ROUND:
            pygame.draw.circle(surface, color,
                               (self.x + self.w - radius, self.y + radius),
                               radius)

        self.render_centered_text(surface, self.label, label_color, color)

        if self._focus:
            self.render_focus(surface)

    def set(self):
        if self.behavior == Button.TOGGLE:
            self.active = True
            if self.function:
                return self.function(self, self.active)

    def clear(self):
        if self.behavior == Button.TOGGLE:
            self.active = False
            if self.function:
                return self.function(self, self.active)

    def on_press(self, touch) -> None:
        """ Hold behavior calls their function on the press event.

        Parameters
        ----------
        touch : Touch
            Touch object; generally not used for buttons
        """
        self.press()

    def press(self):
        if self.behavior == self.HOLD:
            self.active = True
            if self.function:
                self.function(self, self.active)
        super(Button, self).press()

    def on_release(self, touch):
        return self.release()

    def release(self):
        if self.behavior == self.TOGGLE:
            self.active = not self.active
        else:
            self.active = False
        return super(Button, self).release()


class GraphButton(IButtonWidget):
    """ A button with a graph as the body.

        Dynamic content is generated by the `update` method, and the `function`
        is called on release.
    """
    LABEL_INSET = 5

    @doc_inherit(IButtonWidget.__init__, style='numpy_with_merge')
    def __init__(self, position, size,
                 quiet=None,
                 update=None,
                 function=None,
                 **kwargs):
        """ Initialize the graph button.

        Parameters
        ----------
        quiet : Bool
            If set and True, quiets the graph (doesn't print the values)
        update : Callable[[IWidget, int], List[dict]]
            The update is called with the number of samples being requested.
            It should return `data` with that many samples or fewer.  If it
            returns fewer than 2 samples, that trace may not be rendered.
            The dictionary in the list defines the color of the trace and
            a list of float with the sample data:
                {"color": <Color>, "data": [<float>]}
        function : Callable[[IWidget], Optional['Page']]
            Function to call on button release. Optionally returns a new Page,
            to change pages.
        """
        super(GraphButton, self).__init__(position, size,
                                          function=function,
                                          **kwargs)
        self.quiet = quiet
        self.update = update
        if self.update is None:
            # Without an update method, we need a fixed font size
            self.graph_font = pygame.font.Font(FONT_NAME,
                                               int(self.h / 2.5 + 0.5))
        else:
            # Determine how many samples the update method returns, so we can
            # scale the font appropriately.
            samples = self.update(self, 1)
            ratio_h = max(2.5, len(samples))
            self.graph_font = pygame.font.Font(FONT_NAME,
                                               int(self.h / ratio_h + 0.5))

    @doc_inherit(IButtonWidget.render, style='numpy_with_merge')
    def render(self, surface):
        """ Render the button and graph.

        The label is rendered at the left side of the graph, and then a
        rectangle is drawn to the right of the label. The rectangle width
        determines how many data samples we ask for.  The graph(s) are then
        drawn inside of the rectangle, with the last values in the graph
        data rendered as labels to the right (but inside) of the graph.
        """
        color, label_color = self.state_colors()

        text = self.font.render(self.label, 1, label_color)
        textpos = text.get_rect()
        textpos.left = self.x  # + self.LABEL_INSET
        textpos.centery = self.y + (self.h / 2)

        text_width = textpos.w + self.LABEL_INSET  # * 2
        text_size = (text_width, self.h + 1)
        pygame.draw.rect(surface, color, (self.position, text_size))

        graph_pos = (self.x + text_width, self.y)
        graph_size = (self.w - text_width, self.h)
        pygame.draw.rect(surface, color, (graph_pos, graph_size), 2)

        surface.blit(text, textpos)

        # We can only draw the graph(s) if there is an update method
        if self.update is None:
            return

        values = self.update(self, graph_size[0] - 2)
        min_val = 9999
        max_val = -9999
        # Determine the global min/max to scale the vertical range
        for entry in values:
            data = entry["data"]
            if len(data) > 0:
                min_val = entry.get("min", min(min_val, min(data)))
                max_val = entry.get("min", max(max_val, max(data)))
            else:
                min_val = 0
                max_val = 0

        range = max_val - min_val
        # Range must be > 0 so adjust around the existing min/max
        if range < .01:
            min_val -= .01
            max_val += .01
            range = max_val - min_val

        for entry in values:
            color = entry["color"]
            data = entry["data"]

            def scale_y_value(value):
                """ Scale the value to fit the graph box."""
                inset_range = range * 1.1
                center_val = value - (range / 2.0)
                return self.y + (self.h / 2.0
                                 - ((center_val - min_val) / inset_range)
                                 * self.h)

            # Need two data points to start the graph
            if len(data) > 1:
                prev_y = scale_y_value(data[0])
                prev_x = graph_pos[0]
                for val in data:
                    x = prev_x + 1
                    y = scale_y_value(val)
                    pygame.draw.line(surface, color, (prev_x, prev_y), (x, y),
                                     2)
                    prev_x = x
                    prev_y = y

                # Render the graph trailing value
                if self.quiet is not True:
                    text = self.graph_font.render(
                        snapped_value_string(data[-1], 0.1),
                        1,
                        color)
                    textpos = text.get_rect()
                    textpos.right = (self.x + self.w) - self.LABEL_INSET
                    # The text tracks the graph line, unless it's outside the
                    # extents, then we clamp it to inside.
                    if (prev_y + textpos.h + self.LABEL_INSET) > (
                            self.y + self.h):
                        textpos.bottom = prev_y - self.LABEL_INSET
                    elif (prev_y - textpos.h - self.LABEL_INSET) < (self.y):
                        textpos.top = prev_y + self.LABEL_INSET
                    else:
                        textpos.centery = prev_y
                    surface.blit(text, textpos)

        if self._focus:
            self.render_focus(surface)


class ButtonGroup(IWidgetInterface):
    """ A group of ToggleButton where zero or one may be active.

    When a button is made active, all the remaining are
    forced to be inactive.

    It is possible for none of the buttons to be active, if you
    de-activate the active button.

    Starts with the first button in the list active.
    """

    @doc_inherit(IWidgetInterface.__init__, style='numpy_with_merge')
    def __init__(self, toggle_buttons, widget_id=None):
        """ Initialize the button group, and hook the button functions.

        Parameters
        ----------
        toggle_buttons : List[ToggleButton]
            An ordered list of ToggleButton objects, where the first button
            is active on startup.
        widget_id : str
            Optional ID for this group, if not given, gets a random uuid.
        """
        self.label = 'group'
        self._id = str(uuid.uuid4()) if widget_id is None else widget_id
        self._neighbors = [None, None, None, None]

        self._widgets = list()
        for idx, button in enumerate(toggle_buttons):
            if not isinstance(button, Button):
                raise ValueError(f"Item {idx} {type(button)!r} "
                                 f"is not a Button")
            if button.behavior != Button.TOGGLE:
                raise ValueError(f"Button {idx} is not a TOGGLE")

            # Insert the group control method into the button
            old_function = button.function
            button.function = self.group_function
            self._widgets.append((button, old_function))

    @property
    def neighbors(self):
        return self._neighbors

    def group_function(self, button, state):
        """ Enforce the group behavior; one button on, all others off.

        Parameters
        ----------
        button : ToggleButton
            The button that is receiving the event
        state : bool
            The state being sent to the button

        Returns
        -------
        Optional['Page']
            Returns whatever the button's function returns.  The disabled
            button results are ignored.
        """
        # If we are enabling this button, disable all other buttons
        for b, f in self._widgets:
            if b.id != button.id:
                b.active = False
                if f:
                    f(b, False)

        # Now the actual button press
        f = next(f for b, f in self._widgets if b.id == button.id)
        if f:
            return f(button, state)
        return None

    @property
    def id(self):
        return self._id

    def event(self, event, touch) -> Tuple[bool, Optional['Page']]:
        for button, _ in self._widgets:
            consumed, new_page = button.event(event, touch)
            if consumed:
                return True, new_page
        return False, None

    def can_focus(self) -> bool:
        return False

    def render(self, surface):
        for button, _ in self._widgets:
            button.render(surface)

    def render_focus(self, surface):
        pass
