# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['starlite_sessions']

package_data = \
{'': ['*']}

install_requires = \
['starlite>=1.17.1']

setup_kwargs = {
    'name': 'starlite-sessions',
    'version': '1.1.0',
    'description': 'Starlite Sessions Toolkit',
    'long_description': '# Starlite Sessions\n\n<!-- markdownlint-disable -->\n<img alt="Starlite logo" src="./starlite-banner.svg" width="100%" height="auto">\n<!-- markdownlint-restore -->\n\n<div align="center">\n\n[![Quality Gate Status](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_starlite-sessions&metric=alert_status)](https://sonarcloud.io/summary/new_code?id=starlite-api_starlite-sessions)\n[![Coverage](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_starlite-sessions&metric=coverage)](https://sonarcloud.io/summary/new_code?id=starlite-api_starlite-sessions)\n\n[![Maintainability Rating](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_starlite-sessions&metric=sqale_rating)](https://sonarcloud.io/summary/new_code?id=starlite-api_starlite-sessions)\n[![Security Rating](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_starlite-sessions&metric=security_rating)](https://sonarcloud.io/summary/new_code?id=starlite-api_starlite-sessions)\n[![Reliability Rating](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_starlite-sessions&metric=reliability_rating)](https://sonarcloud.io/summary/new_code?id=starlite-api_starlite-sessions)\n[![Code Smells](https://sonarcloud.io/api/project_badges/measure?project=starlite-api_starlite-sessions&metric=code_smells)](https://sonarcloud.io/summary/new_code?id=starlite-api_starlite-sessions)\n\n[![Discord](https://img.shields.io/discord/919193495116337154?color=blue&label=chat%20on%20discord&logo=discord)](https://discord.gg/X3FJqy8d2j)\n[![Matrix](https://img.shields.io/badge/%5Bm%5D%20chat%20on%20Matrix-bridged-blue)](https://matrix.to/#/#starlitespace:matrix.org)\n\n</div>\n\nThis library offers simple session based authentication for [Starlite](https://github.com/starlite-api/starlite).\n\nCheckout [the docs📚](https://starlite-api.github.io/starlite-sessions/).\n\n## Installation\n\n```shell\npip install starlite-sessions\n```\n\n## Example\n\n```python\nimport os\nfrom typing import Any, Optional, Literal\n\nfrom pydantic import BaseModel, EmailStr, SecretStr\nfrom starlite import OpenAPIConfig, Request, Starlite, get, post\n\nfrom starlite_sessions import SessionAuth\n\n\n# Let\'s assume we have a User model that is a pydantic model.\n# This though is not required - we need some sort of user class -\n# but it can be any arbitrary value, e.g. an SQLAlchemy model,\n# a representation of a MongoDB  etc.\nclass User(BaseModel):\n    id: str\n    name: str\n    email: EmailStr\n\n\n# we also have pydantic types for two different\n# kinds of POST request bodies: one for creating\n# a user, e.g. "sign-up", and the other for logging\n# an existing user in.\nclass UserCreatePayload(BaseModel):\n    name: str\n    email: EmailStr\n    password: SecretStr\n\n\nclass UserLoginPayload(BaseModel):\n    email: EmailStr\n    password: SecretStr\n\n\n# The SessionAuth class requires a handler callable\n# that takes the session dictionary, and returns the\n# \'User\' instance correlating to it.\n#\n# The session dictionary itself is a value the user decides\n# upon. So for example, it might be a simple dictionary\n# that holds a user id, for example: { "user_id": "abcd123" }\n#\n# Note: The callable can be either sync or async - both will work.\nasync def retrieve_user_handler(session: dict[str, Any]) -> Optional[User]:\n    # insert logic here to retrieve the user instance based on the session data.\n    ...\n\n\n# The minimal configuration required by the library is the\n# callable for the \'retrieve_user_handler\', and a bytes value for\n# the secret used to encrypt the session cookie.\n#\n# Important: secrets should never be hardcoded, and its considered\n# best practice to inject secrets via env.\n#\n# Important: the secret should be a 16, 24 or 32\n# characters long byte string (128/192/256 bit).\n#\n# Tip: It\'s also a good idea to use the pydantic settings\n# management functionality.\nsession_auth = SessionAuth(\n    retrieve_user_handler=retrieve_user_handler,\n    secret=os.environ.get("JWT_SECRET", os.urandom(16)),\n    # exclude any URLs that should not have authentication.\n    # We exclude the documentation URLs, signup and login.\n    exclude=["/login", "/signup", "/schema"],\n)\n\n\n@post("/login")\nasync def login(data: UserLoginPayload, request: Request) -> User:\n    # we received log-in data via post.\n    # out login handler should retrieve from persistence (a db etc.)\n    # the user data and verify that the login details\n    # are correct. If we are using passwords, we should check that\n    # the password hashes match etc. We will simply assume that we\n    # have done all of that we now have a user value:\n    user: User = ...\n\n    # once verified we can create a session.\n    # to do this we simply need to call the Starlite\n    # \'Request.set_session\' method, which accepts either dictionaries\n    # or pydantic models. In our case, we can simply record a\n    # simple dictionary with the user ID value:\n    request.set_session({"user_id": user.id})\n\n    # you can do whatever we want here. In this case, we will simply return the user data:\n    return user\n\n\n@post("/signup")\nasync def signup(data: UserCreatePayload, request: Request) -> User:\n    # this is similar to the login handler, except here we should\n    # insert into persistence - after doing whatever extra\n    # validation we might require. We will assume that this is done,\n    # and we now have a user instance with an assigned ID value:\n    user: User = ...\n\n    # we are creating a session the same as we do in the\n    # \'login_handler\' above:\n    request.set_session({"user_id": user.id})\n\n    # and again, you can add whatever logic is required here, we\n    # will simply return the user:\n    return user\n\n\n# the endpoint below requires the user to be already authenticated\n# to be able to access it.\n@get("/user")\ndef get_user(request: Request[User, dict[Literal["user_id"], str]]) -> Any:\n    # because this route requires authentication, we can access\n    # `request.user`, which is the authenticated user returned\n    # by the \'retrieve_user_handler\' function we passed to SessionAuth.\n    return request.user\n\n\n# We add the session security schema to the OpenAPI config.\nopenapi_config = OpenAPIConfig(\n    title="My API",\n    version="1.0.0",\n    components=[session_auth.openapi_components],\n    security=[session_auth.security_requirement],\n)\n\n# We initialize the app instance, passing to it the\n# \'jwt_auth.middleware\' and the \'openapi_config\'.\napp = Starlite(\n    route_handlers=[login, signup, get_user],\n    middleware=[session_auth.middleware],\n    openapi_config=openapi_config,\n)\n```\n\n## Contributing\n\nStarlite and all its official libraries are open to contributions big and small.\n\nYou can always [join our discord](https://discord.gg/X3FJqy8d2j) server\nor [join our Matrix](https://matrix.to/#/#starlitespace:matrix.org) space to discuss contributions and project\nmaintenance. For guidelines on how to contribute to this library, please see `CONTRIBUTING.md` in the repository\'s root.\n',
    'author': "Na'aman Hirschfeld",
    'author_email': 'nhirschfeld@gmail.com',
    'maintainer': "Na'aman Hirschfeld",
    'maintainer_email': 'nhirschfeld@gmail.com',
    'url': 'https://github.com/starlite-api/starlite-sessions',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
