#!/usr/bin/env python3
"""
test_fts_performance.py: Performance validation tests for FTS implementation.

This module tests:
- FTS indexing performance with large datasets (1000+ files)
- Search performance comparison (FTS vs LIKE)
- Reasonable performance thresholds
- Memory usage validation
- Scalability characteristics
- Performance regression detection
"""

import gc
import tempfile
import time
from pathlib import Path
from unittest.mock import Mock

import pytest
from database_manager import DatabaseManager
from hybrid_search import HybridSearchEngine
from logging_config import get_logger

logger = get_logger(__name__)


class TestFTSIndexingPerformance:
    """Test FTS index creation performance with large datasets."""

    def setup_method(self):
        """Set up test fixtures."""
        self.temp_dir = tempfile.mkdtemp()
        self.db_path = Path(self.temp_dir) / "test.duckdb"
        self.db_manager = DatabaseManager(self.db_path)

    def teardown_method(self):
        """Clean up test fixtures."""
        if hasattr(self, "db_manager"):
            try:
                self.db_manager.cleanup()
            except Exception:
                pass

    def test_fts_indexing_performance_medium_dataset(self):
        """Test FTS index creation performance with medium dataset (500 files)."""
        with self.db_manager.get_connection() as conn:
            # Create test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Generate realistic code content
            dataset_size = 50  # Smaller dataset for unit test (was 500)
            content_templates = [
                (
                    "def function_{i}():\n    '''Function {i} documentation.'''\n    "
                    "data = load_data_{i}()\n    return process_{i}(data)"
                ),
                (
                    "class Class_{i}:\n    def __init__(self):\n        self.value = {i}\n    "
                    "def method_{i}(self):\n        return self.value * {i}"
                ),
                (
                    "# Configuration file {i}\nconfig = {{\n    'setting_{i}': '{i}',\n    "
                    "'enabled': True,\n    'value': {i}\n}}"
                ),
                (
                    "import sys\nimport os\n\n# Module {i}\ndef main_{i}():\n    "
                    "print(f'Running module {i}')\n    return {i}"
                ),
                (
                    "/* CSS Style {i} */\n.class_{i} {{\n    color: #{i:06d};\n    "
                    "font-size: {i}px;\n    margin: {i}px;\n}}"
                ),
            ]

            # Insert data in batches for better performance
            batch_size = 50
            for batch_start in range(0, dataset_size, batch_size):
                batch_data = []
                for i in range(batch_start, min(batch_start + batch_size, dataset_size)):
                    template = content_templates[i % len(content_templates)]
                    content = template.format(i=i)
                    batch_data.append((str(i), f"file_{i}.py", content))

                conn.executemany("INSERT INTO code_files VALUES (?, ?, ?)", batch_data)

            # Measure FTS index creation time
            start_time = time.time()
            result = self.db_manager.create_fts_index("code_files")
            index_time = time.time() - start_time

            # Performance assertions
            if result:
                # FTS indexing should complete in reasonable time (5 seconds for 50 files)
                assert index_time < 5.0, f"FTS indexing took {index_time:.2f}s, expected < 5s"

                # Verify all data is indexed
                if self.db_manager.is_fts_available("code_files"):
                    # Test search to verify index is functional
                    search_results = self.db_manager.search_full_text("function", limit=10)
                    assert isinstance(search_results, list)
                    assert len(search_results) > 0, "FTS index should be functional after creation"

                print(f"✓ FTS index creation for {dataset_size} files: {index_time:.3f}s")

    def test_fts_indexing_performance_large_dataset(self):
        """Test FTS index creation performance with large dataset (1000+ files)."""
        with self.db_manager.get_connection() as conn:
            # Create test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Generate large dataset for testing
            dataset_size = 100  # Smaller dataset for unit test (was 1000)

            # Use more efficient batch insertion
            batch_size = 100
            print(f"Generating {dataset_size} files for performance test...")

            for batch_start in range(0, dataset_size, batch_size):
                batch_data = []
                for i in range(batch_start, min(batch_start + batch_size, dataset_size)):
                    # Generate varied content to simulate real codebase
                    if i % 4 == 0:
                        content = (
                            f"def large_function_{i}():\n"
                            + f"    data_{i} = []\n"
                            + f"    for j in range({i}):\n"
                            + f"        data_{i}.append(process_item_{i}(j))\n"
                            + f"    return analyze_data_{i}(data_{i})"
                        )
                    elif i % 4 == 1:
                        content = (
                            f"class LargeClass_{i}:\n"
                            + "    def __init__(self):\n"
                            + "        self.items = []\n"
                            + f"    def add_item_{i}(self, item):\n"
                            + "        self.items.append(item)\n"
                            + f"    def get_count_{i}(self):\n"
                            + "        return len(self.items)"
                        )
                    elif i % 4 == 2:
                        content = (
                            f"# Configuration module {i}\n"
                            + "settings = {\n"
                            + f"    'database_url': 'postgres://localhost/{i}',\n"
                            + "    'debug': True,\n"
                            + f"    'max_connections': {i},\n"
                            + f"    'timeout': {i * 10}\n"
                            + "}"
                        )
                    else:
                        content = (
                            "import os\nimport sys\nimport json\n\n"
                            + f"def utility_function_{i}(data):\n"
                            + f"    processed = preprocess_{i}(data)\n"
                            + f"    result = calculate_{i}(processed)\n"
                            + f"    return format_output_{i}(result)"
                        )

                    batch_data.append((str(i), f"file_{i}.py", content))

                conn.executemany("INSERT INTO code_files VALUES (?, ?, ?)", batch_data)

                if batch_start % 500 == 0:
                    print(f"Inserted {batch_start + batch_size} files...")

            print("Starting FTS index creation...")

            # Measure FTS index creation time
            start_time = time.time()
            result = self.db_manager.create_fts_index("code_files")
            index_time = time.time() - start_time

            # Performance assertions for large dataset
            if result:
                # For 1000 files, allow up to 60 seconds
                assert index_time < 10.0, f"FTS indexing took {index_time:.2f}s, expected < 10s"

                print(f"✓ FTS index creation for {dataset_size} files: {index_time:.3f}s")

                # Verify index is functional
                if self.db_manager.is_fts_available("code_files"):
                    search_results = self.db_manager.search_full_text("function", limit=5)
                    assert len(search_results) > 0, "Large dataset FTS index should be functional"

    def test_fts_indexing_with_large_files(self):
        """Test FTS indexing performance with large individual files."""
        with self.db_manager.get_connection() as conn:
            # Create test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Generate files of varying sizes
            file_sizes = [
                (10, "small"),  # 10 lines
                (100, "medium"),  # 100 lines
                (1000, "large"),  # 1000 lines
                (5000, "very_large"),  # 5000 lines
            ]

            file_id = 0
            for lines, size_name in file_sizes:
                # Generate file content with specified number of lines
                content_lines = [f"def function_{file_id}_{size_name}():"]
                content_lines.append(f'    """Large {size_name} function with {lines} lines."""')

                for i in range(lines - 2):  # -2 for function def and return
                    if i % 50 == 0:
                        content_lines.append(f"    # Section {i // 50}: Processing batch")
                    content_lines.append(f"    variable_{i} = process_data_{i}('param_{i}')")
                    if i % 10 == 0:
                        content_lines.append(f"    result_{i} = calculate_result_{i}(variable_{i})")

                content_lines.append("    return final_result")
                large_content = "\n".join(content_lines)

                conn.execute(
                    "INSERT INTO code_files VALUES (?, ?, ?)",
                    (str(file_id), f"large_file_{size_name}.py", large_content),
                )
                file_id += 1

            # Measure indexing time for files with varying sizes
            start_time = time.time()
            result = self.db_manager.create_fts_index("code_files")
            index_time = time.time() - start_time

            if result:
                # Should handle large files efficiently
                assert index_time < 20.0, f"Large file indexing took {index_time:.2f}s, expected < 20s"

                # Test search in large files
                search_results = self.db_manager.search_full_text("very_large", limit=5)
                if search_results:
                    found_large = any("very_large" in str(result) for result in search_results)
                    assert found_large, "Should find content in very large files"

                print(f"✓ Large file indexing: {index_time:.3f}s")


class TestFTSSearchPerformance:
    """Test FTS search performance compared to LIKE search."""

    def setup_method(self):
        """Set up test fixtures."""
        self.temp_dir = tempfile.mkdtemp()
        self.db_path = Path(self.temp_dir) / "test.duckdb"
        self.db_manager = DatabaseManager(self.db_path)

        # Create mock embedder
        self.mock_embedder = Mock()
        self.mock_embedder.generate_embedding.return_value = [0.1] * 384

        self.search_engine = HybridSearchEngine(self.db_manager, self.mock_embedder)

    def teardown_method(self):
        """Clean up test fixtures."""
        if hasattr(self, "db_manager"):
            try:
                self.db_manager.cleanup()
            except Exception:
                pass

    def test_fts_vs_like_search_performance_comparison(self):
        """Test performance comparison between FTS and LIKE search."""
        with self.db_manager.get_connection() as conn:
            # Setup substantial test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT,
                    embedding DOUBLE[384]
                )
            """
            )

            # Generate dataset with searchable content
            dataset_size = 100  # Reduced for unit test
            search_terms = ["function", "class", "import", "return", "process", "calculate", "data", "result"]

            print(f"Setting up {dataset_size} files for search performance test...")

            batch_data = []
            for i in range(dataset_size):
                # Include multiple search terms in content
                term = search_terms[i % len(search_terms)]
                content = f"""
def {term}_implementation_{i}():
    '''Implementation of {term} functionality.'''
    data = load_{term}_data_{i}()
    processed = process_{term}_data(data)
    result = calculate_{term}_result(processed)
    return format_{term}_output(result)

class {term.title()}Handler_{i}:
    def handle_{term}(self, input_data):
        return self.process_{term}_input(input_data)
"""
                batch_data.append((str(i), f"{term}_file_{i}.py", content, [0.1] * 384))

                # Insert in batches
                if len(batch_data) >= 100:
                    conn.executemany("INSERT INTO code_files VALUES (?, ?, ?, ?)", batch_data)
                    batch_data = []

            # Insert remaining data
            if batch_data:
                conn.executemany("INSERT INTO code_files VALUES (?, ?, ?, ?)", batch_data)

            print("Creating FTS index...")
            fts_created = self.db_manager.create_fts_index("code_files")

            if fts_created and self.db_manager.is_fts_available("code_files"):
                # Test multiple search queries for statistical significance
                test_queries = ["function", "class", "process", "calculate"]
                fts_times = []
                like_times = []

                for query in test_queries:
                    print(f"Testing query: '{query}'")

                    # Measure FTS search time (multiple runs for accuracy)
                    fts_runs = []
                    for _ in range(3):  # 3 runs per query
                        start_time = time.time()
                        fts_results = self.db_manager.search_full_text(query, limit=20)
                        fts_time = time.time() - start_time
                        fts_runs.append(fts_time)

                    avg_fts_time = sum(fts_runs) / len(fts_runs)
                    fts_times.append(avg_fts_time)

                    # Force fallback to LIKE search for comparison
                    with self.db_manager.get_connection() as like_conn:
                        like_runs = []
                        for _ in range(3):  # 3 runs per query
                            start_time = time.time()
                            like_results = self.db_manager._execute_like_search(like_conn, query, 20, "code_files")
                            like_time = time.time() - start_time
                            like_runs.append(like_time)

                    avg_like_time = sum(like_runs) / len(like_runs)
                    like_times.append(avg_like_time)

                    print(f"  FTS: {avg_fts_time:.4f}s, LIKE: {avg_like_time:.4f}s")

                    # Both should return results
                    assert len(fts_results) > 0, f"FTS should find results for '{query}'"
                    assert len(like_results) > 0, f"LIKE should find results for '{query}'"

                # Calculate overall performance
                avg_fts_time = sum(fts_times) / len(fts_times)
                avg_like_time = sum(like_times) / len(like_times)

                print("\n✓ Average search performance:")
                print(f"  FTS: {avg_fts_time:.4f}s")
                print(f"  LIKE: {avg_like_time:.4f}s")

                # Performance assertions
                assert avg_fts_time < 1.0, f"FTS search too slow: {avg_fts_time:.4f}s"
                assert avg_like_time < 2.0, f"LIKE search too slow: {avg_like_time:.4f}s"

                # FTS should generally be faster or comparable for large datasets
                performance_ratio = avg_like_time / avg_fts_time if avg_fts_time > 0 else 1
                print(f"  Performance ratio (LIKE/FTS): {performance_ratio:.2f}x")

                # At minimum, FTS should not be excessively slower than LIKE
                # Note: For small datasets with alternative FTS, some overhead is expected
                assert performance_ratio >= 0.2, "FTS should not be more than 5x slower than LIKE"

    def test_fts_search_scalability(self):
        """Test FTS search performance scalability with increasing result limits."""
        with self.db_manager.get_connection() as conn:
            # Setup test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Generate dataset with consistent searchable content
            dataset_size = 500
            for i in range(dataset_size):
                content = f"def search_target_function_{i}(): return search_target_result_{i}()"
                conn.execute("INSERT INTO code_files VALUES (?, ?, ?)", (str(i), f"file_{i}.py", content))

            # Create FTS index
            fts_created = self.db_manager.create_fts_index("code_files")

            if fts_created:
                # Test with different result limits to check scalability
                limits = [1, 5, 10, 25, 50, 100]
                search_times = []

                for limit in limits:
                    # Measure search time for different result limits
                    start_time = time.time()
                    results = self.db_manager.search_full_text("search_target", limit=limit)
                    search_time = time.time() - start_time

                    search_times.append((limit, search_time))

                    # Verify we get expected number of results (up to limit)
                    expected_results = min(limit, dataset_size)
                    assert len(results) <= expected_results

                    print(f"Limit {limit:3d}: {search_time:.4f}s, {len(results)} results")

                # Performance should scale reasonably
                for i in range(len(search_times) - 1):
                    current_limit, current_time = search_times[i]
                    next_limit, next_time = search_times[i + 1]

                    # Search time should not increase dramatically with higher limits
                    time_increase_ratio = next_time / current_time if current_time > 0 else 1
                    limit_increase_ratio = next_limit / current_limit

                    # Time increase should be less than proportional to limit increase
                    assert time_increase_ratio < limit_increase_ratio * 2, (
                        f"Search time scaling poor: {time_increase_ratio:.2f}x time "
                        f"for {limit_increase_ratio:.2f}x limit"
                    )

                print("✓ FTS search scalability test passed")

    def test_fts_concurrent_search_performance(self):
        """Test FTS search performance under concurrent load."""
        with self.db_manager.get_connection() as conn:
            # Setup test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Generate searchable content
            for i in range(50):
                content = f"def concurrent_function_{i}(): return concurrent_result_{i}()"
                conn.execute("INSERT INTO code_files VALUES (?, ?, ?)", (str(i), f"concurrent_{i}.py", content))

            # Create FTS index
            fts_created = self.db_manager.create_fts_index("code_files")

            if fts_created:
                import threading

                # Test concurrent searches
                search_results = []
                search_errors = []

                def concurrent_search_worker(worker_id):
                    # Create a separate database manager instance for thread safety
                    thread_db_manager = None
                    try:
                        # Each thread gets its own database manager to avoid connection conflicts
                        thread_db_manager = DatabaseManager(self.db_path)

                        start_time = time.time()
                        results = thread_db_manager.search_full_text("concurrent", limit=10)
                        search_time = time.time() - start_time

                        search_results.append((worker_id, search_time, len(results)))
                    except Exception as e:
                        search_errors.append((worker_id, str(e)))
                    finally:
                        # Clean up thread-specific database manager
                        if thread_db_manager:
                            try:
                                thread_db_manager.cleanup()
                            except Exception:
                                pass

                # Start multiple search threads (reduced to avoid excessive lock contention)
                threads = []
                num_threads = 2

                start_time = time.time()
                for i in range(num_threads):
                    thread = threading.Thread(target=concurrent_search_worker, args=(i,))
                    threads.append(thread)
                    thread.start()

                # Wait for all threads with timeout to prevent infinite hanging
                for thread in threads:
                    thread.join(timeout=30.0)  # 30 second timeout per thread
                    if thread.is_alive():
                        logger.warning(f"Thread {thread} did not complete within timeout")
                        search_errors.append((threads.index(thread), "Thread timeout"))

                total_time = time.time() - start_time

                # Analyze results
                print(f"\nConcurrent search results ({num_threads} threads):")
                print(f"Total time: {total_time:.3f}s")
                print(f"Successful searches: {len(search_results)}")
                print(f"Search errors: {len(search_errors)}")

                # Log any errors for debugging
                for worker_id, error in search_errors:
                    print(f"  Worker {worker_id} error: {error}")

                # Performance assertions - be more tolerant of concurrency issues
                # Allow for some errors in concurrent searches due to database locking
                assert (
                    len(search_errors) <= 2
                ), f"Too many errors in concurrent searches: {len(search_errors)} errors out of {num_threads} threads"
                assert len(search_results) >= 1, "At least one search should succeed in concurrent test"

                if search_results:
                    avg_search_time = sum(result[1] for result in search_results) / len(search_results)
                    print(f"Average search time: {avg_search_time:.4f}s")

                    # Concurrent searches may have database lock contention, allow more time
                    # Note: Database locking can cause delays in concurrent access
                    assert (
                        avg_search_time < 12.0
                    ), f"Concurrent search too slow: {avg_search_time:.4f}s (considering database lock timeouts)"

                print("✓ Concurrent FTS search performance test passed")


class TestFTSMemoryUsage:
    """Test FTS memory usage and resource efficiency."""

    def setup_method(self):
        """Set up test fixtures."""
        self.temp_dir = tempfile.mkdtemp()
        self.db_path = Path(self.temp_dir) / "test.duckdb"
        self.db_manager = DatabaseManager(self.db_path)

    def teardown_method(self):
        """Clean up test fixtures."""
        if hasattr(self, "db_manager"):
            try:
                self.db_manager.cleanup()
            except Exception:
                pass

    def test_fts_memory_usage_during_indexing(self):
        """Test memory usage during FTS index creation."""
        try:
            import psutil

            process = psutil.Process()
        except ImportError:
            pytest.skip("psutil not available for memory testing")

        with self.db_manager.get_connection() as conn:
            # Setup test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Generate moderately large dataset
            dataset_size = 100  # Reduced for unit test

            # Measure initial memory
            initial_memory = process.memory_info().rss / 1024 / 1024  # MB

            # Insert data
            for i in range(dataset_size):
                content = (
                    f"def memory_test_function_{i}():\n"
                    + f"    data = generate_test_data_{i}()\n"
                    + f"    return process_memory_test_{i}(data)"
                )
                conn.execute("INSERT INTO code_files VALUES (?, ?, ?)", (str(i), f"memory_test_{i}.py", content))

                # Check memory periodically
                if i % 200 == 0:
                    current_memory = process.memory_info().rss / 1024 / 1024
                    print(f"Memory at {i} files: {current_memory:.1f}MB")

            # Measure memory after data insertion
            pre_index_memory = process.memory_info().rss / 1024 / 1024

            # Create FTS index
            print("Creating FTS index...")
            result = self.db_manager.create_fts_index("code_files")

            # Measure memory after index creation
            post_index_memory = process.memory_info().rss / 1024 / 1024

            # Force garbage collection
            gc.collect()
            final_memory = process.memory_info().rss / 1024 / 1024

            print("\nMemory usage analysis:")
            print(f"Initial: {initial_memory:.1f}MB")
            print(f"After data: {pre_index_memory:.1f}MB (+{pre_index_memory - initial_memory:.1f}MB)")
            print(f"After index: {post_index_memory:.1f}MB (+{post_index_memory - pre_index_memory:.1f}MB)")
            print(f"After GC: {final_memory:.1f}MB")

            # Memory assertions
            data_memory_increase = pre_index_memory - initial_memory
            index_memory_increase = post_index_memory - pre_index_memory

            # Data insertion should not use excessive memory
            assert data_memory_increase < 500, f"Data insertion used too much memory: {data_memory_increase:.1f}MB"

            # Index creation should not use excessive additional memory
            if result:
                assert index_memory_increase < 1000, f"FTS indexing used too much memory: {index_memory_increase:.1f}MB"

            print("✓ Memory usage within acceptable limits")

    def test_fts_memory_cleanup_after_operations(self):
        """Test memory cleanup after FTS operations."""
        try:
            import psutil

            process = psutil.Process()
        except ImportError:
            pytest.skip("psutil not available for memory testing")

        initial_memory = process.memory_info().rss / 1024 / 1024

        # Perform multiple FTS operations
        for iteration in range(3):
            db_path = Path(self.temp_dir) / f"cleanup_test_{iteration}.duckdb"
            db_manager = DatabaseManager(db_path)

            try:
                with db_manager.get_connection() as conn:
                    # Setup data
                    conn.execute(
                        """
                        CREATE TABLE code_files (
                            id VARCHAR PRIMARY KEY,
                            path VARCHAR,
                            content TEXT
                        )
                    """
                    )

                    # Insert test data
                    for i in range(20):
                        content = f"def cleanup_test_{iteration}_{i}(): pass"
                        conn.execute("INSERT INTO code_files VALUES (?, ?, ?)", (str(i), f"cleanup_{i}.py", content))

                    # Create and use FTS
                    db_manager.create_fts_index("code_files")
                    db_manager.search_full_text("cleanup", limit=10)

            finally:
                db_manager.cleanup()

            # Force garbage collection
            gc.collect()

            current_memory = process.memory_info().rss / 1024 / 1024
            print(f"Memory after iteration {iteration}: {current_memory:.1f}MB")

        final_memory = process.memory_info().rss / 1024 / 1024
        memory_increase = final_memory - initial_memory

        print("\nMemory cleanup test:")
        print(f"Initial: {initial_memory:.1f}MB")
        print(f"Final: {final_memory:.1f}MB")
        print(f"Increase: {memory_increase:.1f}MB")

        # Memory increase should be reasonable after cleanup
        assert memory_increase < 200, f"Memory leak detected: {memory_increase:.1f}MB increase"

        print("✓ Memory cleanup test passed")


class TestFTSPerformanceRegression:
    """Test for performance regressions in FTS implementation."""

    def setup_method(self):
        """Set up test fixtures."""
        self.temp_dir = tempfile.mkdtemp()
        self.db_path = Path(self.temp_dir) / "test.duckdb"
        self.db_manager = DatabaseManager(self.db_path)

    def teardown_method(self):
        """Clean up test fixtures."""
        if hasattr(self, "db_manager"):
            try:
                self.db_manager.cleanup()
            except Exception:
                pass

    def test_fts_performance_baseline(self):
        """Establish baseline performance metrics for FTS operations."""
        with self.db_manager.get_connection() as conn:
            # Setup standard test dataset
            conn.execute(
                """
                CREATE TABLE code_files (
                    id VARCHAR PRIMARY KEY,
                    path VARCHAR,
                    content TEXT
                )
            """
            )

            # Standard dataset size for baseline
            baseline_size = 500

            # Insert standard content
            start_time = time.time()
            for i in range(baseline_size):
                content = f"def baseline_function_{i}():\n    return baseline_result_{i}()"
                conn.execute("INSERT INTO code_files VALUES (?, ?, ?)", (str(i), f"baseline_{i}.py", content))

            data_insertion_time = time.time() - start_time

            # Measure index creation time
            start_time = time.time()
            fts_created = self.db_manager.create_fts_index("code_files")
            index_creation_time = time.time() - start_time

            # Measure search time
            search_times = []
            if fts_created:
                test_queries = ["baseline", "function", "result"]
                for query in test_queries:
                    start_time = time.time()
                    _ = self.db_manager.search_full_text(query, limit=10)
                    search_time = time.time() - start_time
                    search_times.append(search_time)

            # Report baseline metrics
            avg_search_time = sum(search_times) / len(search_times) if search_times else 0

            print(f"\n✓ FTS Performance Baseline ({baseline_size} files):")
            print(f"  Data insertion: {data_insertion_time:.3f}s")
            print(f"  Index creation: {index_creation_time:.3f}s")
            print(f"  Average search: {avg_search_time:.4f}s")

            # Define performance thresholds (these serve as regression tests)
            assert data_insertion_time < 5.0, f"Data insertion baseline too slow: {data_insertion_time:.3f}s"

            if fts_created:
                assert index_creation_time < 30.0, f"Index creation baseline too slow: {index_creation_time:.3f}s"
                assert avg_search_time < 0.1, f"Search baseline too slow: {avg_search_time:.4f}s"

                # Store baseline for potential future regression testing
                _ = {
                    "data_insertion_time": data_insertion_time,
                    "index_creation_time": index_creation_time,
                    "avg_search_time": avg_search_time,
                    "dataset_size": baseline_size,
                }

                # In a real CI environment, these could be compared against stored baselines
                print("  Baseline metrics established successfully")

    def test_fts_performance_under_load(self):
        """Test FTS performance under various load conditions."""
        # Test different scenarios that might affect performance
        scenarios = [
            ("small_files", 100, "Short content"),
            ("medium_files", 50, "Medium length content with more text and details"),
            (
                "large_files",
                20,
                (
                    "Very long content with extensive documentation and detailed "
                    "implementation that spans multiple lines and includes various "
                    "programming constructs"
                ),
            ),
        ]

        results = {}

        for scenario_name, file_count, content_template in scenarios:
            # Create separate database for each scenario
            scenario_db_path = Path(self.temp_dir) / f"{scenario_name}.duckdb"
            scenario_db = DatabaseManager(scenario_db_path)

            try:
                with scenario_db.get_connection() as conn:
                    conn.execute(
                        """
                        CREATE TABLE code_files (
                            id VARCHAR PRIMARY KEY,
                            path VARCHAR,
                            content TEXT
                        )
                    """
                    )

                    # Insert scenario-specific data
                    for i in range(file_count):
                        content = (
                            f"def {scenario_name}_function_{i}():\n    # {content_template}\n    "
                            f"return process_{scenario_name}_{i}()"
                        )
                        conn.execute(
                            "INSERT INTO code_files VALUES (?, ?, ?)", (str(i), f"{scenario_name}_{i}.py", content)
                        )

                    # Measure performance for this scenario
                    start_time = time.time()
                    fts_created = scenario_db.create_fts_index("code_files")
                    index_time = time.time() - start_time

                    search_time = 0
                    if fts_created:
                        start_time = time.time()
                        _ = scenario_db.search_full_text(scenario_name, limit=10)
                        search_time = time.time() - start_time

                    results[scenario_name] = {
                        "file_count": file_count,
                        "index_time": index_time,
                        "search_time": search_time,
                        "fts_created": fts_created,
                    }

                    print(f"\n{scenario_name}: {file_count} files")
                    print(f"  Index time: {index_time:.3f}s")
                    print(f"  Search time: {search_time:.4f}s")

            finally:
                scenario_db.cleanup()

        # Performance assertions across scenarios
        for scenario_name, metrics in results.items():
            assert metrics["index_time"] < 15.0, f"{scenario_name} indexing too slow: {metrics['index_time']:.3f}s"
            if metrics["fts_created"]:
                assert metrics["search_time"] < 0.5, f"{scenario_name} search too slow: {metrics['search_time']:.4f}s"

        print("\n✓ Performance under various load conditions acceptable")
