# -*- coding: utf-8 -*-
# Copyright 2023 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from typing import Awaitable, Callable, Dict, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import gapic_v1, grpc_helpers_async
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.protobuf import empty_pb2  # type: ignore
import grpc  # type: ignore
from grpc.experimental import aio  # type: ignore

from google.cloud.iap_v1.types import service

from .base import DEFAULT_CLIENT_INFO, IdentityAwareProxyOAuthServiceTransport
from .grpc import IdentityAwareProxyOAuthServiceGrpcTransport


class IdentityAwareProxyOAuthServiceGrpcAsyncIOTransport(
    IdentityAwareProxyOAuthServiceTransport
):
    """gRPC AsyncIO backend transport for IdentityAwareProxyOAuthService.

    API to programmatically create, list and retrieve Identity
    Aware Proxy (IAP) OAuth brands; and create, retrieve, delete and
    reset-secret of IAP OAuth clients.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _grpc_channel: aio.Channel
    _stubs: Dict[str, Callable] = {}

    @classmethod
    def create_channel(
        cls,
        host: str = "iap.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> aio.Channel:
        """Create and return a gRPC AsyncIO channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            aio.Channel: A gRPC AsyncIO channel object.
        """

        return grpc_helpers_async.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    def __init__(
        self,
        *,
        host: str = "iap.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        channel: Optional[aio.Channel] = None,
        api_mtls_endpoint: Optional[str] = None,
        client_cert_source: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        ssl_channel_credentials: Optional[grpc.ChannelCredentials] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            channel (Optional[aio.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
            google.auth.exceptions.MutualTlsChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None
        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @property
    def grpc_channel(self) -> aio.Channel:
        """Create the channel designed to connect to this service.

        This property caches on the instance; repeated calls return
        the same channel.
        """
        # Return the channel from cache.
        return self._grpc_channel

    @property
    def list_brands(
        self,
    ) -> Callable[[service.ListBrandsRequest], Awaitable[service.ListBrandsResponse]]:
        r"""Return a callable for the list brands method over gRPC.

        Lists the existing brands for the project.

        Returns:
            Callable[[~.ListBrandsRequest],
                    Awaitable[~.ListBrandsResponse]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_brands" not in self._stubs:
            self._stubs["list_brands"] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/ListBrands",
                request_serializer=service.ListBrandsRequest.serialize,
                response_deserializer=service.ListBrandsResponse.deserialize,
            )
        return self._stubs["list_brands"]

    @property
    def create_brand(
        self,
    ) -> Callable[[service.CreateBrandRequest], Awaitable[service.Brand]]:
        r"""Return a callable for the create brand method over gRPC.

        Constructs a new OAuth brand for the project if one
        does not exist. The created brand is "internal only",
        meaning that OAuth clients created under it only accept
        requests from users who belong to the same Google
        Workspace organization as the project. The brand is
        created in an un-reviewed status. NOTE: The "internal
        only" status can be manually changed in the Google Cloud
        Console. Requires that a brand does not already exist
        for the project, and that the specified support email is
        owned by the caller.

        Returns:
            Callable[[~.CreateBrandRequest],
                    Awaitable[~.Brand]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_brand" not in self._stubs:
            self._stubs["create_brand"] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/CreateBrand",
                request_serializer=service.CreateBrandRequest.serialize,
                response_deserializer=service.Brand.deserialize,
            )
        return self._stubs["create_brand"]

    @property
    def get_brand(
        self,
    ) -> Callable[[service.GetBrandRequest], Awaitable[service.Brand]]:
        r"""Return a callable for the get brand method over gRPC.

        Retrieves the OAuth brand of the project.

        Returns:
            Callable[[~.GetBrandRequest],
                    Awaitable[~.Brand]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_brand" not in self._stubs:
            self._stubs["get_brand"] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/GetBrand",
                request_serializer=service.GetBrandRequest.serialize,
                response_deserializer=service.Brand.deserialize,
            )
        return self._stubs["get_brand"]

    @property
    def create_identity_aware_proxy_client(
        self,
    ) -> Callable[
        [service.CreateIdentityAwareProxyClientRequest],
        Awaitable[service.IdentityAwareProxyClient],
    ]:
        r"""Return a callable for the create identity aware proxy
        client method over gRPC.

        Creates an Identity Aware Proxy (IAP) OAuth client.
        The client is owned by IAP. Requires that the brand for
        the project exists and that it is set for internal-only
        use.

        Returns:
            Callable[[~.CreateIdentityAwareProxyClientRequest],
                    Awaitable[~.IdentityAwareProxyClient]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_identity_aware_proxy_client" not in self._stubs:
            self._stubs[
                "create_identity_aware_proxy_client"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/CreateIdentityAwareProxyClient",
                request_serializer=service.CreateIdentityAwareProxyClientRequest.serialize,
                response_deserializer=service.IdentityAwareProxyClient.deserialize,
            )
        return self._stubs["create_identity_aware_proxy_client"]

    @property
    def list_identity_aware_proxy_clients(
        self,
    ) -> Callable[
        [service.ListIdentityAwareProxyClientsRequest],
        Awaitable[service.ListIdentityAwareProxyClientsResponse],
    ]:
        r"""Return a callable for the list identity aware proxy
        clients method over gRPC.

        Lists the existing clients for the brand.

        Returns:
            Callable[[~.ListIdentityAwareProxyClientsRequest],
                    Awaitable[~.ListIdentityAwareProxyClientsResponse]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_identity_aware_proxy_clients" not in self._stubs:
            self._stubs[
                "list_identity_aware_proxy_clients"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/ListIdentityAwareProxyClients",
                request_serializer=service.ListIdentityAwareProxyClientsRequest.serialize,
                response_deserializer=service.ListIdentityAwareProxyClientsResponse.deserialize,
            )
        return self._stubs["list_identity_aware_proxy_clients"]

    @property
    def get_identity_aware_proxy_client(
        self,
    ) -> Callable[
        [service.GetIdentityAwareProxyClientRequest],
        Awaitable[service.IdentityAwareProxyClient],
    ]:
        r"""Return a callable for the get identity aware proxy
        client method over gRPC.

        Retrieves an Identity Aware Proxy (IAP) OAuth client.
        Requires that the client is owned by IAP.

        Returns:
            Callable[[~.GetIdentityAwareProxyClientRequest],
                    Awaitable[~.IdentityAwareProxyClient]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_identity_aware_proxy_client" not in self._stubs:
            self._stubs[
                "get_identity_aware_proxy_client"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/GetIdentityAwareProxyClient",
                request_serializer=service.GetIdentityAwareProxyClientRequest.serialize,
                response_deserializer=service.IdentityAwareProxyClient.deserialize,
            )
        return self._stubs["get_identity_aware_proxy_client"]

    @property
    def reset_identity_aware_proxy_client_secret(
        self,
    ) -> Callable[
        [service.ResetIdentityAwareProxyClientSecretRequest],
        Awaitable[service.IdentityAwareProxyClient],
    ]:
        r"""Return a callable for the reset identity aware proxy
        client secret method over gRPC.

        Resets an Identity Aware Proxy (IAP) OAuth client
        secret. Useful if the secret was compromised. Requires
        that the client is owned by IAP.

        Returns:
            Callable[[~.ResetIdentityAwareProxyClientSecretRequest],
                    Awaitable[~.IdentityAwareProxyClient]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "reset_identity_aware_proxy_client_secret" not in self._stubs:
            self._stubs[
                "reset_identity_aware_proxy_client_secret"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/ResetIdentityAwareProxyClientSecret",
                request_serializer=service.ResetIdentityAwareProxyClientSecretRequest.serialize,
                response_deserializer=service.IdentityAwareProxyClient.deserialize,
            )
        return self._stubs["reset_identity_aware_proxy_client_secret"]

    @property
    def delete_identity_aware_proxy_client(
        self,
    ) -> Callable[
        [service.DeleteIdentityAwareProxyClientRequest], Awaitable[empty_pb2.Empty]
    ]:
        r"""Return a callable for the delete identity aware proxy
        client method over gRPC.

        Deletes an Identity Aware Proxy (IAP) OAuth client.
        Useful for removing obsolete clients, managing the
        number of clients in a given project, and cleaning up
        after tests. Requires that the client is owned by IAP.

        Returns:
            Callable[[~.DeleteIdentityAwareProxyClientRequest],
                    Awaitable[~.Empty]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_identity_aware_proxy_client" not in self._stubs:
            self._stubs[
                "delete_identity_aware_proxy_client"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.iap.v1.IdentityAwareProxyOAuthService/DeleteIdentityAwareProxyClient",
                request_serializer=service.DeleteIdentityAwareProxyClientRequest.serialize,
                response_deserializer=empty_pb2.Empty.FromString,
            )
        return self._stubs["delete_identity_aware_proxy_client"]

    def close(self):
        return self.grpc_channel.close()


__all__ = ("IdentityAwareProxyOAuthServiceGrpcAsyncIOTransport",)
