# This file is part of the Reproducible and Reusable Data Analysis Workflow
# Server (flowServ).
#
# Copyright (C) 2019-2021 NYU.
#
# flowServ is free software; you can redistribute it and/or modify it under the
# terms of the MIT License; see LICENSE file for more details.

"""The abstract remote client class is used by the remote workflow controller
to interact with a workflow engine. Different workflow engines will implement
their own version of the remote client. The client provides the functionality
that is required by the workflow controller to execute workflows, cancel
workflow execution, get workflow status, and to download workflow result files.
"""

from __future__ import annotations
from abc import ABCMeta, abstractmethod
from dataclasses import dataclass
from typing import Dict, List

from flowserv.model.base import RunObject
from flowserv.model.template.base import WorkflowTemplate
from flowserv.model.workflow.state import WorkflowState
from flowserv.volume.base import StorageVolume


@dataclass
class RemoteWorkflowHandle:
    """Base class for remote workflow handles. Remote workflows may have an
    identifier that differs from the run identifier that is managed by
    flowServ. The remote identifier is maintained by the handle, together with
    the last reported workflow state, and the list of output files that are
    generated by successful workflow runs. The names of output files are path
    that are relative to the root directory of the remote environment that
    executes the workflow.
    """
    # Unique run identifier.
    run_id: str
    # Unique identifier for the remote workflow.
    workflow_id: str
    # Last reported state of the remote forkflow.
    state: WorkflowState
    # Identifier (relative paths) of the generated output files.
    output_files: List[str]
    # Storage volume for run result files.
    runstore: StorageVolume
    # remote client that is used to interact with the workflow engine.
    client: RemoteClient

    def is_active(self) -> bool:
        """Test if the last state of the workflow is an active state.

        Returns
        -------
        bool
        """
        return self.state.is_active()

    def poll_state(self) -> WorkflowState:
        """Get the current state of the workflow from the responsible backend.

        Uses the client to poll the workflow state. Updates the internal state
        if the workflow state has changed.

        If the state of the workflow has not changed the result of this method
        is None. Otherwise, the returned value is the new (current) workflow
        state.

        Returns
        -------
        flowserv.model.workflow.state.WorkflowState
        """
        # Get the current state of the workflow.
        current_state = self.client.get_workflow_state(
            workflow_id=self.workflow_id,
            current_state=self.state
        )
        if current_state == self.state:
            # The workflow state has not changed.
            return None
        # Modifiy the internal state and return the updated workflow state.
        self.state = current_state
        return self.state


class RemoteClient(metaclass=ABCMeta):
    """The remote client class is an abstract interface that defines the
    methods that are required by the remote workflow controller to execute and
    monitor remote workflows. Different workflow engies will implement their
    own version of the interface.
    """
    @abstractmethod
    def create_workflow(
        self, run: RunObject, template: WorkflowTemplate, arguments: Dict,
        staticfs: StorageVolume
    ) -> RemoteWorkflowHandle:
        """Create a new instance of a workflow from the given workflow
        template and user-provided arguments.

        The static storage volume provides access to static workflow template
        files that were created when the workflow template was installed.

        Implementations of this method will also upload any files to the
        remomote engine that are required to execute the workflow.

        A created workflow may not be running immediately but at minimum
        scheduled for execution. There is no separate signal to trigger
        execution start.

        The result is a handle to access the remote workflow object.

        Parameters
        ----------
        run: flowserv.model.base.RunObject
            Handle for the run that is being executed.
        template: flowserv.model.template.base.WorkflowTemplate
            Workflow template containing the parameterized specification and
            the parameter declarations.
        arguments: dict
            Dictionary of argument values for parameters in the template.
        staticfs: flowserv.volume.base.StorageVolume
            Storage volume that contains the static files from the workflow
            template.

        Returns
        -------
        flowserv.controller.remote.client.RemoteWorkflowHandle
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def get_workflow_state(
        self, workflow_id: str, current_state: WorkflowState
    ) -> WorkflowState:
        """Get information about the current state of a given workflow.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier
        current_state: flowserv.model.workflw.state.WorkflowState
            Last known state of the workflow by the workflow controller

        Returns
        -------
        flowserv.model.workflw.state.WorkflowState
        """
        raise NotImplementedError()  # pragma: no cover

    @abstractmethod
    def stop_workflow(self, workflow_id: str):
        """Stop the execution of the workflow with the given identifier.

        Parameters
        ----------
        workflow_id: string
            Unique workflow identifier
        """
        raise NotImplementedError()  # pragma: no cover
