"use strict";
const aws_s3_1 = require("@aws-cdk/aws-s3");
const lib_1 = require("../../core/lib");
const ec2 = require("../lib");
module.exports = {
    'can create Windows user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forWindows();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>command1\ncommand2</powershell>');
        test.done();
    },
    'can create Windows user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'onexit1\n' +
            'onexit2\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'command2\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can create Windows with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addSignalOnExitCommand(resource);
        userData.addCommands('command1');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>trap {\n' +
            '$success=($PSItem.Exception.Message -eq "Success")\n' +
            'cfn-signal --stack Stack --resource RESOURCE1989552F --region ${Token[AWS::Region.4]} --success ($success.ToString().ToLower())\n' +
            'break\n' +
            '}\n' +
            'command1\n' +
            'throw "Success"</powershell>');
        test.done();
    },
    'can windows userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forWindows();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket2', 'test2');
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.bat'
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: 'filename2.bat',
            localFile: 'c:\\test\\location\\otherScript.bat'
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>mkdir (Split-Path -Path \'C:/temp/filename.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test\' -key \'filename.bat\' -file \'C:/temp/filename.bat\' -ErrorAction Stop\n' +
            'mkdir (Split-Path -Path \'c:\\test\\location\\otherScript.bat\' ) -ea 0\n' +
            'Read-S3Object -BucketName \'test2\' -key \'filename2.bat\' -file \'c:\\test\\location\\otherScript.bat\' -ErrorAction Stop</powershell>');
        test.done();
    },
    'can windows userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forWindows();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: 'C:\\test\\filename.bat',
        });
        userData.addExecuteFileCommand({
            filePath: 'C:\\test\\filename2.bat',
            arguments: 'arg1 arg2 -arg $variable'
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '<powershell>&\'C:\\test\\filename.bat\' undefined\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename.bat"\' -ErrorAction Stop }\n' +
            '&\'C:\\test\\filename2.bat\' arg1 arg2 -arg $variable\n' +
            'if (!$?) { Write-Error \'Failed to execute the file "C:\\test\\filename2.bat"\' -ErrorAction Stop }</powershell>');
        test.done();
    },
    'can create Linux user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1', 'command2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\ncommand1\ncommand2');
        test.done();
    },
    'can create Linux user data with commands on exit'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addCommands('command1', 'command2');
        userData.addOnExitCommands('onexit1', 'onexit2');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            'onexit1\n' +
            'onexit2\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1\n' +
            'command2');
        test.done();
    },
    'can create Linux with Signal Command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.forLinux();
        userData.addCommands('command1');
        userData.addSignalOnExitCommand(resource);
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource RESOURCE1989552F --region ${Token[AWS::Region.4]} -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            '}\n' +
            'trap exitTrap EXIT\n' +
            'command1');
        test.done();
    },
    'can linux userdata download S3 files'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.forLinux();
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        const bucket2 = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket2', 'test2');
        // WHEN
        userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.sh'
        });
        userData.addS3DownloadCommand({
            bucket: bucket2,
            bucketKey: 'filename2.sh',
            localFile: 'c:\\test\\location\\otherScript.sh'
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'mkdir -p $(dirname \'/tmp/filename.sh\')\n' +
            'aws s3 cp \'s3://test/filename.sh\' \'/tmp/filename.sh\'\n' +
            'mkdir -p $(dirname \'c:\\test\\location\\otherScript.sh\')\n' +
            'aws s3 cp \'s3://test2/filename2.sh\' \'c:\\test\\location\\otherScript.sh\'');
        test.done();
    },
    'can linux userdata execute files'(test) {
        // GIVEN
        const userData = ec2.UserData.forLinux();
        // WHEN
        userData.addExecuteFileCommand({
            filePath: '/tmp/filename.sh',
        });
        userData.addExecuteFileCommand({
            filePath: '/test/filename2.sh',
            arguments: 'arg1 arg2 -arg $variable'
        });
        // THEN
        const rendered = userData.render();
        test.equals(rendered, '#!/bin/bash\n' +
            'set -e\n' +
            'chmod +x \'/tmp/filename.sh\'\n' +
            '\'/tmp/filename.sh\' undefined\n' +
            'set -e\n' +
            'chmod +x \'/test/filename2.sh\'\n' +
            '\'/test/filename2.sh\' arg1 arg2 -arg $variable');
        test.done();
    },
    'can create Custom user data'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('Some\nmultiline\ncontent');
        // THEN
        const rendered = userData.render();
        test.equals(rendered, 'Some\nmultiline\ncontent');
        test.done();
    },
    'Custom user data throws when adding on exit commands'(test) {
        // GIVEN
        // WHEN
        const userData = ec2.UserData.custom('');
        // THEN
        test.throws(() => userData.addOnExitCommands('a command goes here'));
        test.done();
    },
    'Custom user data throws when adding signal command'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const resource = new ec2.Vpc(stack, 'RESOURCE');
        // WHEN
        const userData = ec2.UserData.custom('');
        // THEN
        test.throws(() => userData.addSignalOnExitCommand(resource));
        test.done();
    },
    'Custom user data throws when downloading file'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const userData = ec2.UserData.custom('');
        const bucket = aws_s3_1.Bucket.fromBucketName(stack, 'testBucket', 'test');
        // WHEN
        // THEN
        test.throws(() => userData.addS3DownloadCommand({
            bucket,
            bucketKey: 'filename.sh'
        }));
        test.done();
    },
    'Custom user data throws when executing file'(test) {
        // GIVEN
        const userData = ec2.UserData.custom('');
        // WHEN
        // THEN
        test.throws(() => userData.addExecuteFileCommand({
            filePath: '/tmp/filename.sh',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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