"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
        }
    }
}
exports.UserData = UserData;
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        const shebang = this.props.shebang !== undefined ? this.props.shebang : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${params.arguments}`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])
        ].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        this.addCommands(`&'${params.filePath}' ${params.arguments}`, `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
//# sourceMappingURL=data:application/json;base64,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