"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * InvalidCidrRangeError is thrown when attempting to perform operations on a CIDR
 * range that is either not valid, or outside of the VPC size limits.
 */
class InvalidCidrRangeError extends Error {
    constructor(cidr) {
        super(cidr + ' is not a valid VPC CIDR range (must be between /16 and /28)');
        // The following line is required for type checking of custom errors, and must be called right after super()
        // https://stackoverflow.com/questions/31626231/custom-error-class-in-typescript
        Object.setPrototypeOf(this, InvalidCidrRangeError.prototype);
    }
}
exports.InvalidCidrRangeError = InvalidCidrRangeError;
/**
 * NetworkUtils contains helpers to work with network constructs (subnets/ranges)
 */
class NetworkUtils {
    /**
     * Validates an IPv4 string
     *
     * returns true of the string contains 4 numbers between 0-255 delimited by
     * a `.` character
     */
    static validIp(ipAddress) {
        const octets = ipAddress.split('.');
        if (octets.length !== 4) {
            return false;
        }
        return octets.map((octet) => parseInt(octet, 10)).
            filter((octet) => octet >= 0 && octet <= 255).length === 4;
    }
    /**
     * Converts a string IPv4 to a number
     *
     * takes an IP Address (e.g. 174.66.173.168) and converts to a number
     * (e.g 2923605416); currently only supports IPv4
     *
     * Uses the formula:
     * (first octet * 256³) + (second octet * 256²) + (third octet * 256) +
     * (fourth octet)
     *
     * @param  {string} the IP address (e.g. 174.66.173.168)
     * @returns {number} the integer value of the IP address (e.g 2923605416)
     */
    static ipToNum(ipAddress) {
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not valid`);
        }
        return ipAddress
            .split('.')
            .reduce((p, c, i) => p + parseInt(c, 10) * 256 ** (3 - i), 0);
    }
    /**
     * Takes number and converts it to IPv4 address string
     *
     * Takes a number (e.g 2923605416) and converts it to an IPv4 address string
     * currently only supports IPv4
     *
     * @param  {number} the integer value of the IP address (e.g 2923605416)
     * @returns {string} the IPv4 address (e.g. 174.66.173.168)
     */
    static numToIp(ipNum) {
        // this all because bitwise math is signed
        let remaining = ipNum;
        const address = new Array();
        for (let i = 0; i < 4; i++) {
            if (remaining !== 0) {
                address.push(Math.floor(remaining / 256 ** (3 - i)));
                remaining = remaining % 256 ** (3 - i);
            }
            else {
                address.push(0);
            }
        }
        const ipAddress = address.join('.');
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not a valid IP Address`);
        }
        return ipAddress;
    }
}
exports.NetworkUtils = NetworkUtils;
/**
 * Creates a network based on a CIDR Block to build contained subnets
 */
class NetworkBuilder {
    /**
     * Create a network using the provided CIDR block
     *
     * No subnets are allocated in the constructor, the maxIpConsumed is set one
     * less than the first IP in the network
     *
     */
    constructor(cidr) {
        /**
         * The list of CIDR blocks for subnets within this network
         */
        this.subnetCidrs = [];
        this.networkCidr = new CidrBlock(cidr);
        this.subnetCidrs = [];
        this.nextAvailableIp = this.networkCidr.minAddress();
    }
    /**
     * Add a subnet to the network and update the maxIpConsumed
     */
    addSubnet(mask) {
        return this.addSubnets(mask, 1)[0];
    }
    /**
     * Add {count} number of subnets to the network and update the maxIpConsumed
     */
    addSubnets(mask, count = 1) {
        if (mask < 16 || mask > 28) {
            throw new InvalidCidrRangeError(`x.x.x.x/${mask}`);
        }
        const maxIp = this.nextAvailableIp + (CidrBlock.calculateNetsize(mask) * count);
        if (this.networkCidr.maxAddress() < maxIp - 1) {
            throw new Error(`${count} of /${mask} exceeds remaining space of ${this.networkCidr.cidr}`);
        }
        const subnets = [];
        for (let i = 0; i < count; i++) {
            const subnet = new CidrBlock(this.nextAvailableIp, mask);
            this.nextAvailableIp = subnet.nextBlock().minAddress();
            this.subnetCidrs.push(subnet);
            subnets.push(subnet);
        }
        return subnets.map((subnet) => (subnet.cidr));
    }
    /**
     * return the CIDR notation strings for all subnets in the network
     */
    get cidrStrings() {
        return this.subnetCidrs.map((subnet) => (subnet.cidr));
    }
    /**
     * Calculates the largest subnet to create of the given count from the
     * remaining IP space
     */
    maskForRemainingSubnets(subnetCount) {
        const remaining = this.networkCidr.maxAddress() - this.nextAvailableIp + 1;
        const ipsPerSubnet = Math.floor(remaining / subnetCount);
        return 32 - Math.floor(Math.log2(ipsPerSubnet));
    }
}
exports.NetworkBuilder = NetworkBuilder;
/**
 * A block of IP address space with a given bit prefix
 */
class CidrBlock {
    constructor(ipAddressOrCidr, mask) {
        if (typeof ipAddressOrCidr === 'string') {
            this.mask = parseInt(ipAddressOrCidr.split('/')[1], 10);
            this.networkAddress = NetworkUtils.ipToNum(ipAddressOrCidr.split('/')[0]) +
                CidrBlock.calculateNetsize(this.mask) - 1;
        }
        else {
            if (typeof mask === 'number') {
                this.mask = mask;
            }
            else {
                // this should be impossible
                this.mask = 16;
            }
            this.networkAddress = ipAddressOrCidr + CidrBlock.calculateNetsize(this.mask) - 1;
            this.networkSize = 2 ** (32 - this.mask);
        }
        this.networkSize = 2 ** (32 - this.mask);
        this.cidr = `${this.minIp()}/${this.mask}`;
    }
    /**
     * Calculates the netmask for a given CIDR mask
     *
     * For example:
     * CidrBlock.calculateNetmask(24) returns '255.255.255.0'
     */
    static calculateNetmask(mask) {
        return NetworkUtils.numToIp(2 ** 32 - 2 ** (32 - mask));
    }
    /**
     * Calculates the number IP addresses in a CIDR Mask
     *
     * For example:
     * CidrBlock.calculateNetsize(24) returns 256
     */
    static calculateNetsize(mask) {
        return 2 ** (32 - mask);
    }
    /*
     * The maximum IP in the CIDR Blcok e.g. '10.0.8.255'
     */
    maxIp() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return NetworkUtils.numToIp(this.maxAddress());
    }
    /*
     * The minimum IP in the CIDR Blcok e.g. '10.0.0.0'
     */
    minIp() {
        return NetworkUtils.numToIp(this.minAddress());
    }
    /*
     * Returns the number representation for the minimum IPv4 address
     */
    minAddress() {
        const div = this.networkAddress % this.networkSize;
        return this.networkAddress - div;
    }
    /*
     * Returns the number representation for the maximum IPv4 address
     */
    maxAddress() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return this.minAddress() + this.networkSize - 1;
    }
    /*
     * Returns the next CIDR Block of the same mask size
     */
    nextBlock() {
        return new CidrBlock(this.maxAddress() + 1, this.mask);
    }
    /*
     * Returns true if this CidrBlock fully contains the provided CidrBlock
     */
    containsCidr(other) {
        return (this.maxAddress() >= other.maxAddress()) &&
            (this.minAddress() <= other.minAddress());
    }
}
exports.CidrBlock = CidrBlock;
//# sourceMappingURL=data:application/json;base64,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