"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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