"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * NAT providers
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
class NatProvider {
    /**
     * Use NAT Gateways to provide NAT services for your VPC
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     */
    static gateway() {
        return new NatGateway();
    }
    /**
     * Use NAT instances to provide NAT services for your VPC
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     */
    static instance(props) {
        return new NatInstance(props);
    }
}
exports.NatProvider = NatProvider;
class NatGateway extends NatProvider {
    constructor() {
        super(...arguments);
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        // Create the NAT gateways
        for (const sub of options.natSubnets) {
            const gateway = sub.addNatGateway();
            this.gateways.add(sub.availabilityZone, gateway.ref);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
class NatInstance extends NatProvider {
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        const sg = new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
        });
        sg.connections.allowFromAnyIpv4(port_1.Port.allTcp());
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: sg,
                role,
                keyName: this.props.keyName
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
}
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * Machine image representing the latest NAT instance image
 *
 * @experimental
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
//# sourceMappingURL=data:application/json;base64,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