from typing import Any, Dict, Optional

from pyspark.sql import DataFrame as SparkDataFrame
from pyspark.sql import SparkSession

from pytalog.base.data_sources.data_source import DataSource, WriteableDataSource
from pytalog.spark.data_sources.guarantee import guarantee_spark


class SparkSqlSource(DataSource[SparkDataFrame]):
    def __init__(
        self,
        query: str,
        spark_session: Optional[SparkSession] = None,
    ) -> None:
        """Use Spark SQL as an input data source.

        Args:
            query (str): The query to run to get the data.
            spark_session (Optional[SparkSession]): The current SparkSession. Defaults to fetching
                the current active session.
        """
        super().__init__()
        self.query = query
        self.spark_session = guarantee_spark(spark_session)

    def read(self) -> SparkDataFrame:
        """Execute the query stored in this object.

        Returns:
            SparkDataFrame: The DataFrame generated by the query.
        """
        return self.spark_session.sql(self.query)


class SparkFileSource(WriteableDataSource[SparkDataFrame]):
    def __init__(
        self,
        path: str,
        format: str,
        spark_session: Optional[SparkSession] = None,
        read_args: Optional[Dict[str, Any]] = None,
        write_args: Optional[Dict[str, Any]] = None,
        mode: str = "overwrite",
    ) -> None:
        """A DataSource for reading files into Spark.

        Args:
            path (str): The path to the file.
            format (str): The format of the file. SparkSession.read should support this. Examples
                include: csv, json, parquet, delta (only on DataBricks).
            spark_session (Optional[SparkSession], optional): The current SparkSession.
                If not provided, we'll fetch the acive session.
            read_args (Optional[Dict[str, Any]]): Keyword arguments to be passed to `read.load`.
                By default no args are passed.
            write_args (Optional[Dict[str, Any]]): Keyword arguments to be passed to `write.save`.
                By default no args are passed.
            mode (str): Write mode of Spark. Defaults to overwrite.
        """
        super().__init__()
        self.spark_session = guarantee_spark(spark_session)

        self.read_args = {} if read_args is None else read_args
        self.write_args = {} if write_args is None else write_args

        self.mode = mode
        self.path = path
        self.format = format

    def read(self) -> SparkDataFrame:
        """Read the configured file into Spark.

        Returns:
            SparkDataFrame: The DataFrame generated by the query.
        """
        return self.spark_session.read.format(self.format).load(path=self.path, **self.read_args)

    def write(self, data: SparkDataFrame) -> None:
        """Writes the given data to the given file.

        Args:
            data (SparkDataFrame): The Spark DataFrame to be written.
        """
        writer = data.write.format(self.format).mode(self.mode).option("mergeSchema", True)
        if self.mode == "overwrite":
            # Even if the mode is set to overwrite, schemas are still not overwritten unless this is set.
            writer = writer.option("overwriteSchema", True)
        writer.save(self.path, **self.write_args)
