"""
The purpose of this module is to make user database data readily writable in bw2.
Activities and exchanges are organized in a graph-like fashion (Activities being the
nodes, Activity producing the reference flow being the root node, Exchanges being the
edges) to ease LCA scope-dependent features such as parameter propagation or Activity
duplication.
"""

from __future__ import annotations

import itertools
import re
from numbers import Number
from typing import Dict, List, Optional, Tuple, Union

import pydantic
from sympy import parse_expr

from appabuild.database.bw_databases import BwDatabase
from appabuild.database.serialized_data import (
    ActivityIdentifier,
    SerializedActivity,
    SerializedExchange,
    Switch,
    SwitchOption,
)
from appabuild.exceptions import SerializedDataError

# Switch and SwitchOption must be imported despite being unused for
# UserDatabaseContext typing.
from appabuild.logger import logger


class UserDatabaseContext(pydantic.BaseModel):
    """
    Gathers all context information necessary for generating activities.
    """

    serialized_activities: List[SerializedActivity]
    """All serialized activities necessary to build activity tree with reference flow
    activity as a root."""
    activities: List[Activity]
    """Activities already produced, necessary to update an activity's uuid and name in
    case its being used several times. Activities used several times should be
    duplicated with different uuid as it could have a different parameterization
    between different instances, and with different names as it could become independent
    ImpactModel nodes."""
    database: BwDatabase
    "UserDatabase that is being created and populated."


class DatabaseElement(pydantic.BaseModel):
    """
    Contains common attributes between Activity and Exchange.
    """

    name: str
    "Name of the activity/exchange."
    type: str
    "According to Brightway, can be production, technosphere, or biosphere."
    amount: Optional[Number] = None
    "Amount of output flow generated for given amount of exchanges."  # TODO comment c'est utilisé ?
    comment: Optional[str] = None
    "Free text for any context information about the activity/exchange."
    context: UserDatabaseContext
    """Gathers all context information necessary for generating activities. Typically
    generated by UserDatabase object."""

    class Config:
        arbitrary_types_allowed = True


class Exchange(DatabaseElement):
    """Exchange are connection between an input and an output dataset."""

    input: Union[Activity, ActivityIdentifier]
    """Input/downstream activity. If from background database, will be of type
    ActivityIdentifier."""
    output: Activity
    "Output/upstream activity."
    formula: Optional[str] = None
    "Scipy-compliant formula to dynamically determine quantity function of parameters."
    parameters_matching: Optional[Dict[str, Union[str, Dict[str, Number], Number]]] = {}
    """Name or values of input's parameters can be dynamically changed. Key is the name
    of the input parameter's name to change, and value the replacing variable. A float
    will set the parameter to a fixed value, str will update parameter's name or affect
    it to a formula, and dict is used to fix value of a categorical parameter."""

    def to_bw_format(self) -> Dict:
        """
        Converts an Exchange to a format importable in a Brightway database
        :return: dictionary with all fields compatible with Brightway representation of
        Exchange.
        """
        exchange = {
            "input": (self.input.database, self.input.code),
            "output": (self.output.database, self.output.code),
            "name": self.name,
            "type": self.type,
            "amount": self.amount,
        }
        if self.formula is not None:
            # Handling formula separately otherwise None gets transformed into str
            exchange["formula"] = self.formula
        return exchange

    @classmethod
    def from_serialized_exchange(
        cls,
        serialized_exchange: SerializedExchange,
        context: UserDatabaseContext,
        calling_activity: Activity,
    ) -> List[Exchange]:
        """
        Generates an Exchange from a SerializedExchange object, after completing
        following steps:
        - Resolve input if necessary, i.e. determine its uuid
        - Resolve switch, i.e. transform a SerializedExchange with Switch to a list of
        SerializedExchange without Switch
        - Generate a new Activity for each Exchange which input database is equal to
        context database
        :param serialized_exchange: information required to generate Exchange
        :param context: gathers all context information necessary for generating
        activities. Typically generated by UserDatabase object.
        :param calling_activity: Activity containing the Exchange (should be Exchange's
        output)
        :return: a list of Exchange, as resolving a Switch can lead to several Exchanges
        """
        # Resolve switches
        solved_serialized_exchanges = serialized_exchange.resolve_switch()
        solved_exchanges = []
        exchange_output = calling_activity
        # Find activity in background databases if uuid are not provided for each
        # exchange's input
        for solved_serialized_exchange in solved_serialized_exchanges:
            if isinstance(solved_serialized_exchange.input, ActivityIdentifier):
                if solved_serialized_exchange.input.is_unresolved:
                    resolved_activity_identifier = BwDatabase(
                        name=solved_serialized_exchange.input.database
                    ).resolve_activity_identifier(solved_serialized_exchange.input)
                    solved_serialized_exchange.input = resolved_activity_identifier

        # For each solved SerializedExchange which input's database is context database,
        # generate a new Activity
        for solved_serialized_exchange in solved_serialized_exchanges:
            if solved_serialized_exchange.input.database != context.database.name:
                exchange_input = solved_serialized_exchange.input
            else:
                if solved_serialized_exchange.input.uuid != exchange_output.code:
                    serialized_input = [
                        serialized_activity
                        for serialized_activity in context.serialized_activities
                        if serialized_activity.uuid
                        == solved_serialized_exchange.input.uuid
                    ]
                    if len(serialized_input) != 1:
                        raise ValueError(
                            f"Cannot find a unique serialized activity with uuid "
                            f"{solved_serialized_exchange.input.uuid} (found "
                            f"{len(serialized_input)})."
                        )
                    serialized_input = serialized_input[0]
                    if solved_serialized_exchange.use_exchange_name:
                        serialized_input.name = solved_serialized_exchange.name
                    exchange_input = Activity.from_serialized_activity(
                        serialized_input, context
                    )
                else:
                    exchange_input = exchange_output
            solved_exchanges.append(
                Exchange(
                    name=solved_serialized_exchange.name,
                    type=solved_serialized_exchange.type,
                    input=exchange_input,
                    output=exchange_output,
                    formula=(
                        solved_serialized_exchange.amount
                        if isinstance(solved_serialized_exchange.amount, str)
                        else None
                    ),
                    amount=(
                        solved_serialized_exchange.amount
                        if not isinstance(solved_serialized_exchange.amount, str)
                        else 0
                    ),
                    parameters_matching=solved_serialized_exchange.parameters_matching,
                    context=context,
                )
            )
        return solved_exchanges

    def replace_parameters(
        self, parameters_matching: Dict[str, Union[str, Dict[str, Number], Number]]
    ):
        """
        Replace Exchange's parameters matching with parameters_matching keys by
        parameters matching values. Values can be constant, formulas or parameter name.
        :param parameters_matching: keys should be Exchange parameter to update, and
        values should be replacing value.
        A float will set the parameter to a fixed value, str will update parameter's
        name or affect it to a formula, and dict is used to fix value of a categorical
        parameter.
        :return:
        """
        logger.info(
            f"Parameters replacement; exchange {self.name}; parameters matching is "
            f"{parameters_matching}."
        )
        if self.formula is not None:
            try:
                formula = parse_expr(self.formula)
            except AttributeError as e:
                raise SerializedDataError(
                    f"Invalid amount for exchange {self.name}: {e}"
                )
            for param_to_replace, param_replacing in parameters_matching.items():
                if isinstance(param_replacing, dict):
                    all_oh_params = {
                        str(free_symbol): 0
                        for free_symbol in formula.free_symbols
                        if str(free_symbol).startswith(param_to_replace)
                    }
                    all_oh_params.update(
                        {
                            f"{param_to_replace}_{key}": value
                            for key, value in param_replacing.items()
                        }
                    )
                    for (
                        oh_param_to_replace,
                        oh_param_replacing,
                    ) in all_oh_params.items():
                        if oh_param_to_replace in {
                            str(free_symbol) for free_symbol in formula.free_symbols
                        }:
                            formula = formula.subs(
                                oh_param_to_replace, oh_param_replacing
                            )
                            logger.info(
                                f"Parameters replacement; exchange {self.name}; "
                                f"replacing {oh_param_to_replace} by "
                                f"{oh_param_replacing} in formula."
                            )
                else:
                    if param_to_replace in {
                        str(free_symbol) for free_symbol in formula.free_symbols
                    }:
                        try:
                            formula = formula.subs(param_to_replace, param_replacing)
                        except TypeError as e:
                            raise SerializedDataError(
                                f"Invalid amount for exchange {self.name}: {e}"
                            )
                        logger.info(
                            f"Parameters replacement; exchange {self.name}; replacing "
                            f"{param_to_replace} by {param_replacing} in formula."
                        )
            self.formula = str(formula)
        self.update_parameters_matching(parameters_matching)

    def propagate_parameters(self, context: UserDatabaseContext):
        """
        Propagate Exchange's parameter_matching to input activity's exchanges.
        :param context: gathers all context information necessary for generating
        activities. Typically generated by UserDatabase object.
        :return:
        """
        if self.input.database == context.database.name:
            self.input.replace_parameters(self.parameters_matching)
            self.input.propagate_parameters(context)

    def propagate_include_in_tree(self, context: UserDatabaseContext):
        """
        If an activity has include_in_tree set to False, its children should also have
        include_in_tree set to False. Check include_in_tree value of output, change
        include_in_tree value of input if needed.
        :param context: gathers all context information necessary for generating
        activities. Typically generated by UserDatabase object.
        :return:
        """
        if self.input.database == context.database.name:
            if not self.output.include_in_tree:
                self.input.include_in_tree = False
            self.input.propagate_include_in_tree(context)

    def update_parameters_matching(
        self, new_parameters_matching: Dict[str, Union[str, Dict[str, Number], Number]]
    ):
        """
        Update exchange's parameters_matching values according to a new
        parameters_matching dict.
        :param new_parameters_matching: each new_parameters_matching item will be
        appended to Exchange's parameters_matching attribute. Pre-existing keys will be
        overwritten.
        :return:
        """
        for (
            new_param_to_replace,
            new_param_replacing,
        ) in new_parameters_matching.items():
            self.parameters_matching[new_param_to_replace] = new_param_replacing


class Activity(DatabaseElement):
    database: str
    "Name of the user database."
    exchanges: List[Exchange]
    "Emissions or consumptions generated when a unit of the activity is used."
    location: str
    "Location of the activity. Default value is GLO for global."  # TODO comment c'est utilisé ?
    unit: str
    "Unit of the amount."
    code: Optional[str] = None  # TODO should be referred to as uuid instead
    "Must be unique in the user database (equivalent to the notion of uuid)."
    data_quality: Optional[dict] = None
    "Deprecated."
    include_in_tree: Optional[bool] = False
    "If True, activity will become a node in built ImpactModel."
    parameters: Optional[List[str]] = []
    "Optional list of parameters necessary to execute this dataset."
    properties: Optional[Dict[str, Union[str, float, bool]]] = {}
    """Properties will remain on impact model, and can be used by apparun to breakdown
    the results according to life cycle phase, for exemple. Properties can be key/value
    (ex: {"phase": "production"} or flags (ex: {production_phase: True})."""

    def to_bw_format(self) -> Tuple[Tuple[str, str], dict]:
        """
        Converts an Activity to a format importable in a Brightway database
        :return: dictionary with all fields compatible with Brightway representation of
        Activity.
        """
        return (self.database, self.code), {
            "name": self.name,
            "location": self.location,
            "unit": self.unit,
            "type": self.type,
            "data_quality": self.data_quality,
            "comment": self.comment,
            "amount": self.amount,
            "include_in_tree": self.include_in_tree,
            "exchanges": [exchange.to_bw_format() for exchange in self.exchanges],
            "properties": self.properties,
        }

    @classmethod
    def from_serialized_activity(
        cls, serialized_activity: SerializedActivity, context: UserDatabaseContext
    ):
        """
        Generates an Activity from a SerializedActivity object, after completing
        following steps:
        - If Activity code is already in context, generates a duplicate by adding a
        suffix to code.
        - If Activity name is already in context and activity will become an ImpactModel
        node, also make the name unique by adding a suffix to name.
        - Create Activity and add in context
        - Populate exchange attribute by creating Exchange objects
        :param serialized_activity: information required to generate Activity
        :param context: gathers all context information necessary for generating
        activities. Typically generated by UserDatabase object.
        :return: generated Activity
        """
        activity_code = serialized_activity.uuid
        if (
            len(
                [
                    activity
                    for activity in context.activities
                    if activity.code == activity_code
                ]
            )
            > 0
        ):
            amount_of_copies = len(
                [
                    activity.code
                    for activity in context.activities
                    if re.match(f"^{activity_code}" + r"(_[0-9]+)?$", activity.code)
                ]
            )
            activity_code = f"{activity_code}_{amount_of_copies}"
            if activity_code in [activity.code for activity in context.activities]:
                raise SerializedDataError(
                    f"Cannot create a duplicate {serialized_activity.name} activity. "
                    f"Code {activity_code} is not unique."
                )
        activity_name = serialized_activity.name
        if (
            len(
                [
                    activity
                    for activity in context.activities
                    if activity.name == activity_name
                ]
            )
            > 0
            and serialized_activity.include_in_tree
        ):
            amount_of_copies = len(
                [
                    activity.name
                    for activity in context.activities
                    if re.match(f"^{activity_name}" + r"(_[0-9]+)?$", activity.name)
                ]
            )
            activity_name = f"{activity_name}_{amount_of_copies}"
            if activity_name in [activity.name for activity in context.activities]:
                raise SerializedDataError(
                    f"Cannot create a duplicate {serialized_activity.name} activity. "
                    f"Name {activity_name} is not unique, and activity will be a node."
                )

        new_activity = Activity(
            code=activity_code,
            database=serialized_activity.database,
            name=activity_name,
            location=serialized_activity.location,
            unit=serialized_activity.unit,
            type=serialized_activity.type,
            data_quality=serialized_activity.data_quality,
            comment=serialized_activity.comment,
            amount=serialized_activity.amount,
            parameters=serialized_activity.parameters,
            include_in_tree=serialized_activity.include_in_tree,
            exchanges=[],
            properties=serialized_activity.properties,
            context=context,
        )
        context.activities.append(new_activity)
        # Create Exchange objects
        exchanges = [
            Exchange.from_serialized_exchange(exchange, context, new_activity)
            for exchange in serialized_activity.exchanges
        ]
        exchanges = list(itertools.chain.from_iterable(exchanges))
        new_activity.exchanges = exchanges

        # If new activity is already present in activities, make a copy. New
        # code will be old one with a suffix incremented for each copy.
        return new_activity

    def replace_parameters(
        self, parameters_matching: Dict[str, Union[str, Dict[str, Number]]]
    ):
        """
        Replace Activity's exchanges' parameters_matching with parameters_matching keys
        by parameters matching values. Values can be constant, formulas or parameter
        name.
        :param parameters_matching: keys should be Exchange parameter to update, and
        values should be replacing value. Will be passed to all activity's exchanges.
        A float will set the parameter to a fixed value, str will update parameter's
        name or affect it to a formula, and dict is used to fix value of a categorical
        parameter.
        :return:
        """
        for exchange in self.exchanges:
            exchange.replace_parameters(parameters_matching)
        for parameter in [
            parameter
            for parameter in self.parameters
            if parameter in parameters_matching
        ]:
            self.parameters.remove(parameter)
            if not isinstance(parameters_matching[parameter], float):
                self.parameters.append(parameters_matching[parameter])

    def propagate_parameters(self, context: UserDatabaseContext):
        """
        Propagate Activity's exchanges' parameter_matching to activity's exchanges'
        input activities.
        :param context: gathers all context information necessary for generating
        activities. Typically generated by UserDatabase object.
        :return:
        """
        for exchange in self.exchanges:
            exchange.propagate_parameters(context)

    def propagate_include_in_tree(self, context: UserDatabaseContext):
        """
        If an activity has include_in_tree set to False, its children should also have
        include_in_tree set to False. Propagate Activity's exchanges' include_in_tree to
        activity's exchanges' input activities.
        :param context: gathers all context information necessary for generating
        activities. Typically generated by UserDatabase object.
        :return:
        """
        for exchange in self.exchanges:
            exchange.propagate_include_in_tree(context)
