import os

"""
Module containing all required functions to generate Appa Build compatible yaml files
for datasets exported from Eime V6. Each dataset will be exported at the impact level,
meaning the elementary flow information will be lost. Instructions to use this module
in corresponding CLI function and EimeV6Generator class.
"""
import re
from enum import Enum
from typing import Dict, List, Optional

import pandas as pd
from apparun.impact_methods import MethodFullName
from pydantic import BaseModel

from appabuild.database.serialized_data import SerializedActivity, SerializedExchange


class MethodEimeV6Name(str, Enum):
    """
    Map EFV3 methods known by Appa Run with method names in Eime V6 exports.
    """

    EFV3_ACIDIFICATION = "PEF-AP"
    EFV3_CLIMATE_CHANGE = "PEF-GWP"
    EFV3_CLIMATE_CHANGE_BIOGENIC = "PEF-GWPb"
    EFV3_CLIMATE_CHANGE_FOSSIL = "PEF-GWPf"
    EFV3_CLIMATE_CHANGE_LAND_USE = "PEF-GWPlu"
    EFV3_ECOTOXICITY_FRESHWATER = "PEF-CTUe"
    EFV3_PARTICULATE_MATTER_FORMATION = "PEF-PM"
    EFV3_EUTROPHICATION_FRESHWATER = "PEF-Epf"
    EFV3_EUTROPHICATION_MARINE = "PEF-Epm"
    EFV3_EUTROPHICATION_TERRESTRIAL = "PEF-Ept"
    EFV3_HUMAN_TOXICITY_CARCINOGENIC = "PEF-CTUh-c"
    EFV3_HUMAN_TOXICITY_NON_CARCINOGENIC = "PEF-CTUh-nc"
    EFV3_IONISING_RADIATION = "PEF-IR"
    EFV3_LAND_USE = "PEF-LU"
    EFV3_OZONE_DEPLETION = "PEF-ODP"
    EFV3_PHOTOCHEMICAL_OZONE_FORMATION = "PEF-POCP"
    EFV3_ENERGY_RESOURCES = "PEF-ADPf"
    EFV3_MATERIAL_RESOURCES = "PEF-ADPe"
    EFV3_WATER_USE = "PEF-WU"


class EimeV6DatasetDescription(BaseModel):
    """
    Information required to build serialized activities from Eime V6 exports.
    """

    name_in_export: Optional[str]
    name: Optional[str]
    database: Optional[str]
    uuid: Optional[str]
    unit: Optional[str]
    location: Optional[str]
    type: Optional[str] = None
    data_quality: Optional[Dict[str, int]] = None
    comment: Optional[str] = None
    amount: Optional[float] = None
    include_in_tree: Optional[bool] = False

    def to_serialized_activity(
        self, exchanges: List[SerializedExchange]
    ) -> SerializedActivity:
        """
        Convert to SerializedActivity.
        :param exchanges: list of SerializedExchange
        :return: SerializedActivity readily usable by Appa Build.
        """
        fields = dict(self)
        fields.pop("name_in_export")
        fields["exchanges"] = exchanges
        return SerializedActivity(**fields)


class EimeV6Impact(BaseModel):
    """
    Impact as generated by Eime V6 exports.
    """

    name_in_export: str
    "Name of the dataset in Eime V6 export."
    impact_name: str
    "Method name in Eime V6 export."
    amount: float
    "Amount of impact."

    def get_method_full_name(self) -> MethodFullName:
        """
        Convert Eime V6 export method name to Appa Run method name.
        :return: Method name known by Appa Run.
        """
        method_key = MethodEimeV6Name(self.impact_name).name
        return MethodFullName[method_key]

    def to_serialized_exchange(self, database: str) -> SerializedExchange:
        """
        Transforms the impact to a SerializedExchange. Each impact will be linked to a
        impact proxy exchange. And impact proxy is an elementary flow with a
        characterisation factor of 1 to the corresponding impact method. Impact proxies
        and corresponding characterisation factors are automatically generated at Appa
        Build start up.
        :param database: name of the user database the exchange will be attached to.
        Likely the database name of the calling Activity.
        :return: SerializedExchange readily usable by Appa Build.
        """
        fields = {
            "database": database,
            "type": "technosphere",
            "location": "GLO",
            "amount": self.amount,
            "name": f"Impact proxy for {self.impact_name}",
            "input": {
                "database": "impact_proxies",
                "uuid": f"{self.get_method_full_name().value}" f"_technosphere_proxy",
            },
        }
        return SerializedExchange(**fields)


class EimeV6Generator:
    """
    Generate yaml datasets readily usable by Appa Build from configuration file and
    Eime V6 impact exports. To generate the adequate export file, create a new Eime V6
    project, add a single phase (no particular phase name required), and add a copy of
    every dataset you want to export. Then, go to analysis page, tick all required PEF
    impact assessment method, and export the result as a xlsx file.
    Configuration file should contain all other information of the datasets you want
    to generate. Common fields across all datasets can be set once in a "default"
    dict. Specific fields can be provided as a list in a "datasets" dict. Each dataset
    should have a "name_in_export" key, which value is the name of the corresponding
    dataset in Eime V6 export.
    """

    def __init__(self):
        self.impacts: List[EimeV6Impact] = []

    def generate_datasets(
        self, eime_v6_impacts_export_path: str, datasets_description: dict
    ) -> List[SerializedActivity]:
        """
         Generate yaml datasets readily usable by Appa Build.
        :param eime_v6_impacts_export_path: Eime V6 impact exports. To generate the
        adequate export file, create a new Eime V6 project, add a single phase (no
        particular phase name required), and add a copy of every dataset you want to
        export. Then, go to analysis page, tick all required PEF impact assessment
        method, and export the result as a xlsx file.
        :param datasets_description: should contain all other information of the
        datasets you want to generate. Common fields across all datasets can be set
        once in a "default" dict. Specific fields can be provided as a list in a
        "datasets" dict. Each dataset should have a "name_in_export" key, which value
        is the name of the corresponding dataset in Eime V6 export.
        :return: list of SerializedActivity readily usable by Appa Build.
        """
        generated_datasets = []
        default_fields = datasets_description["default"]
        all_impacts = self.parse_impacts_export(eime_v6_impacts_export_path)
        for dataset_fields in datasets_description["datasets"]:
            fields = {**default_fields}
            fields.update(dataset_fields)
            exchanges = [
                impact.to_serialized_exchange(fields["database"])
                for impact in all_impacts
                if impact.name_in_export == dataset_fields["name_in_export"]
            ]
            dataset = EimeV6DatasetDescription(**fields)
            dataset = dataset.to_serialized_activity(exchanges)
            generated_datasets.append(dataset)
        return generated_datasets

    @staticmethod
    def save_datasets(datasets: List[SerializedActivity], output_path: str):
        """
        Export SerializedActivity generated to yaml files.
        :param datasets: SerialiazedActivity datasets to save
        :param output_path: directory to save yaml files.
        :return:
        """
        os.makedirs(output_path, exist_ok=True)
        for dataset in datasets:
            dataset.to_yaml(os.path.join(output_path, f"{dataset.uuid}.yaml"))

    def parse_impacts_export(
        self, eime_v6_impacts_export_path: str
    ) -> List[EimeV6Impact]:
        """
        Parse Eime V6 export to find every impact, with corresponding Eime V6 impact
        method name and dataset name.
        :param eime_v6_impacts_export_path: path to the export file exported by Eime V6.
        :return: a list of EimeV6impact present in the export file.
        """
        impacts = []
        raw = pd.read_excel(
            eime_v6_impacts_export_path, skiprows=[1, 2], converters={"Unnamed: 0": str}
        )
        for row in [
            r for _, r in raw.iterrows() if re.match(r"^1\.[0-9]+$", str(r[0]))
        ]:
            name_in_export = row[1]
            for i in range(row.size):
                if re.match("PEF-*", row.index[i]):
                    impact_name = row.index[i]
                    amount = row[i]
                    impacts.append(
                        EimeV6Impact(
                            name_in_export=name_in_export,
                            impact_name=impact_name,
                            amount=amount,
                        )
                    )
        return impacts
