"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHost = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * ## Bastion Host
 *
 * The database is located in an isolated subnet, meaning that it is not accessible from the public internet. As such, to interact with the database directly, a user must tunnel through a bastion host.
 *
 * ### Configuring
 *
 * This codebase controls _who_ is allowed to connect to the bastion host. This requires two steps:
 *
 * 1. Adding the IP address from which you are connecting to the `ipv4Allowlist` array
 * 1. Creating a bastion host system user by adding the user's configuration inform to `userdata.yaml`
 *
 * #### Adding an IP address to the `ipv4Allowlist` array
 *
 * The `BastionHost` construct takes in an `ipv4Allowlist` array as an argument. Find your IP address (eg `curl api.ipify.org`) and add that to the array along with the trailing CIDR block (likely `/32` to indicate that you are adding a single IP address).
 *
 * #### Creating a user via `userdata.yaml`
 *
 * Add an entry to the `users` array with a username (likely matching your local systems username, which you can get by running the `whoami` command in your terminal) and a public key (likely your default public key, which you can get by running `cat ~/.ssh/id_*.pub` in your terminal).
 *
 * <details>
 *
 * <summary>Tips & Tricks when using the Bastion Host</summary>
 *
 * #### Connecting to RDS Instance via SSM
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID \
 * --document-name AWS-StartPortForwardingSessionToRemoteHost \
 * --parameters '{
 * "host": [
 * "example-db.c5abcdefghij.us-west-2.rds.amazonaws.com"
 * ],
 * "portNumber": [
 * "5432"
 * ],
 * "localPortNumber": [
 * "9999"
 * ]
 * }' \
 * --profile $AWS_PROFILE
 * ```
 *
 * ```sh
 * psql -h localhost -p 9999 # continue adding username (-U) and db (-d) here...
 * ```
 *
 * Connect directly to Bastion Host:
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID --profile $AWS_PROFILE
 * ```
 *
 * #### Setting up an SSH tunnel
 *
 * In your `~/.ssh/config` file, add an entry like:
 *
 * ```
 * Host db-tunnel
 * Hostname {the-bastion-host-address}
 * LocalForward 54322 {the-db-hostname}:5432
 * ```
 *
 * Then a tunnel can be opened via:
 *
 * ```
 * ssh -N db-tunnel
 * ```
 *
 * And a connection to the DB can be made via:
 *
 * ```
 * psql -h 127.0.0.1 -p 5433 -U {username} -d {database}
 * ```
 *
 * #### Handling `REMOTE HOST IDENTIFICATION HAS CHANGED!` error
 *
 * If you've redeployed a bastion host that you've previously connected to, you may see an error like:
 *
 * ```
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * @    WARNING: REMOTE HOST IDENTIFICATION HAS CHANGED!     @
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY!
 * Someone could be eavesdropping on you right now (man-in-the-middle attack)!
 * It is also possible that a host key has just been changed.
 * The fingerprint for the ECDSA key sent by the remote host is
 * SHA256:mPnxAOXTpb06PFgI1Qc8TMQ2e9b7goU8y2NdS5hzIr8.
 * Please contact your system administrator.
 * Add correct host key in /Users/username/.ssh/known_hosts to get rid of this message.
 * Offending ECDSA key in /Users/username/.ssh/known_hosts:28
 * ECDSA host key for ec2-12-34-56-789.us-west-2.compute.amazonaws.com has changed and you have requested strict checking.
 * Host key verification failed.
 * ```
 *
 * This is due to the server's fingerprint changing. We can scrub the fingerprint from our system with a command like:
 *
 * ```
 * ssh-keygen -R 12.34.56.789
 * ```
 *
 * </details>
 */
class BastionHost extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { stackName } = aws_cdk_lib_1.Stack.of(this);
        // Build ec2 instance
        this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, "bastion-host", {
            vpc: props.vpc,
            vpcSubnets: { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC },
            instanceName: stackName,
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE4_GRAVITON, aws_cdk_lib_1.aws_ec2.InstanceSize.NANO),
            machineImage: aws_cdk_lib_1.aws_ec2.MachineImage.latestAmazonLinux({
                generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.ARM_64,
            }),
            userData: props.userData,
            userDataCausesReplacement: true,
        });
        // Assign elastic IP
        new aws_cdk_lib_1.aws_ec2.CfnEIP(this, "IP", {
            instanceId: this.instance.instanceId,
            tags: [{ key: "Name", value: stackName }],
        });
        // Allow bastion host to connect to db
        this.instance.connections.allowTo(props.db.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connection from bastion host");
        // Allow IP access to bastion host
        for (const ipv4 of props.ipv4Allowlist) {
            this.instance.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4(ipv4), aws_cdk_lib_1.aws_ec2.Port.tcp(props.sshPort || 22), "SSH Access");
        }
        // Integrate with SSM
        this.instance.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: [
                "ssmmessages:*",
                "ssm:UpdateInstanceInformation",
                "ec2messages:*",
            ],
            resources: ["*"],
        }));
        new aws_cdk_lib_1.CfnOutput(this, "instance-id-output", {
            value: this.instance.instanceId,
            exportName: `${stackName}-instance-id`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-ip-output", {
            value: this.instance.instancePublicIp,
            exportName: `${stackName}-instance-public-ip`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-dns-name-output", {
            value: this.instance.instancePublicDnsName,
            exportName: `${stackName}-public-dns-name`,
        });
    }
}
exports.BastionHost = BastionHost;
_a = JSII_RTTI_SYMBOL_1;
BastionHost[_a] = { fqn: "cdk-pgstac.BastionHost", version: "1.2.0" };
//# sourceMappingURL=data:application/json;base64,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