import typing as tp
from datetime import date

import httpx
import pendulum

from pstock.schemas import Trends
from pstock.yahoo_finance.quote import get_quote_summary


def _trend_period_to_timedelta(period: str) -> pendulum.Duration:
    return pendulum.duration(months=int(period[:-1]))


def _parse_trends_from_quote_summary(quote_summary: tp.Dict[str, tp.Any]) -> Trends:
    recommendation_trend = quote_summary.get("recommendationTrend", {})
    if not recommendation_trend:
        return Trends.parse_obj([])
    trends = recommendation_trend.get("trend", [])
    return Trends.parse_obj(
        [
            {
                "date": date.today() + _trend_period_to_timedelta(trend["period"]),
                "strong_buy": trend.get("strongBuy", 0),
                "buy": trend.get("buy", 0),
                "hold": trend.get("hold", 0),
                "sell": trend.get("sell", 0),
                "strong_sell": trend.get("stronSell", 0),
            }
            for trend in trends
        ]
    )


async def get_trends(
    symbol: str, client: tp.Optional[httpx.AsyncClient] = None
) -> Trends:
    """Get [Trends][pstock.schemas.Trends] data from yahoo-finance.

    [Trends][pstock.schemas.Trends] are parsed from the `quote_summary` generated by
    [get_quote_summary][pstock.yahoo_finance.get_quote_summary].

    Is provided symbol has no Trends (crypto/ETF): an empty
    [Trends][pstock.schemas.Trends] object is returned.

    Args:
        symbol (str): A stock symbol availlable in yahoo-finance
        client (tp.Optional[httpx.AsyncClient], optional): Defaults to None.

    Returns:
        [pstock.schemas.Trends][]
    """
    quote_summary = await get_quote_summary(symbol, client=client)
    return _parse_trends_from_quote_summary(quote_summary)


if __name__ == "__main__":
    import logging

    import asyncer

    from pstock.core.log import setup_logging

    setup_logging(level="INFO")
    logger = logging.getLogger()

    async def _worker(symbol: str, client: httpx.AsyncClient) -> None:
        trends = await get_trends(symbol, client=client)
        logger.info(symbol)
        logger.info(trends.df)

    async def _main(symbols):
        async with httpx.AsyncClient() as client:
            async with asyncer.create_task_group() as tg:
                for symbol in symbols:
                    tg.soonify(_worker)(symbol, client=client)

    asyncer.runnify(_main)(
        [
            "TSLA",
            "AAPL",
            "GOOG",
            "AMZN",
            "AMD",
            "GME",
            "SPCE",
            "^QQQ",
            "ETH-USD",
            "BTC-EUR",
        ]
    )
