import typing as tp

import httpx
import numpy as np

from pstock.schemas import Earning, Earnings
from pstock.yahoo_finance.quote import get_quote_summary

__all__ = "get_earnings"


def _parse_quarterly_earnings(
    quarterly_earnings: tp.List[tp.Dict[str, tp.Any]],
    quarterly_financial_chart: tp.List[tp.Dict[str, tp.Any]],
) -> tp.List[Earning]:
    date_to_earnings = {
        e.get("date", ""): {
            "actual": e.get("actual", {}).get("raw", np.nan),
            "estimate": e.get("estimate", {}).get("raw", np.nan),
        }
        for e in quarterly_earnings
        if "date" in e
    }
    date_to_fin_chart = {
        c.get("date", ""): {
            "revenue": c.get("revenue", {}).get("raw", np.nan),
            "earnings": c.get("earnings", {}).get("raw", np.nan),
        }
        for c in quarterly_financial_chart
        if "date" in c
    }
    all_dates = set(list(date_to_earnings.keys()) + list(date_to_fin_chart.keys()))

    return [
        Earning(
            quarter=quarter,
            actual=date_to_earnings.get(quarter, {}).get("actual", np.nan),
            estimate=date_to_earnings.get(quarter, {}).get("estimate", np.nan),
            revenue=date_to_fin_chart.get(quarter, {}).get("revenue", np.nan),
            earnings=date_to_fin_chart.get(quarter, {}).get("earnings", np.nan),
        )
        for quarter in all_dates
    ]


def _parse_next_earnings(earnings_chart: tp.Dict[str, tp.Any]) -> Earning:
    return Earning(
        quarter=(
            f"{earnings_chart.get('currentQuarterEstimateDate', '')}"
            f"{earnings_chart.get('currentQuarterEstimateYear', '')}"
        ),
        estimate=earnings_chart.get("currentQuarterEstimate", {}).get("raw", np.nan),
        actual=np.nan,
        revenue=np.nan,
        earnings=np.nan,
    )


def _get_next_earnings_date(earnings_chart: tp.Dict[str, tp.Any]) -> tp.Optional[int]:
    e_dates = earnings_chart.get("earningsDate", [])
    if not e_dates:
        return None
    return e_dates[0].get("raw")


def _parse_earnings_from_quote_summary(
    quote_summary: tp.Dict[str, tp.Any]
) -> tp.Tuple[Earnings, tp.Optional[int]]:
    earnings = quote_summary.get("earnings")
    if earnings is None or not isinstance(earnings, dict):
        return Earnings.parse_obj([]), None

    earnings_chart = earnings.get("earningsChart", {})
    quarterly_earnings = earnings_chart.get("quarterly", [])
    quarterly_financial_chart = earnings.get("financialsChart", {}).get("quarterly", [])

    parsed_list = _parse_quarterly_earnings(
        quarterly_earnings, quarterly_financial_chart
    ) + [_parse_next_earnings(earnings_chart)]
    nex_earnings_date = _get_next_earnings_date(earnings_chart)
    return Earnings.parse_obj(parsed_list), nex_earnings_date


async def get_earnings(
    symbol: str, client: tp.Optional[httpx.AsyncClient] = None
) -> Earnings:
    """Get [Earnings][pstock.schemas.Earnings] data from yahoo-finance.

    [Earnings][pstock.schemas.Earnings] are parsed from the `quote_summary` generated by
    [get_quote_summary][pstock.yahoo_finance.get_quote_summary].

    Is provided symbol has no earnings (crypto/ETF) and empty
    [Earnings][pstock.schemas.Earnings] is returned.

    Args:
        symbol (str): A stock symbol availlable in yahoo-finance
        client (tp.Optional[httpx.AsyncClient], optional): Defaults to None.

    Returns:
        [pstock.schemas.Earnings][]
    """
    quote_summary = await get_quote_summary(symbol, client=client)
    earnings, _ = _parse_earnings_from_quote_summary(quote_summary)
    return earnings


if __name__ == "__main__":
    import logging

    import asyncer

    from pstock.core.log import setup_logging

    setup_logging(level="INFO")
    logger = logging.getLogger()

    async def _worker(symbol: str, client: httpx.AsyncClient) -> None:
        asset = await get_earnings(symbol, client=client)
        logger.info(symbol)
        logger.info(asset.df)

    async def _main(symbols):
        async with httpx.AsyncClient() as client:
            async with asyncer.create_task_group() as tg:
                for symbol in symbols:
                    tg.soonify(_worker)(symbol, client=client)

    asyncer.runnify(_main)(
        [
            "TSLA",
            "AAPL",
            "GOOG",
            "AMZN",
            "AMD",
            "GME",
            "SPCE",
            "^QQQ",
            "ETH-USD",
            "BTC-EUR",
        ]
    )
