"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResourceProvider = void 0;
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const lambda_layer_node_proxy_agent_1 = require("@aws-cdk/lambda-layer-node-proxy-agent");
const HANDLER_DIR = path.join(__dirname, 'cluster-resource-handler');
const HANDLER_RUNTIME = lambda.Runtime.NODEJS_12_X;
/**
 * A custom resource provider that handles cluster operations. It serves
 * multiple custom resources such as the cluster resource and the fargate
 * resource.
 *
 * @internal
 */
class ClusterResourceProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        // The NPM dependency proxy-agent is required in order to support proxy routing with the AWS JS SDK.
        const nodeProxyAgentLayer = new lambda_layer_node_proxy_agent_1.NodeProxyAgentLayer(this, 'NodeProxyAgentLayer');
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'onEvent handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: {
                AWS_STS_REGIONAL_ENDPOINTS: 'regional',
                ...props.environment,
            },
            handler: 'index.onEvent',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            // Allow user to override the layer.
            layers: props.onEventLayer ? [props.onEventLayer] : [nodeProxyAgentLayer],
        });
        const isComplete = new lambda.Function(this, 'IsCompleteHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'isComplete handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: {
                AWS_STS_REGIONAL_ENDPOINTS: 'regional',
                ...props.environment,
            },
            handler: 'index.isComplete',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            layers: [nodeProxyAgentLayer],
        });
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: core_1.Duration.hours(1),
            queryInterval: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
        });
        props.adminRole.grant(onEvent.role, 'sts:AssumeRole');
        props.adminRole.grant(isComplete.role, 'sts:AssumeRole');
    }
    static getOrCreate(scope, props) {
        var _a;
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-eks.ClusterResourceProvider';
        return (_a = stack.node.tryFindChild(uid)) !== null && _a !== void 0 ? _a : new ClusterResourceProvider(stack, uid, props);
    }
    /**
     * The custom resource service token for this provider.
     */
    get serviceToken() { return this.provider.serviceToken; }
}
exports.ClusterResourceProvider = ClusterResourceProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci1yZXNvdXJjZS1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsdXN0ZXItcmVzb3VyY2UtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBRzdCLDhDQUE4QztBQUM5Qyx3Q0FBNkQ7QUFDN0QsZ0RBQWdEO0FBQ2hELDBGQUE2RTtBQU83RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO0FBQ3JFLE1BQU0sZUFBZSxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO0FBc0NuRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLHVCQUF3QixTQUFRLGtCQUFXO0lBYXRELFlBQW9CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW1DO1FBQ25GLEtBQUssQ0FBQyxLQUFzQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWxDLG9HQUFvRztRQUNwRyxNQUFNLG1CQUFtQixHQUFHLElBQUksbURBQW1CLENBQUMsSUFBSSxFQUFFLHFCQUFxQixDQUFDLENBQUM7UUFFakYsTUFBTSxPQUFPLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUMxRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDO1lBQ3hDLFdBQVcsRUFBRSxtREFBbUQ7WUFDaEUsT0FBTyxFQUFFLGVBQWU7WUFDeEIsV0FBVyxFQUFFO2dCQUNYLDBCQUEwQixFQUFFLFVBQVU7Z0JBQ3RDLEdBQUcsS0FBSyxDQUFDLFdBQVc7YUFDckI7WUFDRCxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDNUIsR0FBRyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDMUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNsRSxjQUFjLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDdkUsb0NBQW9DO1lBQ3BDLE1BQU0sRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ2hFLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUM7WUFDeEMsV0FBVyxFQUFFLHNEQUFzRDtZQUNuRSxPQUFPLEVBQUUsZUFBZTtZQUN4QixXQUFXLEVBQUU7Z0JBQ1gsMEJBQTBCLEVBQUUsVUFBVTtnQkFDdEMsR0FBRyxLQUFLLENBQUMsV0FBVzthQUNyQjtZQUNELE9BQU8sRUFBRSxrQkFBa0I7WUFDM0IsT0FBTyxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzVCLEdBQUcsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzFDLFVBQVUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbEUsY0FBYyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3ZFLE1BQU0sRUFBRSxDQUFDLG1CQUFtQixDQUFDO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDaEQsY0FBYyxFQUFFLE9BQU87WUFDdkIsaUJBQWlCLEVBQUUsVUFBVTtZQUM3QixZQUFZLEVBQUUsZUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDL0IsYUFBYSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ2xDLEdBQUcsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzFDLFVBQVUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDbEUsY0FBYyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ3hFLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFLLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUN2RCxLQUFLLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSyxFQUFFLGdCQUFnQixDQUFDLENBQUM7S0FDM0Q7SUE5RE0sTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFnQixFQUFFLEtBQW1DOztRQUM3RSxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLDBDQUEwQyxDQUFDO1FBQ3ZELGFBQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUE0QixtQ0FBSSxJQUFJLHVCQUF1QixDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDbEg7SUE0REQ7O09BRUc7SUFDSCxJQUFXLFlBQVksS0FBSyxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLEVBQUU7Q0FDakU7QUF0RUQsMERBc0VDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IER1cmF0aW9uLCBOZXN0ZWRTdGFjaywgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGNyIGZyb20gJ0Bhd3MtY2RrL2N1c3RvbS1yZXNvdXJjZXMnO1xuaW1wb3J0IHsgTm9kZVByb3h5QWdlbnRMYXllciB9IGZyb20gJ0Bhd3MtY2RrL2xhbWJkYS1sYXllci1ub2RlLXByb3h5LWFnZW50JztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG4vLyB2MiAtIGtlZXAgdGhpcyBpbXBvcnQgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHRvIHJlZHVjZSBtZXJnZSBjb25mbGljdCB3aGVuIGZvcndhcmQgbWVyZ2luZyB3aXRoIHRoZSB2MiBicmFuY2guXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmNvbnN0IEhBTkRMRVJfRElSID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJ2NsdXN0ZXItcmVzb3VyY2UtaGFuZGxlcicpO1xuY29uc3QgSEFORExFUl9SVU5USU1FID0gbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1g7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ2x1c3RlclJlc291cmNlUHJvdmlkZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgdG8gYXNzdW1lIGluIG9yZGVyIHRvIGludGVyYWN0IHdpdGggdGhlIGNsdXN0ZXIuXG4gICAqL1xuICByZWFkb25seSBhZG1pblJvbGU6IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIFZQQyB0byBwcm92aXNpb24gdGhlIGZ1bmN0aW9ucyBpbi5cbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgc3VibmV0cyB0byBwbGFjZSB0aGUgZnVuY3Rpb25zIGluLlxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0cz86IGVjMi5JU3VibmV0W107XG5cbiAgLyoqXG4gICAqIEVudmlyb25tZW50IHRvIGFkZCB0byB0aGUgaGFuZGxlci5cbiAgICovXG4gIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogQW4gQVdTIExhbWJkYSBsYXllciB0aGF0IGluY2x1ZGVzIHRoZSBOUE0gZGVwZW5kZW5jeSBgcHJveHktYWdlbnRgLlxuICAgKlxuICAgKiBJZiBub3QgZGVmaW5lZCwgYSBkZWZhdWx0IGxheWVyIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IG9uRXZlbnRMYXllcj86IGxhbWJkYS5JTGF5ZXJWZXJzaW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgc2VjdXJpdHkgZ3JvdXAgdG8gYXNzb2NpYXRlIHdpdGggdGhlIGZ1bmN0aW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBzZWN1cml0eSBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBlYzIuSVNlY3VyaXR5R3JvdXA7XG59XG5cbi8qKlxuICogQSBjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXIgdGhhdCBoYW5kbGVzIGNsdXN0ZXIgb3BlcmF0aW9ucy4gSXQgc2VydmVzXG4gKiBtdWx0aXBsZSBjdXN0b20gcmVzb3VyY2VzIHN1Y2ggYXMgdGhlIGNsdXN0ZXIgcmVzb3VyY2UgYW5kIHRoZSBmYXJnYXRlXG4gKiByZXNvdXJjZS5cbiAqXG4gKiBAaW50ZXJuYWxcbiAqL1xuZXhwb3J0IGNsYXNzIENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyIGV4dGVuZHMgTmVzdGVkU3RhY2sge1xuXG4gIHB1YmxpYyBzdGF0aWMgZ2V0T3JDcmVhdGUoc2NvcGU6IENvbnN0cnVjdCwgcHJvcHM6IENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyUHJvcHMpIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCB1aWQgPSAnQGF3cy1jZGsvYXdzLWVrcy5DbHVzdGVyUmVzb3VyY2VQcm92aWRlcic7XG4gICAgcmV0dXJuIHN0YWNrLm5vZGUudHJ5RmluZENoaWxkKHVpZCkgYXMgQ2x1c3RlclJlc291cmNlUHJvdmlkZXIgPz8gbmV3IENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyKHN0YWNrLCB1aWQsIHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY3VzdG9tIHJlc291cmNlIHByb3ZpZGVyIHRvIHVzZSBmb3IgY3VzdG9tIHJlc291cmNlcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwcm92aWRlcjogY3IuUHJvdmlkZXI7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2x1c3RlclJlc291cmNlUHJvdmlkZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlIGFzIENvcmVDb25zdHJ1Y3QsIGlkKTtcblxuICAgIC8vIFRoZSBOUE0gZGVwZW5kZW5jeSBwcm94eS1hZ2VudCBpcyByZXF1aXJlZCBpbiBvcmRlciB0byBzdXBwb3J0IHByb3h5IHJvdXRpbmcgd2l0aCB0aGUgQVdTIEpTIFNESy5cbiAgICBjb25zdCBub2RlUHJveHlBZ2VudExheWVyID0gbmV3IE5vZGVQcm94eUFnZW50TGF5ZXIodGhpcywgJ05vZGVQcm94eUFnZW50TGF5ZXInKTtcblxuICAgIGNvbnN0IG9uRXZlbnQgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdPbkV2ZW50SGFuZGxlcicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChIQU5ETEVSX0RJUiksXG4gICAgICBkZXNjcmlwdGlvbjogJ29uRXZlbnQgaGFuZGxlciBmb3IgRUtTIGNsdXN0ZXIgcmVzb3VyY2UgcHJvdmlkZXInLFxuICAgICAgcnVudGltZTogSEFORExFUl9SVU5USU1FLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgQVdTX1NUU19SRUdJT05BTF9FTkRQT0lOVFM6ICdyZWdpb25hbCcsXG4gICAgICAgIC4uLnByb3BzLmVudmlyb25tZW50LFxuICAgICAgfSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5vbkV2ZW50JyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICB2cGM6IHByb3BzLnN1Ym5ldHMgPyBwcm9wcy52cGMgOiB1bmRlZmluZWQsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5zdWJuZXRzID8geyBzdWJuZXRzOiBwcm9wcy5zdWJuZXRzIH0gOiB1bmRlZmluZWQsXG4gICAgICBzZWN1cml0eUdyb3VwczogcHJvcHMuc2VjdXJpdHlHcm91cCA/IFtwcm9wcy5zZWN1cml0eUdyb3VwXSA6IHVuZGVmaW5lZCxcbiAgICAgIC8vIEFsbG93IHVzZXIgdG8gb3ZlcnJpZGUgdGhlIGxheWVyLlxuICAgICAgbGF5ZXJzOiBwcm9wcy5vbkV2ZW50TGF5ZXIgPyBbcHJvcHMub25FdmVudExheWVyXSA6IFtub2RlUHJveHlBZ2VudExheWVyXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGlzQ29tcGxldGUgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdJc0NvbXBsZXRlSGFuZGxlcicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChIQU5ETEVSX0RJUiksXG4gICAgICBkZXNjcmlwdGlvbjogJ2lzQ29tcGxldGUgaGFuZGxlciBmb3IgRUtTIGNsdXN0ZXIgcmVzb3VyY2UgcHJvdmlkZXInLFxuICAgICAgcnVudGltZTogSEFORExFUl9SVU5USU1FLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgQVdTX1NUU19SRUdJT05BTF9FTkRQT0lOVFM6ICdyZWdpb25hbCcsXG4gICAgICAgIC4uLnByb3BzLmVudmlyb25tZW50LFxuICAgICAgfSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5pc0NvbXBsZXRlJyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICB2cGM6IHByb3BzLnN1Ym5ldHMgPyBwcm9wcy52cGMgOiB1bmRlZmluZWQsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5zdWJuZXRzID8geyBzdWJuZXRzOiBwcm9wcy5zdWJuZXRzIH0gOiB1bmRlZmluZWQsXG4gICAgICBzZWN1cml0eUdyb3VwczogcHJvcHMuc2VjdXJpdHlHcm91cCA/IFtwcm9wcy5zZWN1cml0eUdyb3VwXSA6IHVuZGVmaW5lZCxcbiAgICAgIGxheWVyczogW25vZGVQcm94eUFnZW50TGF5ZXJdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5wcm92aWRlciA9IG5ldyBjci5Qcm92aWRlcih0aGlzLCAnUHJvdmlkZXInLCB7XG4gICAgICBvbkV2ZW50SGFuZGxlcjogb25FdmVudCxcbiAgICAgIGlzQ29tcGxldGVIYW5kbGVyOiBpc0NvbXBsZXRlLFxuICAgICAgdG90YWxUaW1lb3V0OiBEdXJhdGlvbi5ob3VycygxKSxcbiAgICAgIHF1ZXJ5SW50ZXJ2YWw6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICB2cGM6IHByb3BzLnN1Ym5ldHMgPyBwcm9wcy52cGMgOiB1bmRlZmluZWQsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5zdWJuZXRzID8geyBzdWJuZXRzOiBwcm9wcy5zdWJuZXRzIH0gOiB1bmRlZmluZWQsXG4gICAgICBzZWN1cml0eUdyb3VwczogcHJvcHMuc2VjdXJpdHlHcm91cCA/IFtwcm9wcy5zZWN1cml0eUdyb3VwXSA6IHVuZGVmaW5lZCxcbiAgICB9KTtcblxuICAgIHByb3BzLmFkbWluUm9sZS5ncmFudChvbkV2ZW50LnJvbGUhLCAnc3RzOkFzc3VtZVJvbGUnKTtcbiAgICBwcm9wcy5hZG1pblJvbGUuZ3JhbnQoaXNDb21wbGV0ZS5yb2xlISwgJ3N0czpBc3N1bWVSb2xlJyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGN1c3RvbSByZXNvdXJjZSBzZXJ2aWNlIHRva2VuIGZvciB0aGlzIHByb3ZpZGVyLlxuICAgKi9cbiAgcHVibGljIGdldCBzZXJ2aWNlVG9rZW4oKSB7IHJldHVybiB0aGlzLnByb3ZpZGVyLnNlcnZpY2VUb2tlbjsgfVxufVxuIl19