"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const path = require("path");
const version_1 = require("../version");
const version_provider_1 = require("../version-provider");
const versionProvider = new version_provider_1.VersionProvider(path.join(__dirname, 'index-test.json'));
const indexTest = versionProvider['readInstallersIndex']();
const productSection = indexTest[version_provider_1.Product.deadline];
test.each([[version_provider_1.Platform.linux, '10.1.9.2'],
    [version_provider_1.Platform.mac, '10.1.9.2'],
    [version_provider_1.Platform.windows, '10.1.8.5'],
])('latest version', (platform, versionString) => {
    const result = versionProvider['getLatestVersion'](platform, productSection);
    expect(result).toEqual(versionString);
});
test.each([
    [version_provider_1.Platform.linux, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
            clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
            repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
        }],
    [version_provider_1.Platform.windows, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/Deadline-10.1.9.2-windows-installers.zip',
            clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/DeadlineClient-10.1.9.2-windows-installer.exe',
            repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/DeadlineRepository-10.1.9.2-windows-installer.exe',
        }],
    [version_provider_1.Platform.mac, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Mac/Deadline-10.1.9.2-osx-installers.dmg',
        }],
])('get Uri for platform', async (platform, versionedUris) => {
    const result = versionProvider['getUrisForPlatform'](version_provider_1.Product.deadline, productSection, platform, '10.1.9.2');
    expect(result).not.toBeNull();
    expect(result?.Uris).toEqual(versionedUris);
});
test('get Uri for platform - bad version', async () => {
    const badVersion = 'badVersionString';
    expect(() => versionProvider['getUrisForPlatform'](version_provider_1.Product.deadline, productSection, version_provider_1.Platform.linux, badVersion)).toThrowError(`Couldn't parse version from ${badVersion}`);
});
test('get deadline version', async () => {
    const result = await versionProvider.getVersionUris({
        product: version_provider_1.Product.deadline,
        platform: version_provider_1.Platform.linux,
        versionString: '10.1',
    });
    expect(result).not.toBeNull();
    const installerVersion = result.get(version_provider_1.Platform.linux);
    expect(installerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(installerVersion?.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
    });
    expect(installerVersion?.MajorVersion).toEqual('10');
    expect(installerVersion?.MinorVersion).toEqual('1');
    expect(installerVersion?.ReleaseVersion).toEqual('9');
    expect(installerVersion?.PatchVersion).toEqual('2');
});
test('product is not in file', async () => {
    await expect(versionProvider.getVersionUris({
        product: version_provider_1.Product.deadlineDocker,
    })).rejects.toThrowError(/Information about product DeadlineDocker can't be found/);
});
test('get deadline version for all platforms', async () => {
    const result = await versionProvider.getVersionUris({
        product: version_provider_1.Product.deadline,
    });
    expect(result).not.toBeNull();
    const linuxInstallerVersion = result.get(version_provider_1.Platform.linux);
    expect(linuxInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(linuxInstallerVersion?.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
    });
    expect(linuxInstallerVersion?.MajorVersion).toEqual('10');
    expect(linuxInstallerVersion?.MinorVersion).toEqual('1');
    expect(linuxInstallerVersion?.ReleaseVersion).toEqual('9');
    expect(linuxInstallerVersion?.PatchVersion).toEqual('2');
    const macInstallerVersion = result.get(version_provider_1.Platform.mac);
    expect(macInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(macInstallerVersion?.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Mac/Deadline-10.1.9.2-osx-installers.dmg',
    });
    expect(macInstallerVersion?.MajorVersion).toEqual('10');
    expect(macInstallerVersion?.MinorVersion).toEqual('1');
    expect(macInstallerVersion?.ReleaseVersion).toEqual('9');
    expect(macInstallerVersion?.PatchVersion).toEqual('2');
    const windowsInstallerVersion = result.get(version_provider_1.Platform.windows);
    expect(windowsInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(windowsInstallerVersion?.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/Deadline-10.1.8.5-windows-installers.zip',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/DeadlineClient-10.1.8.5-windows-installer.exe',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/DeadlineRepository-10.1.8.5-windows-installer.exe',
    });
    expect(windowsInstallerVersion?.MajorVersion).toEqual('10');
    expect(windowsInstallerVersion?.MinorVersion).toEqual('1');
    expect(windowsInstallerVersion?.ReleaseVersion).toEqual('8');
    expect(windowsInstallerVersion?.PatchVersion).toEqual('5');
});
test('not defined file path', () => {
    expect(() => (new version_provider_1.VersionProvider())['readInstallersIndex']()).toThrowError(/File path should be defined./);
});
test('invalid file path', () => {
    expect(() => (new version_provider_1.VersionProvider('test.txt'))['readInstallersIndex']()).toThrowError(/File test.txt was not found/);
});
test('get latest version without latest section', () => {
    expect(() => versionProvider['getLatestVersion']('linux', {})).toThrowError(/Information about latest version can not be found/);
});
test('get latest version without informtion for platform', () => {
    expect(() => versionProvider['getLatestVersion']('linux', { latest: {} })).toThrowError(/Information about latest version for platform linux can not be found/);
});
test('get requested Uri version for existing product.', () => {
    const requestedVersion = version_1.Version.parseFromVersionString('10.1.9.2');
    expect(requestedVersion).not.toBeNull();
    if (requestedVersion === null) {
        return;
    }
    expect(versionProvider['getRequestedUriVersion'](requestedVersion, {
        10: {
            1: {
                9: {
                    2: {
                        linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                    },
                },
            },
        },
    }, version_provider_1.Platform.linux, version_provider_1.Product.deadlineDocker)).toEqual({
        MajorVersion: '10',
        MinorVersion: '1',
        ReleaseVersion: '9',
        PatchVersion: '2',
        Uris: { bundle: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz' },
    });
});
test('get requested Uri version for not existing product.', () => {
    const requestedVersion = version_1.Version.parseFromVersionString('10.1.9.2');
    expect(requestedVersion).not.toBeNull();
    if (requestedVersion === null) {
        return;
    }
    expect(versionProvider['getRequestedUriVersion'](requestedVersion, {
        10: {
            1: {
                9: {
                    2: {
                        linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                    },
                },
            },
        },
    }, version_provider_1.Platform.windows, version_provider_1.Product.deadlineDocker)).toEqual(undefined);
});
test('get requested Uri version for not existing product version.', () => {
    const requestedVersion = version_1.Version.parseFromVersionString('10.2');
    expect(requestedVersion).not.toBeNull();
    if (requestedVersion === null) {
        return;
    }
    expect(() => {
        versionProvider['getRequestedUriVersion'](requestedVersion, {
            10: {
                1: {
                    9: {
                        2: {
                            linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                        },
                    },
                },
            },
        }, version_provider_1.Platform.windows, version_provider_1.Product.deadlineDocker);
    }).toThrow('DeadlineDocker version 10.2 is not available on windows');
});
//# sourceMappingURL=data:application/json;base64,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