/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
/// <reference types="node" />
import * as http from 'http';
import * as https from 'https';
/**
 * Properties for setting up an {@link TLSProps}.
 */
export interface TLSProps {
    /**
     * The content of the CA certificate.
     */
    readonly ca?: string;
    /**
     * The content of the PFX certificate.
     */
    readonly pfx?: string;
    /**
     * The shared passphrase used for a single private key and/or a PFX.
     */
    readonly passphrase?: string;
}
/**
 * Properties for setting up an {@link DeadlineClient}.
 */
export interface DeadlineClientProps {
    /**
     * The IP address or DNS name of the Render Queue.
     */
    readonly host: string;
    /**
     * The port number address of the Render Queue.
     */
    readonly port: number;
    /**
     * The protocol to use when connecting to the Render Queue.
     * Supported values: HTTP, HTTPS
     */
    readonly protocol: string;
    /**
     * The certificate, private key, and root CA certificate if SSL/TLS is used.
     */
    readonly tls?: TLSProps;
    /**
     * The number of retries if received status code 500 or higher.
     * @default 3
     */
    readonly retries?: number;
    /**
     * The amount of time in milliseconds to wait between the retries.
     * @default 10000 milliseconds
     */
    readonly retryWaitMs?: number;
}
/**
 * Properties for setting up an {@link RequestOptions}.
 */
interface RequestOptions {
    /**
     * The IP address or DNS name of the Render Queue.
     */
    readonly host: string;
    /**
     * The port Render Queue is listening to.
     */
    readonly port: number;
    /**
     * The agent used for TLS connection.
     */
    agent?: https.Agent;
}
/**
 * The response returned from the requests.
 */
export interface Response {
    /**
     * The data of the response to a request.
     */
    readonly data: any;
    /**
     * The full response obtained from the POST and GET requests.
     */
    readonly fullResponse: http.IncomingMessage;
}
/**
 * Implements a simple client that supports HTTP/HTTPS GET and POST requests.
 * It is intended to be used within Custom Resources that need to send the requests to the Render Queue.
 */
export declare class DeadlineClient {
    /**
     * The default number of retry attempts.
     */
    private static readonly DEFAULT_RETRY_COUNT;
    /**
     * Specifies the default waiting period between two requests.
     */
    private static readonly DEFAULT_RETRY_PERIOD_MS;
    readonly requestOptions: RequestOptions;
    private protocol;
    private readonly retries;
    private readonly retryWaitMs;
    constructor(props: DeadlineClientProps);
    /**
     * Perform an HTTP GET request.
     *
     * @param path The resource to request for.
     * @param requestOptions Other request options, including headers, timeout, etc.
     */
    GetRequest(path: string, requestOptions?: https.RequestOptions): Promise<Response>;
    /**
     * Perform an HTTP POST request.
     *
     * @param path The resource to request for.
     * @param data The data (body) of the request that contains the information to be sent.
     * @param requestOptions Other request options, including headers, timeout, etc.
     */
    PostRequest(path: string, data?: any, requestOptions?: https.RequestOptions): Promise<Response>;
    private FillRequestOptions;
    private performRequestWithRetry;
    private performRequest;
}
export {};
