"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.configureSEP = exports.SEPConfiguratorResource = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const configure_spot_event_plugin_1 = require("../lib/configure-spot-event-plugin");
const custom_resource_1 = require("../lib/custom-resource");
const deadline_client_1 = require("../lib/deadline-client");
const secrets_manager_1 = require("../lib/secrets-manager");
const conversion_1 = require("./conversion");
/**
 * A custom resource used to save Spot Event Plugin server data and configurations.
 */
class SEPConfiguratorResource extends custom_resource_1.SimpleCustomResource {
    constructor(secretsManagerClient) {
        super();
        this.secretsManagerClient = secretsManagerClient;
    }
    /**
     * @inheritdoc
     */
    validateInput(data) {
        return this.implementsSEPConfiguratorResourceProps(data);
    }
    /**
     * @inheritdoc
     */
    async doCreate(_physicalId, resourceProperties) {
        const spotEventPluginClient = await this.spotEventPluginClient(resourceProperties.connection);
        if (!await spotEventPluginClient.addGroups(resourceProperties.deadlineGroups)) {
            throw new Error(`Failed to add Deadline group(s) ${resourceProperties.deadlineGroups}`);
        }
        if (!await spotEventPluginClient.addPools(resourceProperties.deadlinePools)) {
            throw new Error(`Failed to add Deadline pool(s) ${resourceProperties.deadlinePools}`);
        }
        if (resourceProperties.spotFleetRequestConfigurations) {
            const convertedSpotFleetRequestConfigs = conversion_1.convertSpotFleetRequestConfiguration(resourceProperties.spotFleetRequestConfigurations);
            const stringConfigs = JSON.stringify(convertedSpotFleetRequestConfigs);
            const response = await spotEventPluginClient.saveServerData(stringConfigs);
            if (!response) {
                throw new Error('Failed to save spot fleet request with configuration');
            }
        }
        if (resourceProperties.spotPluginConfigurations) {
            const convertedSpotPluginConfigs = conversion_1.convertSpotEventPluginSettings(resourceProperties.spotPluginConfigurations);
            const pluginSettings = this.toKeyValueArray(convertedSpotPluginConfigs);
            const securitySettings = this.securitySettings();
            const response = await spotEventPluginClient.configureSpotEventPlugin([...pluginSettings, ...securitySettings]);
            if (!response) {
                throw new Error('Failed to save Spot Event Plugin Configurations');
            }
        }
        return undefined;
    }
    /**
     * @inheritdoc
     */
    async doDelete(_physicalId, _resourceProperties) {
        // Nothing to do -- we don't modify anything.
        return;
    }
    implementsSEPConfiguratorResourceProps(value) {
        if (!value || typeof (value) !== 'object' || Array.isArray(value)) {
            return false;
        }
        if (!this.implementsConnectionOptions(value.connection)) {
            return false;
        }
        return true;
    }
    implementsConnectionOptions(value) {
        if (!value || typeof (value) !== 'object' || Array.isArray(value)) {
            return false;
        }
        if (!value.hostname || typeof (value.hostname) !== 'string') {
            return false;
        }
        if (!value.port || typeof (value.port) !== 'string') {
            return false;
        }
        const portNum = Number.parseInt(value.port, 10);
        if (Number.isNaN(portNum) || portNum < 1 || portNum > 65535) {
            return false;
        }
        if (!value.protocol || typeof (value.protocol) !== 'string') {
            return false;
        }
        if (value.protocol !== 'HTTP' && value.protocol !== 'HTTPS') {
            return false;
        }
        if (!this.isSecretArnOrUndefined(value.caCertificateArn)) {
            return false;
        }
        return true;
    }
    isSecretArnOrUndefined(value) {
        if (value) {
            if (typeof (value) !== 'string' || !secrets_manager_1.isArn(value)) {
                return false;
            }
        }
        return true;
    }
    async spotEventPluginClient(connection) {
        // The calculation of retries is approximate. The real number of retries will be smaller,
        // because we quit Lambda before the timeout (see SimpleCustomResource implementation).
        const lambdaTimeoutMins = parseInt(process.env.LAMBDA_TIMEOUT_MINS || '15'); // The maximum Lambda execution time is 15 mins.
        const MS_IN_A_MINUTE = 60000;
        const timeRemaining = lambdaTimeoutMins * MS_IN_A_MINUTE;
        const retryWaitMs = 10000;
        const retries = Math.floor(timeRemaining / retryWaitMs);
        return new configure_spot_event_plugin_1.SpotEventPluginClient(new deadline_client_1.DeadlineClient({
            host: connection.hostname,
            port: Number.parseInt(connection.port, 10),
            protocol: connection.protocol,
            tls: {
                ca: connection.caCertificateArn ? await secrets_manager_1.readCertificateData(connection.caCertificateArn, this.secretsManagerClient) : undefined,
            },
            retries,
            retryWaitMs,
        }));
    }
    toKeyValueArray(input) {
        const configs = [];
        for (const [key, value] of Object.entries(input)) {
            if (value === undefined) {
                throw new Error(`Value for ${key} should be defined.`);
            }
            configs.push({
                Key: key,
                Value: value,
            });
        }
        return configs;
    }
    securitySettings() {
        return [
            {
                Key: 'UseLocalCredentials',
                Value: true,
            },
            {
                Key: 'NamedProfile',
                Value: '',
            },
        ];
    }
}
exports.SEPConfiguratorResource = SEPConfiguratorResource;
/**
 * The lambda handler that is used to log in to MongoDB and perform some configuration actions.
 */
/* istanbul ignore next */
async function configureSEP(event, context) {
    const handler = new SEPConfiguratorResource(new aws_sdk_1.SecretsManager());
    return await handler.handler(event, context);
}
exports.configureSEP = configureSEP;
//# sourceMappingURL=data:application/json;base64,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