"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const lib_1 = require("../../core/lib");
const asset_constants_1 = require("../../core/test/asset-constants");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_2 = require("../lib");
const asset_constants_2 = require("./asset-constants");
const test_helper_1 = require("./test-helper");
let app;
let stack;
let wfstack;
let vpc;
let renderQueue;
let rcsImage;
beforeEach(() => {
    app = new aws_cdk_lib_1.App();
    stack = new aws_cdk_lib_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
    const version = new lib_2.VersionQuery(stack, 'VersionQuery');
    renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
        vpc,
        images: { remoteConnectionServer: rcsImage },
        repository: new lib_2.Repository(stack, 'Repository', {
            vpc,
            version,
            secretsManagementSettings: { enabled: false },
        }),
        trafficEncryption: { externalTLS: { enabled: false } },
        version,
    });
    wfstack = new aws_cdk_lib_1.Stack(app, 'workerFleetStack', {
        env: {
            region: 'us-east-1',
        },
    });
});
test('default worker fleet is created correctly', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    assertions_1.Template.fromStack(wfstack).resourceCountIs('AWS::AutoScaling::AutoScalingGroup', 1);
    assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        IamInstanceProfile: {
            Ref: assertions_1.Match.stringLikeRegexp('^workerFleetInstanceProfile.*'),
        },
        ImageId: 'ami-any',
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    assertions_1.Match.stringLikeRegexp('^workerFleetInstanceSecurityGroup.*'),
                    'GroupId',
                ],
            },
        ],
        spotPrice: assertions_1.Match.absent(),
    });
    assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                'GroupId',
            ],
        },
        GroupId: {
            'Fn::ImportValue': 'infraStack:ExportsOutputFnGetAttRQLBSecurityGroupAC643AEDGroupId8F9F7830',
        },
    });
    assertions_1.Template.fromStack(wfstack).hasResourceProperties('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/workerFleet',
    });
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.'));
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy'));
});
test('security group is added to fleet after its creation', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    fleet.addSecurityGroup(aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
        allowAllOutbound: false,
    }));
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                    'GroupId',
                ],
            },
            'sg-123456789',
        ],
    });
});
test('WorkerFleet uses given security group', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false,
        }),
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            'sg-123456789',
        ],
    });
});
describe('allowing log listener port', () => {
    test('from CIDR', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        fleet.allowListenerPortFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24').connections);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
            SecurityGroupIngress: [
                {
                    CidrIp: '127.0.0.1/24',
                    Description: 'Worker remote command listening port',
                    FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
                    IpProtocol: 'tcp',
                    ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
                },
            ],
        });
    });
    test('to CIDR', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        fleet.allowListenerPortTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24').connections);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
            SecurityGroupIngress: [
                {
                    CidrIp: '127.0.0.1/24',
                    Description: 'Worker remote command listening port',
                    FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
                    IpProtocol: 'tcp',
                    ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
                },
            ],
        });
    });
    test('from SecurityGroup', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
        fleet.allowListenerPortFrom(securityGroup);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        });
    });
    test('to SecurityGroup', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
        fleet.allowListenerPortTo(securityGroup);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        });
    });
    test('from other stack', () => {
        const otherStack = new aws_cdk_lib_1.Stack(app, 'otherStack', {
            env: { region: 'us-east-1' },
        });
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
        fleet.allowListenerPortFrom(securityGroup);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        });
    });
    test('to other stack', () => {
        const otherStack = new aws_cdk_lib_1.Stack(app, 'otherStack', {
            env: { region: 'us-east-1' },
        });
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
        });
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
        fleet.allowListenerPortTo(securityGroup);
        // THEN
        assertions_1.Template.fromStack(otherStack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            FromPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'],
            IpProtocol: 'tcp',
            SourceSecurityGroupId: 'sg-123456789',
            ToPort: lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT'] + lib_2.WorkerInstanceFleet['MAX_WORKERS_PER_HOST'],
        });
    });
});
test('default worker fleet is created correctly with linux image', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    // 3 = repository + renderqueue + worker fleet
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::AutoScaling::AutoScalingGroup', 3);
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        IamInstanceProfile: {
            Ref: assertions_1.Match.stringLikeRegexp('^workerFleetInstanceProfile.*'),
        },
        ImageId: 'ami-any',
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    assertions_1.Match.stringLikeRegexp('^workerFleetInstanceSecurityGroup.*'),
                    'GroupId',
                ],
            },
        ],
        spotPrice: assertions_1.Match.absent(),
    });
});
test('default worker fleet is created correctly with spot config', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        spotPrice: 2.5,
    });
    // THEN
    assertions_1.Template.fromStack(wfstack).resourceCountIs('AWS::AutoScaling::AutoScalingGroup', 1);
    assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        SpotPrice: '2.5',
    });
});
test('default worker fleet is not created with incorrect spot config', () => {
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT + 1,
        });
    }).toThrowError(/Invalid value: 256 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT / 2,
        });
    }).toThrowError(/Invalid value: 0.0005 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
});
test('default worker fleet is created correctly custom Instance type', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
    });
});
test.each([
    'test-prefix/',
    '',
])('default worker fleet is created correctly with custom LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'workerFleet';
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, id, {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: testPrefix + id,
    });
});
test('worker fleet uses given UserData', () => {
    // GIVEN
    const id = 'workerFleet';
    const userData = aws_ec2_1.UserData.forLinux();
    // WHEN
    const workerFleet = new lib_2.WorkerInstanceFleet(stack, id, {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        userData,
    });
    // THEN
    expect(workerFleet.fleet.userData).toBe(userData);
});
test('default linux worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
        IamInstanceProfile: {
            Ref: assertions_1.Match.stringLikeRegexp('workerFleetInstanceProfile.*'),
        },
        UserData: {
            'Fn::Base64': {
                'Fn::Join': [
                    '',
                    [
                        '#!/bin/bash\n' +
                            'function exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\n' +
                            'trap exitTrap EXIT\n' +
                            `mkdir -p $(dirname '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n` +
                            `set -e\nchmod +x '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n'/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' -i us-east-1 `,
                        {
                            Ref: assertions_1.Match.stringLikeRegexp('^workerFleetStringParameter.*'),
                        },
                        `\nmkdir -p $(dirname '/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_2.RQ_CONNECTION_ASSET.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py' '/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py'\n` +
                            'if [ -f "/etc/profile.d/deadlineclient.sh" ]; then\n  source "/etc/profile.d/deadlineclient.sh"\nfi\n' +
                            `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py" --render-queue "http://`,
                        {
                            'Fn::GetAtt': [
                                'RQLB3B7B1CBC',
                                'DNSName',
                            ],
                        },
                        `:8080" \nrm -f "/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py"` +
                            `\nmkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py' '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py'\n` +
                            `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                            'set -e\n' +
                            `chmod +x '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                            `'/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '' '' '' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py`,
                    ],
                ],
            },
        },
    });
});
test('default windows worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
        IamInstanceProfile: {
            Ref: assertions_1.Match.stringLikeRegexp('workerFleetInstanceProfile.*'),
        },
        UserData: {
            'Fn::Base64': {
                'Fn::Join': [
                    '',
                    [
                        '<powershell>trap {\n$success=($PSItem.Exception.Message -eq "Success")\n' +
                            'cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 --success ($success.ToString().ToLower())\nbreak\n}\n' +
                            `mkdir (Split-Path -Path 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                        {
                            'Fn::Sub': asset_constants_1.CWA_ASSET_WINDOWS.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `' -key '${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n&'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -i us-east-1 `,
                        {
                            Ref: assertions_1.Match.stringLikeRegexp('^workerFleetStringParameter.*'),
                        },
                        `\nif (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }\n` +
                            `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py' ) -ea 0\nRead-S3Object -BucketName '`,
                        {
                            'Fn::Sub': asset_constants_2.RQ_CONNECTION_ASSET.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `' -key '${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py' -file 'C:/temp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py' -ErrorAction Stop\n` +
                            '$ErrorActionPreference = "Stop"\n' +
                            '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                            `& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py" --render-queue "http://`,
                        {
                            'Fn::GetAtt': [
                                'RQLB3B7B1CBC',
                                'DNSName',
                            ],
                        },
                        ':8080"  2>&1\n' +
                            `Remove-Item -Path "C:/temp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py"\n` +
                            `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' ) -ea 0\nRead-S3Object -BucketName '`,
                        {
                            'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `' -key '${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -ErrorAction Stop\n` +
                            `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                        {
                            'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `' -key '${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n` +
                            `&'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' '' '' '' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py\n` +
                            `if (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }\n` +
                            'throw \"Success\"</powershell>',
                    ],
                ],
            },
        },
    });
});
test('default worker fleet is created correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        groups: ['A', 'B'],
        pools: ['C', 'D'],
        region: 'E',
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
        IamInstanceProfile: {
            Ref: assertions_1.Match.stringLikeRegexp('workerFleetInstanceProfile.*'),
        },
        UserData: {
            'Fn::Base64': {
                'Fn::Join': [
                    '',
                    [
                        '#!/bin/bash\n' +
                            'function exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\n' +
                            'trap exitTrap EXIT\n' +
                            `mkdir -p $(dirname '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n` +
                            `set -e\nchmod +x '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n'/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' -i us-east-1 `,
                        {
                            Ref: assertions_1.Match.stringLikeRegexp('^workerFleetStringParameter.*'),
                        },
                        `\nmkdir -p $(dirname '/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_2.RQ_CONNECTION_ASSET.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py' '/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py'\n` +
                            'if [ -f "/etc/profile.d/deadlineclient.sh" ]; then\n  source "/etc/profile.d/deadlineclient.sh"\nfi\n' +
                            `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py" --render-queue "http://`,
                        {
                            'Fn::GetAtt': [
                                'RQLB3B7B1CBC',
                                'DNSName',
                            ],
                        },
                        `:8080" \nrm -f "/tmp/${asset_constants_2.RQ_CONNECTION_ASSET.Key}.py"` +
                            `\nmkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py' '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py'\n` +
                            `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                        {
                            'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Bucket.replace('${AWS::Region}', 'us-east-1'),
                        },
                        `/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                            'set -e\n' +
                            `chmod +x '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                            `'/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' 'a,b' 'c,d' 'E' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_2.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py`,
                    ],
                ],
            },
        },
    });
});
test('worker fleet does validation correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // group name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'none'],
        });
    }).toThrowError();
    // group name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet1', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'no ne'],
        });
    }).toThrowError(/Invalid value: no ne for property 'groups'/);
    // pool name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // pool name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet3', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // region as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet4', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none',
        });
    }).toThrowError(/Invalid value: none for property 'region'/);
    // region as 'all'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet5', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'all',
        });
    }).toThrowError(/Invalid value: all for property 'region'/);
    // region as 'unrecognized'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet6', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'unrecognized',
        });
    }).toThrowError(/Invalid value: unrecognized for property 'region'/);
    // region with invalid characters
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet7', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none@123',
        });
    }).toThrowError(/Invalid value: none@123 for property 'region'/);
    // region with reserved name as substring
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet8', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none123',
        });
    }).not.toThrowError();
    // region with case-insensitive name
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet9', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'None',
        });
    }).toThrowError(/Invalid value: None for property 'region'/);
});
describe('Block Device Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Warning if no BlockDevices provided', () => {
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.'));
    });
    test('No Warnings if Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                }],
        });
        //THEN
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: true,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        });
        assertions_1.Annotations.fromStack(wfstack).hasNoInfo(`/${fleet.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(wfstack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(wfstack).hasNoError(`/${fleet.node.path}`, assertions_1.Match.anyValue());
    });
    test('Warnings if non-Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                }],
        });
        //THEN
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: false,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        });
        assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`));
    });
    test('Warnings for BlockDevices without encryption specified', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                }],
        });
        //THEN
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        });
        assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`));
    });
    test('No warnings for Ephemeral blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                }],
        });
        //THEN
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                    VirtualName: 'ephemeral0',
                },
            ],
        });
        assertions_1.Annotations.fromStack(wfstack).hasNoInfo(`/${fleet.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(wfstack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(wfstack).hasNoError(`/${fleet.node.path}`, assertions_1.Match.anyValue());
    });
    test('No warnings for Suppressed blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                }],
        });
        //THEN
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                },
            ],
        });
        assertions_1.Annotations.fromStack(wfstack).hasNoInfo(`/${fleet.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(wfstack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.anyValue());
        assertions_1.Annotations.fromStack(wfstack).hasNoError(`/${fleet.node.path}`, assertions_1.Match.anyValue());
    });
});
describe('HealthMonitor Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Monitor is configured for Windows', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toContain(`&'C:/temp/${asset_constants_2.CONFIG_WORKER_HEALTHCHECK_WINDOWS.Key}.ps1' '63415' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`);
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        });
    });
    test('Monitor is configured for Linux', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toContain(`'/tmp/${asset_constants_2.CONFIG_WORKER_HEALTHCHECK_LINUX.Key}.sh' '63415' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`);
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assertions_1.Template.fromStack(wfstack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        });
    });
    test('UserData is added', () => {
        // WHEN
        class UserDataProvider extends lib_2.InstanceUserDataProvider {
            preCloudWatchAgent(host) {
                host.userData.addCommands('echo preCloudWatchAgent');
            }
            preRenderQueueConfiguration(host) {
                host.userData.addCommands('echo preRenderQueueConfiguration');
            }
            preWorkerConfiguration(host) {
                host.userData.addCommands('echo preWorkerConfiguration');
            }
            postWorkerLaunch(host) {
                host.userData.addCommands('echo postWorkerLaunch');
            }
        }
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            userDataProvider: new UserDataProvider(wfstack, 'UserDataProvider'),
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        expect(userData).toContain('echo preCloudWatchAgent');
        expect(userData).toContain('echo preRenderQueueConfiguration');
        expect(userData).toContain('echo preWorkerConfiguration');
        expect(userData).toContain('echo postWorkerLaunch');
    });
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'WorkerInstanceFleet',
        createConstruct: () => {
            // GIVEN
            const healthMonitorStack = new aws_cdk_lib_1.Stack(app, 'HealthMonitorStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const healthMonitor = new lib_1.HealthMonitor(healthMonitorStack, 'healthMonitor', {
                vpc,
            });
            const deviceName = '/dev/xvda';
            // WHEN
            new lib_2.WorkerInstanceFleet(wfstack, 'WorkerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                healthMonitor,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                    }],
            });
            return wfstack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
test('worker fleet signals when non-zero minCapacity', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        minCapacity: 1,
    });
    // WHEN
    const userData = fleet.fleet.userData.render();
    // THEN
    expect(userData).toContain('cfn-signal');
    assertions_1.Template.fromStack(wfstack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
            },
        },
    });
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.'));
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy'));
});
test('worker fleet does not signal when zero minCapacity', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        minCapacity: 0,
    });
    // WHEN
    const userData = fleet.fleet.userData.render();
    // THEN
    // There should be no cfn-signal call in the UserData.
    expect(userData).not.toContain('cfn-signal');
    // Make sure we don't have a CreationPolicy
    test_helper_1.resourcePropertiesCountIs(wfstack, 'AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: assertions_1.Match.anyValue(),
    }, 0);
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*Deploying with 0 minimum capacity\..*'));
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.'));
    assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy'));
});
describe('secrets management enabled', () => {
    let props;
    // GIVEN
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_2.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_2.Repository(stack, 'Repository', {
                vpc,
                version,
            }),
            version,
        });
        wfstack = new aws_cdk_lib_1.Stack(app, 'workerFleetStack');
        props = {
            renderQueue,
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({}),
        };
    });
    test('vpc subnets not specified => warns about dedicated subnets', () => {
        // WHEN
        const workerInstanceFleet = new lib_2.WorkerInstanceFleet(wfstack, 'WorkerInstanceFleet', props);
        // THEN
        assertions_1.Annotations.fromStack(wfstack).hasWarning(`/${workerInstanceFleet.node.path}`, 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
    });
    test('vpc subnets specified => does not emit dedicated subnets warning', () => {
        // WHEN
        const workerInstanceFleet = new lib_2.WorkerInstanceFleet(wfstack, 'WorkerInstanceFleet', {
            ...props,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
        });
        // THEN
        assertions_1.Annotations.fromStack(wfstack).hasNoWarning(`/${workerInstanceFleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*dedicated subnet.*'));
    });
});
//# sourceMappingURL=data:application/json;base64,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